/*
 * Copyright 2005 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.dav.daf.main.impl;

import de.bsvrz.dav.daf.communication.dataRepresentation.datavalue.SendDataObject;
import de.bsvrz.dav.daf.communication.lowLevel.telegrams.BaseSubscriptionInfo;
import de.bsvrz.dav.daf.main.DataNotSubscribedException;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.impl.config.AttributeGroupUsageIdentifications;
import de.bsvrz.dav.daf.main.impl.config.DafDataModel;
import de.bsvrz.dav.daf.main.impl.config.telegrams.AuthentificationAnswer;
import de.bsvrz.dav.daf.main.impl.config.telegrams.AuthentificationRequest;
import de.bsvrz.dav.daf.main.impl.config.telegrams.ConfigTelegram;
import de.bsvrz.dav.daf.main.impl.config.telegrams.TransmitterConnectionInfo;
import de.bsvrz.dav.daf.main.impl.config.telegrams.TransmitterConnectionInfoAnswer;
import de.bsvrz.dav.daf.main.impl.config.telegrams.TransmitterConnectionInfoRequest;
import de.bsvrz.sys.funclib.debug.Debug;
import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.LinkedList;
import java.util.ListIterator;

/**
 * @author Kappich Systemberatung
 */
public class ConfigurationManager {

    /** DebugLogger für Debug-Ausgaben */
    private static final Debug DEBUG = Debug.getLogger();

    /** Das Datenmodel */
    private final DataModel _dataModel;
    /** Die ID des Konfigurationsverantwortlichen der Konfiguration. */
    private final long _configurationId;
    /** Die Pid der Konfiguration */
    private final String _configurationPid;
    /** Der Pfad der Konfiguration */
    private final String _configurationPath;
    /** Der Name der Applikation */
    private final String _applicationName;
    /** Interne Liste der ankommenden Konfigurationsnachrichten */
    private final LinkedList<ConfigTelegram> _pendingResponses;
    /** Der Anmeldemanager */
    private SubscriptionManager _subscriptionManager;
    /** Der Index der Konfigurationsendung */
    private int _configSendIndex;

    /**
     * Erzeugt ein neues Objekt mit den gegebenen Parametern.
     *
     * @param configurationId   ID des Konfigurationsverantwortlichen
     * @param configurationPid  Pid der Konfiguration
     * @param configurationPath Pfad der Konfiguration
     * @param applicationName   Name der Applikation
     * @param dataModel         Datenmodel
     */
    public ConfigurationManager(long configurationId, String configurationPid, String configurationPath, String applicationName,
                                DataModel dataModel) {
        _configurationId = configurationId;
        _configurationPid = configurationPid;
        _configurationPath = configurationPath;
        _applicationName = applicationName;
        _dataModel = dataModel;
	    _pendingResponses = new LinkedList<>();
        _configSendIndex = 0;
    }

    /**
     * Erzeugt ein Datenmodel und beendet die initialisierungsphase.
     *
     * @param subscriptionManager Anmeldemanager
     */
    public final void completeInitialisation(SubscriptionManager subscriptionManager) {
        _subscriptionManager = subscriptionManager;
        if (_dataModel instanceof DafDataModel) {
            ((DafDataModel) _dataModel).init(this, _configurationId);
        }
        _subscriptionManager.setConfigurationManager(this);
    }

    /**
     * Gibt die Pid der Konfiguration zurück.
     *
     * @return Pid der Konfiguration
     */
    public final String getConfigurationPid() {
        return _configurationPid;
    }

    /**
     * Gibt die ID der Konfiguration zurück
     *
     * @return Id der Konfiguration
     */
    public final long getConfigurationId() {
        return _configurationId;
    }

    /**
     * Gibt den Pfad der Konfiguration zurück.
     *
     * @return Pfad der Konfiguration
     */
    public final String getConfigurationPath() {
        return _configurationPath;
    }

    /**
     * Gibt den Name der Applikation zurück.
     *
     * @return Name der Applikation
     */
    public final String getApplicationName() {
        return _applicationName;
    }

    /**
     * Gibt das Datenmodel zurück.
     *
     * @return Datenmodel
     */
    public final DataModel getDataModel() {
        return _dataModel;
    }

    /**
     * Diese Methode wird aufgerufen wenn eine neues Konfigurationstelegram erhalten wird.
     *
     * @param newData Die neue Konfigurationsdaten.
     */
    public void update(SendDataObject newData) {
        if (_dataModel instanceof DafDataModel) {
            final DafDataModel dataModel = (DafDataModel) this._dataModel;
            byte[] b = newData.getData();
            if (b == null) {
                return;
            }
            DataInputStream in = new DataInputStream(new ByteArrayInputStream(b));
            try {
                long configurationId = in.readLong();
                if (_configurationId == configurationId) {
//				if (_dataModel.getConfigurationAuthorityId() == configurationId) {
                    String info = in.readUTF();
                    byte telegramType = in.readByte();
                    int length = in.readInt();
                    ConfigTelegram telegram = ConfigTelegram.getTelegram(telegramType, dataModel);
                    telegram.read(in);
                    telegram.setInfo(info);
                    switch (telegramType) {
                        case ConfigTelegram.TRANSMITTER_CONNECTION_INFO_ANSWER_TYPE:
                        case ConfigTelegram.AUTHENTIFICATION_ANSWER_TYPE: {
                            synchronized (_pendingResponses) {
                                _pendingResponses.add(telegram);
                                _pendingResponses.notifyAll();
                            }
                        }
                        default: {
                            dataModel.update(telegram);
                        }
                    }
                }
            } catch (IOException ex) {
                ex.printStackTrace();
            }
        }
    }

    /**
     * Sendet eine Konfigurationsanfrage zur Kommunikationsschicht
     *
     * @param baseSubscriptionInfo Basisanmeldeinformationen
     * @param telegram             Kofigurationstelegramm
     */
    public final void sendConfigData(BaseSubscriptionInfo baseSubscriptionInfo, ConfigTelegram telegram) {
        if ((_subscriptionManager == null) || (telegram == null)) {
            return;
        }

        ++_configSendIndex;
        if (_configSendIndex > 0x3FFFFFFF) {
            _configSendIndex = 1;
        }
        long dataIndex = (long) (_configSendIndex << 2) & 0x00000000FFFFFFFCL;
        long index = (CommunicationConstant.START_TIME | dataIndex) & 0xFFFFFFFFFFFFFFFCL;

        ByteArrayOutputStream byteBuffer = new ByteArrayOutputStream();
        try {
            DataOutputStream out = new DataOutputStream(byteBuffer);
            telegram.write(out);
            out.flush();
            byte[] telegramData = byteBuffer.toByteArray();
            byteBuffer.reset();

            out.writeLong(_subscriptionManager.getHighLevelCommunication().getApplicationId());
            out.writeUTF(telegram.getInfo());
            out.writeByte(telegram.getType());
            out.writeInt(telegramData.length);
            for (byte telegramDatum : telegramData) {
                out.write(telegramDatum);
            }
            out.close();
        } catch (IOException ex) {
            ex.printStackTrace();
        }

        byte[] data = byteBuffer.toByteArray();
        SendDataObject sendData = new SendDataObject(baseSubscriptionInfo, false, index, System.currentTimeMillis(), (byte) 0, null, data);
        try {
            _subscriptionManager.sendData(sendData);
        } catch (DataNotSubscribedException ex) {
            DEBUG.error("Konfigurationsanfrage konnte nicht versendet werden", ex);
            throw new RuntimeException("Konfigurationsanfrage konnte nicht versendet werden", ex);
        }
    }

    /**
     * Überprüfft ob die Authentificationsdaten existent und gültig sind. Wenn der Benutzer gültig ist und sein Passwort mit den gegebenen
     * verschlüsselten Passwort übereinstimmt, dann schickt die Konfiguration die Id des Benutzers zurück, sonst -1
     *
     * @param userName          der Benutzername
     * @param encryptedPassword verschlüsselte Passwort
     * @param text              der Zufallstext mit den der Passwort verschlüsselt wurde
     * @param processName       der Authentifikationsvervahren
     *
     * @deprecated Funktioniert mit der neuen SRP-Authentifizierung nicht mehr, stattdessen
     * {@link de.bsvrz.dav.daf.main.config.management.UserAdministration}-Interface
     *     verwenden.
     */
    @Deprecated
    public final long isValidUser(String userName, byte[] encryptedPassword, String text, String processName) {
        BaseSubscriptionInfo readBaseSubscriptionInfo =
            new BaseSubscriptionInfo(_configurationId, AttributeGroupUsageIdentifications.CONFIGURATION_READ_REQUEST, (short) 0);

        ConfigTelegram telegram = new AuthentificationRequest(userName, encryptedPassword, text, processName);
        String info = Integer.toString(telegram.hashCode());
        telegram.setInfo(info);
        sendConfigData(readBaseSubscriptionInfo, telegram);

        // Waiting for Answer
        ConfigTelegram response = null;
        long waitingTime = 0, startTime = System.currentTimeMillis();
        long sleepTime = 10;
        while (waitingTime < CommunicationConstant.MAX_WAITING_TIME_FOR_SYNC_RESPONCE) {
            try {
                synchronized (_pendingResponses) {
                    _pendingResponses.wait(sleepTime);
                    if (sleepTime < 1000) {
                        sleepTime *= 2;
                    }
                    ListIterator<ConfigTelegram> _iterator = _pendingResponses.listIterator(_pendingResponses.size());
                    while (_iterator.hasPrevious()) {
                        response = _iterator.previous();
                        if ((response != null) && (response.getType() == ConfigTelegram.AUTHENTIFICATION_ANSWER_TYPE) &&
                            info.equals(response.getInfo())) {
                            _iterator.remove();
                            try {
                                AuthentificationAnswer authentificationAnswer = (AuthentificationAnswer) response;
                                return authentificationAnswer.getUserId();
                            } catch (ClassCastException ex) {
                                ex.printStackTrace();
                            }
                        }
                    }
                }
                waitingTime = System.currentTimeMillis() - startTime;
            } catch (InterruptedException ex) {
                ex.printStackTrace();
                break;
            }
        }
        throw new RuntimeException("Die Konfiguration antwortet nicht");
    }

    /**
     * Führt eine Konfigurationsanfrage durch um die versorgte Datenverteilertopologie zu ermitteln.
     *
     * @param transmitterId Die Id des Datenverteilers, dessen Verbindungsinformationen bestimmt werden müssen.
     *
     * @return Array mit Verbindungsinformationen
     */
    public final TransmitterConnectionInfo[] getTransmitterConnectionInfo(long transmitterId) {
        BaseSubscriptionInfo readBaseSubscriptionInfo =
            new BaseSubscriptionInfo(_configurationId, AttributeGroupUsageIdentifications.CONFIGURATION_READ_REQUEST, (short) 0);

        TransmitterConnectionInfoRequest telegram = new TransmitterConnectionInfoRequest(2, transmitterId);
        String info = Integer.toString(telegram.hashCode());
        telegram.setInfo(info);
        DEBUG.finer("Sende Anfrage nach Topologie: " + telegram.parseToString());
        sendConfigData(readBaseSubscriptionInfo, telegram);

        // Waiting for Answer
        ConfigTelegram response = null;
        long waitingTime = 0, startTime = System.currentTimeMillis();
        long sleepTime = 10;
        while (waitingTime < CommunicationConstant.MAX_WAITING_TIME_FOR_SYNC_RESPONCE) {
            try {
                synchronized (_pendingResponses) {
                    _pendingResponses.wait(sleepTime);
                    if (sleepTime < 1000) {
                        sleepTime *= 2;
                    }
                    ListIterator<ConfigTelegram> _iterator = _pendingResponses.listIterator(_pendingResponses.size());
                    while (_iterator.hasPrevious()) {
                        response = _iterator.previous();
                        if ((response != null) && (response.getType() == ConfigTelegram.TRANSMITTER_CONNECTION_INFO_ANSWER_TYPE) &&
                            info.equals(response.getInfo())) {
                            try {
                                TransmitterConnectionInfoAnswer transmitterConnectionInfoAnswer = (TransmitterConnectionInfoAnswer) response;
                                if (transmitterConnectionInfoAnswer.getTransmitterId() == telegram.getTransmitterId()) {
                                    _iterator.remove();
                                    DEBUG.finer("Empfangene Antwort mit Topologie: " + transmitterConnectionInfoAnswer.parseToString());
                                    return transmitterConnectionInfoAnswer.getTransmitterConnectionInfos();
                                }
                            } catch (ClassCastException ex) {
                                ex.printStackTrace();
                            }
                        }
                    }
                }
                waitingTime = System.currentTimeMillis() - startTime;
            } catch (InterruptedException ex) {
                ex.printStackTrace();
                break;
            }
        }
        throw new RuntimeException("Die Konfiguration antwortet nicht");
    }
}
