/*
 * Copyright 2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.main.impl.config.request.util;

import de.bsvrz.dav.daf.main.impl.config.request.RequestException;
import de.bsvrz.sys.funclib.debug.Debug;

import java.util.Map;
import java.util.concurrent.*;

/**
 * Basisklasse, die beim Empfangen von Konfigurationsantworten hilft, indem aus den bereits bekannten Antworten diejenige mit der korrekten Request-ID
 * geholt wird, bzw. wenn noch keine vorhanden ist, auf eine entsprechende Antwort gewartet wird.
 *
 * @author Kappich Systemberatung
 */
public class RequestReceiver<T> {

    private static final Debug _debug = Debug.getLogger();

    /**
     * Timeout für den Empfang einer Konfigurationsantwort. Wenn diese Dauer (in _timeoutUnit) überschritten wird, wird davon ausgegangen dass keine
     * Antwort mehr erfolgt.
     */
    private final long _timeoutDuration;

    /**
     * Einheit für das Feld _timeoutDuration.
     */
    private final TimeUnit _timeoutUnit;

    /**
     * In diesem Cache werden die bereits eingegangenen Antworten gespeichert, die noch nicht "abgeholt" wurden. Hier wird ein Guava-Cache
     * anstelle einer Map benutzt, da dieser veraltete Einträge automatisch entfernen kann. Ansonsten könnten "Konfigurationsantworten" die keiner
     * Anfrage entsprechen zu einem Memory-Leak führen.
     */
    private final Map<Integer, Answer> _cache;

    /**
     * Erstellt einen neuen RequestReceiver.
     *
     * @param timeoutDuration Timeout für den Empfang einer Konfigurationsantwort. Wenn diese Dauer (in _timeoutUnit) überschritten wird, wird davon
     *                        ausgegangen dass keine Antwort mehr erfolgt.
     * @param unit            definiert die Einheit für timeoutDuration.
     */
    public RequestReceiver(final long timeoutDuration, final TimeUnit unit) {
        _timeoutDuration = timeoutDuration;
        _timeoutUnit = unit;
	    _cache = new ConcurrentHashMap<>();
    }

    /**
     * Holt eine bereits eingegangene Konfigurationsantwort ab, oder wartet maximal {@link #getTimeoutDuration() timeoutDuration}, bis diese noch
     * ankommt.
     *
     * @param requestIndex RequestIndex der Nachricht, auf die gewartet werden soll.
     *
     * @return Nachricht
     *
     * @throws RequestException Wenn die Konfiguration nicht innerhalb des Zeitlimits eine gültige Antwort geschickt hat.
     */
    public T getAnswer(int requestIndex) throws RequestException {
	    CompletableFuture<T> future = _cache.computeIfAbsent(requestIndex, this::initializeAnswer).future;
        try {
            return future.get(_timeoutDuration, _timeoutUnit);
        } catch (TimeoutException e) {
            _debug.warning("Fehler beim Warten auf Konfigurationsdaten", e);
            throw new RequestException("Die Konfiguration antwortet nicht", e);
        } catch (InterruptedException e) {
            throw new RequestException("Unterbrochen", e);
        } catch (ExecutionException e) {
            _debug.warning("Fehler beim Verarbeiten einer Konfigurationsantwort", e);
            throw new RequestException("Fehler beim Verarbeiten einer Konfigurationsantwort", e);
        } finally {
	        _cache.remove(requestIndex);
	        purgeOldRequests();
        }
    }

	/**
	 * Räumt alte Anfragen auf, die nicht innerhalb des Timeouts abgefragt wurden.
	 */
	private void purgeOldRequests() {
		if (_cache.isEmpty()) {
			return;
		}
		long checkTime = System.nanoTime();
		long maxDuration = getTimeoutUnit().toNanos(getTimeoutDuration());
		_cache.values().removeIf(answer -> checkTime - answer.initializationTimeNanos > maxDuration);
    }

    /**
     * Wird aufgerufen, wenn eine Konfigurationsantwort eingegangen ist. Die Nachricht sorgt dann dafür, dass ein aktuell wartender oder ein
     * zukünftiger {@link #getAnswer(int)}-Aufruf die hier übergebene Nachricht zurückgibt.
     *
     * @param requestIndex Request-Index der hier übergebenen Nachricht
     * @param answer       Eigentliche Nachricht
     */
    public void answerReceived(int requestIndex, T answer) {
	    CompletableFuture<T> future = _cache.computeIfAbsent(requestIndex, this::initializeAnswer).future;
        future.complete(answer);
    }

	private Answer initializeAnswer(Integer integer) {
		return new Answer();
	}

    /**
     * Timeout für den Empfang einer Konfigurationsantwort. Wenn diese Dauer (in timeoutUnit) überschritten wird, wird davon ausgegangen, dass keine
     * Antwort mehr erfolgt.
     *
     * @return Wert TimeoutDuration
     */
    public long getTimeoutDuration() {
        return _timeoutDuration;
    }

    /**
     * Gibt die Einheit von {@link #getTimeoutDuration() TimeoutDuration} zurück.
     *
     * @return die Einheit von {@link #getTimeoutDuration()}
     */
    public TimeUnit getTimeoutUnit() {
        return _timeoutUnit;
    }

	private class Answer {
		final CompletableFuture<T> future = new CompletableFuture<>();
		final long initializationTimeNanos = System.nanoTime();
	}
}
