/*
 * Copyright 2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.main.impl.notification;

import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.function.Consumer;

/**
 * Diese Klasse verwaltet eine Menge von Event-Listenern von Typ T, bei denen zwischen internen und öffentlichen/asynchronen Listenern unterschieden
 * wird. Wenn ein Ereignis ausgelöst wird, werden interne Listener im selben Thread ausgelöst der mit {@link #postEvent(Consumer)} die
 * Benachrichtigung auslöst. Asynchrone EventListener werden im {@link DafMainEventThread} asynchron benachrichtigt, sodass der auslösende Thread
 * nicht blockiert werden kann.
 * <ul>
 *     <li>Interne Eventlistener sollten keine blockierenden Aktionen durchführen.</li>
 *     <li>Externe Eventlistener werden u. U. erst verzögert ausgelöst, blockieren aber nicht die ausführung von internen Benachrichtungen. Sie
 *     blockieren sich aber ggf. gegenseitig.</li>
 * </ul>
 *
 * @author Kappich Systemberatung
 */
public class EventManager<T> {

    /**
     * Debug-Logger.
     */
    private static final Debug _debug = Debug.getLogger();

    /**
     * Thread-Objekt, auf dem die asynchronen Events veröffentlicht werden. Kann {@code null} sein, dann werden Threads auf beliebigen THreads
     * veröffentlicht.
     */
    @Nullable
    private final DafMainEventThread _notificationThread;
    /**
     * Set mit asynchronen/öffentlichen Listenern.
     */
    private final Set<T> _asyncListeners = new CopyOnWriteArraySet<>();
    /**
     * Set mit internen (synchronen) Listenern.
     */
    private final Set<T> _internalListeners = new CopyOnWriteArraySet<>();

    /**
     * Erstellt einen neuen EventManager ohne eigenen Dispatch-Thread. Asnychrone Events werden dann gff. jedesmal in einem neuen Thread publiziert.
     */
    public EventManager() {
        _notificationThread = null;
    }

    /**
     * Erstellt einen neuen EventManager mit Dispatch-Thread.
     *
     * @param notificationThread Event-Thread
     */
    public EventManager(DafMainEventThread notificationThread) {
        _notificationThread = notificationThread;
    }

    /**
     * Fügt einen Listener hinzu, der die Event-Benachrichtigung asynchron erhält.
     *
     * @param listener Listener
     */
    public void addAsyncNotificationListener(T listener) {
        _asyncListeners.add(Objects.requireNonNull(listener));
    }

    /**
     * Entfernt einen asynchronen Listener.
     *
     * @param listener zu entfernender Listener
     *
     * @return {@code true} wenn entfernt, {@code false} falls nicht vorhanden.
     */
    public boolean removeAsyncNotificationListener(T listener) {
        return _asyncListeners.remove(Objects.requireNonNull(listener));
    }

    /**
     * Fügt einen Listener hinzu, der die Event-Benachrichtigung synchron erhält.
     *
     * @param listener Listener
     */
    public void addInternalNotificationListener(T listener) {
        _internalListeners.add(Objects.requireNonNull(listener));
    }

    /**
     * Entfernt einen synchronen Listener.
     *
     * @param listener zu entfernender Listener
     *
     * @return {@code true} wenn entfernt, {@code false} falls nicht vorhanden.
     */
    public boolean removeInternalNotificationListener(T listener) {
        return _internalListeners.remove(Objects.requireNonNull(listener));
    }

    /**
     * Veröffentlicht ein Ereignis. Als Parameter wird ein Lambda übergeben, der auf einem Listener von Typ T das Ereignis auslöst.
     * <p>
     * Diese Methode veröffentlicht zuerst das Ereignis auf den internen Listenern im selben Thread und dann auf den externen Listenern im
     * notificationThread.
     * </p>
     *
     * @param eventGenerator Ereignis-Generator
     */
    public void postEvent(Consumer<T> eventGenerator) {
        notifyListeners(eventGenerator, (T[]) _internalListeners.toArray());
        T[] asyncListeners = (T[]) _asyncListeners.toArray();
        if (asyncListeners.length > 0) {
            Runnable runnable = () -> notifyListeners(eventGenerator, asyncListeners);
            if (_notificationThread == null) {
                new Thread(runnable).start();
            } else {
                _notificationThread.invoke(runnable);
            }
        }
    }

    /**
     * Benachrichtigt eine Menge von Listenern im aktuellen Thread. Diese methode ist etwas aufwändiger, da Fehler behandelt werden, die innerhalb des
     * Listener-Aufrufs auftreten könnten. Hier sollte wegen des unbehandelten Fehlers gewarnt werden, weitere Listener aber ausgeführt werden.
     *
     * @param eventGenerator Ereignis-Generator
     * @param asyncListeners Array mit Listenern
     */
    private void notifyListeners(Consumer<T> eventGenerator, T[] asyncListeners) {
        for (T publicListener : asyncListeners) {
            try {
                eventGenerator.accept(publicListener);
            } catch (Exception e) {
                _debug.warning("Unbehandelter Fehler in Listener-Benachrichtigung", e);
                Thread.UncaughtExceptionHandler exceptionHandler = Thread.currentThread().getUncaughtExceptionHandler();
                if (exceptionHandler != null) {
                    exceptionHandler.uncaughtException(Thread.currentThread(), e);
                }
            }
        }
    }

    /**
     * Gibt {@code true} zurück, wenn es keine EventListener gibt.
     *
     * @return {@code true}, wenn es keine EventListener gibt, sonst {@code false}
     */
    public boolean isEmpty() {
        return _internalListeners.isEmpty() && _asyncListeners.isEmpty();
    }
}
