/*
 * Copyright 2016-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.userManagement;

import de.bsvrz.dav.daf.communication.srpAuthentication.SrpClientAuthentication;
import de.bsvrz.dav.daf.communication.srpAuthentication.SrpCryptoParameter;
import de.bsvrz.dav.daf.communication.srpAuthentication.SrpVerifierAndUser;
import de.bsvrz.dav.daf.communication.srpAuthentication.SrpVerifierData;
import de.bsvrz.dav.daf.main.InconsistentLoginException;
import de.bsvrz.dav.daf.main.authentication.ClientCredentials;
import de.bsvrz.dav.daf.main.config.ConfigurationTaskException;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

/**
 * Internes Interface für Benutzerverwaltungsfunktionen.
 * <p>
 * Dieses Interface hat die Instanzen {@link UserManagementFileOffline} und {@link UserManagementFileOnline} um die Benutzer eines
 * Datenverteilersystems entweder Online (über die Benutzerschnittstelle im DAF) oder Offline (über die Bearbeitung der Benutzerverwaltung.xml) zu verwalten.
 * <p> 
 * Hinweis: Die Offline-Benutzerverwaltung kann kein Benutzerobjekt in der Konfiguration anlegen und ignoriert gewisse Parameter,
 * wie z. B. Pid und Konfigurationsbereich in {@link #createUser(String, ClientCredentials, boolean, String, String)}.
 * 
 * @author Kappich Systemberatung
 */
public interface UserManagementFileInterface {

    /**
     * Gibt alle Benutzer zurück, die dem System bekannt sind. Es ist undefiniert, welche Auswirkung das Verändern des zurückgegebenen Sets hat,
     * im Zweifelsfall bitte eine Kopie erstellen.
     * 
     * @return Set mit Benutzernamen
     */
    Set<String> getUsers();

    /** 
     * Gibt {@code true} zurück, wenn der angegebene Benutzer ein Administrator ist.
     * @return {@code true}, wenn der angegebene Benutzer ein Administrator ist, sonst {@code false}
     * @throws ConfigurationTaskException wenn der Benutzer unbekannt ist oder die Information aus anderen Gründen nicht ermittelt werden konnte
     */
    boolean isUserAdmin(String userName) throws ConfigurationTaskException;

    /**
     * Setzt für den angebenden Benutzer, ob er Admin-Rechte hat.
     * @param userName Benutzer
     * @param admin    Administrator-Rechte ja/nein?
     * @throws ConfigurationTaskException wenn der Benutzer unbekannt ist oder die Anfrage aus anderen Gründen (z. B. fehlende Berechtigung) abgelehnt wurde
     */
    void setUserAdmin(String userName, boolean admin) throws ConfigurationTaskException;

    /** 
     * Gibt die SRP6-Verschlüsselungsparameter für das angegebene Passwort zurück.
     * @param userName Benutzername
     * @param passwordIndex Einmal-Passwort-Index oder -1 für das normale Passwort
     * @return die SRP6-Verschlüsselungsparameter für das angegebene Passwort oder {@code null}, falls Passwort nicht mit SRP6 verschlüsselt wurde
     * @throws ConfigurationTaskException wenn der Benutzer unbekannt ist oder die Anfrage aus anderen Gründen (z. B. fehlende Berechtigung) abgelehnt wurde
     */
    default SrpCryptoParameter getCryptoParameter(String userName, int passwordIndex) throws ConfigurationTaskException {
        SrpVerifierAndUser userData = getVerifier(userName, passwordIndex);
        if (userData.isPlainTextPassword()) {
            return null;
        }
        return userData.getVerifier().getSrpCryptoParameter();
    }

    /**
     * Gibt zurück, ob das angegebene Passwort oder der angegebene Login-Token für den angegebenen Benutzernamen ein korrekter Login ist.
     * @param userName Benutzername
     * @param clientCredentials Passwort oder Login-Token, siehe z. B. {@link ClientCredentials#ofPassword(char[])}
     * @param passwordIndex Einmal-Passwort-Index oder -1 für das normale Passwort
     * @return die SRP6-Verschlüsselungsparameter für das angegebene Passwort oder {@code null}, falls Passwort nicht mit SRP6 verschlüsselt wurde
     * @throws ConfigurationTaskException wenn der Benutzer unbekannt ist oder die Anfrage aus anderen Gründen (z. B. fehlende Berechtigung) abgelehnt wurde
     */
    default boolean validateClientCredentials(String userName, ClientCredentials clientCredentials, int passwordIndex)
        throws ConfigurationTaskException {
        SrpVerifierAndUser srpVerifier = getVerifier(userName, passwordIndex);
        return SrpClientAuthentication.validateVerifier(srpVerifier.getVerifier(), userName, clientCredentials);
    }

    /**
     * Setzt für den angebenden Benutzer ein neues Passwort.
     * @param userName Benutzer
     * @param password Neues Passwort
     * @return Passwort des Benutzers als {@link ClientCredentials} (in der Regel als Login-Token verschlüsselt)
     * @throws ConfigurationTaskException wenn der Benutzer unbekannt ist oder die Anfrage aus anderen Gründen (z. B. fehlende Berechtigung) abgelehnt wurde
     */
    ClientCredentials setUserPassword(String userName, char[] password) throws ConfigurationTaskException;   
    
    /**
     * Setzt für den angebenden Benutzer einen neuen Benutzernamen und ein neues Passwort. Bei einer Änderung des Benutzernamens muss immer das Passwort
     * mit gesetzt werden, da das Passwort anhand des Benutzernamens verschlüsselt wird. Als neues Passwort kann selbstverständlich das alte Passwort
     * noch einmal angegeben werden, wodurch es mit dem neuen Benutzernamen neu verschlüsselt wird.
     * <p>
     * Im Online-Modus wird das Benutzerobjekt umbenannt, die Pid (sofern vorhanden) bleibt jedoch gleich, damit Parameter, Referenzen usw. erhalten bleiben.
     * 
     * @param userName Benutzer
     * @param password Neues Passwort
     * @throws ConfigurationTaskException wenn der Benutzer unbekannt ist oder die Anfrage aus anderen Gründen (z. B. fehlende Berechtigung) abgelehnt wurde
     */
    ClientCredentials setUserNameAndPassword(String userName, String newUserName, char[] password) throws ConfigurationTaskException;

    /**
     * Setzt für den angebenden Benutzer einen neuen zufälligen Login-Token (ohne Passwort) und gobt dieses zurück.
     * @param userName Benutzer
     * @return zufälliger Login-Token
     * @throws ConfigurationTaskException wenn der Benutzer unbekannt ist oder die Anfrage aus anderen Gründen (z. B. fehlende Berechtigung) abgelehnt wurde
     */
    ClientCredentials setRandomToken(String userName) throws ConfigurationTaskException;

    /**
     * Erzeugt ein Login-Token für den automatischen Login für einen Benutzer
     * @param userName      Benutzername
     * @param password      Benutzerpasswort
     * @param passwordIndex Einmal-Passwort-Index oder -1 für das normale Passwort
     * @return Verschlüsseltes Passwort
     * @throws ConfigurationTaskException wenn der Benutzer unbekannt ist oder die Anfrage aus anderen Gründen (z. B. fehlende Berechtigung) abgelehnt wurde
     */
    default ClientCredentials getLoginToken(String userName, char[] password, int passwordIndex) throws ConfigurationTaskException {
        SrpVerifierAndUser srpVerifier = getVerifier(userName, passwordIndex);
        try {
            return SrpClientAuthentication.createLoginToken(srpVerifier.getVerifier(), userName, password);
        } catch (InconsistentLoginException e) {
            throw new ConfigurationTaskException(e);
        }
    }

    /**
     * Legt einen neuen Benutzer interaktiv an
     * @param userName         Benutzername
     * @param password         Initiales Passwort des Benutzers
     * @param admin            Soll der Benutzer Administratorrechte haben?
     * @param consoleInterface Interface für die interaktive Abfrage von weiteren Informationen (z. B. Konfigurationsbereich, Pid)
     * @throws ConfigurationTaskException Wenn die Anfrage aus diversen Gründen (z. B. fehlende Berechtigung) abgelehnt wurde
     */
    void createUser(String userName, ClientCredentials password, boolean admin, final ConsoleInterface consoleInterface)
        throws ConfigurationTaskException;

    /**
     * Legt einen neuen Benutzer an. Die beiden letzten Parameter sind nur für die {@link UserManagementFileOnline}-Implementierung relevant
     * @param userName         Benutzername
     * @param password         Initiales Passwort des Benutzers
     * @param admin            Soll der Benutzer Administratorrechte haben?
     * @param configArea       Pid des Konfigurationsbereichs, in dem das Benutzerobjekt im Datenverteiler angelegt werden soll
     * @param pid              Pid des neuen Benutzerobjekts im Datenverteiler
     * @throws ConfigurationTaskException Wenn die Anfrage aus diversen Gründen (z. B. fehlende Berechtigung) abgelehnt wurde
     */
    void createUser(String userName, ClientCredentials password, boolean admin, String configArea, String pid)
        throws ConfigurationTaskException;

    /**
     * Legt einen neuen Benutzer an. Konfigurationsbereich und Benutzer-Pid werden bei Bedarf automatisch festgelegt.
     * @param userName         Benutzername
     * @param password         Initiales Passwort des Benutzers
     * @param admin            Soll der Benutzer Administratorrechte haben?
     * @throws ConfigurationTaskException Wenn die Anfrage aus diversen Gründen (z. B. fehlende Berechtigung) abgelehnt wurde
     */
    void createUser(String userName, ClientCredentials password, boolean admin) throws ConfigurationTaskException;

    /**
     * Löscht den angegebenen Benutzer
     * @param userName Benutzername
     * @throws ConfigurationTaskException Wenn die Anfrage aus diversen Gründen (z. B. fehlende Berechtigung) abgelehnt wurde
     */
    void deleteUser(String userName) throws ConfigurationTaskException;

    /**
     * Fügt dem angegebenen Benutzer Einmalpasswörter hinzu
     * @param userName  Benutzer
     * @param passwords Einmalpasswörter
     * @return Map, die die übergebenen Einmalpasswörter nach ihrem Passwortindex indiziert speichert. Der verwendete Passwortindex wird automatisch ermittelt.
     * @throws ConfigurationTaskException Wenn die Anfrage aus diversen Gründen (z. B. fehlende Berechtigung) abgelehnt wurde
     */
    Map<Integer, String> createOneTimePasswords(String userName, Collection<? extends String> passwords) throws ConfigurationTaskException;

    /**
     * Löscht alle vorhandenen Einmalpasswörter eines Benutzers
     * @param userName Benutzername
     * @throws ConfigurationTaskException Wenn die Anfrage aus diversen Gründen (z. B. fehlende Berechtigung) abgelehnt wurde
     */
    void clearOneTimePasswords(final String userName) throws ConfigurationTaskException;

    /**
     * Gibt von einem Benutzer die Indizes der noch unbenutzten, verwendbaren, Einmalpasswörter zurück.
 
     * @param userName        Benutzername, für den Passwörter abgefragt werden sollen
     * @return Indizes der noch verwendbaren Einmalpasswörter (leeres Array, falls keine Passwörter mehr verfügbar sind), aufsteigend sortiert.
     * @throws ConfigurationTaskException Wenn die Anfrage aus diversen Gründen (z. B. fehlende Berechtigung) abgelehnt wurde
     */
    int[] getOneTimePasswordIDs(final String userName) throws ConfigurationTaskException;

    /**
     * Deaktiviert ein einzelnes Einmalpasswort eines Benutzers (markiert dieses als benutzt)
     * @param userName Benutzername
     * @throws ConfigurationTaskException Wenn die Anfrage aus diversen Gründen (z. B. fehlende Berechtigung) abgelehnt wurde
     */
    void disableOneTimePassword(final String userName, int passwordID) throws ConfigurationTaskException;

    /** 
     * Gibt die Pid des Datenverteilerobjekts zurück. Diese Methode wird für die Generierung eines Login-Tokens für fremde Server benutzt.
     * @return die Pid des Datenverteilerobjekts oder einen Ersatz-String wie "&lt;Lokal&gt;" falls nicht ermittelbar.
     */
    String getDavPid();

    /**
     * Fragt den serverseitigen Überprüfungscode eines Benutzerpassworts ab. Warnung: der Überprüfungscode kann ggf. Brute-Force-Angriffe auf das
     * Benutzerpasswort erleichtern. Diese Operation wird in der Regel nur für den eigenen Benutzer zugelassen oder benötigt sonst Admin-Rechte.
     * 
     * @param userName      Benutzername
     * @param passwordIndex Einmal-Passwort-Index oder -1 für das normale Passwort
     * @return Objekt, das daten über den Benutzer und den Überprüfungscode enthält
     * @throws ConfigurationTaskException Wenn die Anfrage aus diversen Gründen (z. B. fehlende Berechtigung) abgelehnt wurde
     */
    SrpVerifierAndUser getVerifier(String userName, int passwordIndex) throws ConfigurationTaskException;

    /**
     * Setzt einen neuen Überprüfungscode. Dieser kann z. B. mit 
     * {@link SrpClientAuthentication#createVerifier(SrpCryptoParameter, String, ClientCredentials)} erzeugt werden.
     * Dies ist eine fortgeschrittene Version von {@link #setUserPassword(String, char[])}, bei der das Klartextpasswort
     * nicht zwingend bekannt sein muss.
     * 
     * @param userName Benutzername
     * @param srpVerifierData neuer Überprüfungscode
     * @throws ConfigurationTaskException Wenn die Anfrage aus diversen Gründen (z. B. fehlende Berechtigung) abgelehnt wurde
     */
    void setVerifier(String userName, SrpVerifierData srpVerifierData) throws ConfigurationTaskException;
}
