/*
 * Copyright 2015-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.dav.daf.util;

import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.RandomAccessFile;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;

/**
 * Implementierung eines gepufferten {@link java.io.RandomAccessFile}. Diese Klasse implementiert die Interfaces {@link java.io.DataInput}, {@link
 * java.io.DataOutput} und {@link java.nio.channels.ByteChannel} und unterstützt alle wesentlichen Methoden eines {@link java.io.RandomAccessFile}s
 * <p>
 * Diese Klasse ist (anders als {@link java.nio.channels.FileChannel}) nicht für die Verwendung durch mehrere unsynchronisierte Threads geeignet.
 *
 * @author Kappich Systemberatung
 */
public class BufferedRandomAccessFile extends FileAccess {

    /** ungepufferter EingabeStream */
    protected final InputStream _rawInStream;

    /** ungepufferter AusgabeStream */
    protected final OutputStream _rawOutStream;

    /** FileChannel zum Lesen und Schreiben der Datei */
    private final FileChannel _channel;

    /**
     * Erstellt ein neues gepuffertes BufferedFile als gepufferten Ersatz eines {@link java.io.RandomAccessFile}.
     *
     * @param file Datei
     *
     * @throws FileNotFoundException Falls Datei nicht gefunden
     */
    public BufferedRandomAccessFile(final File file) throws FileNotFoundException {
        this(file, "rw");
    }

    /**
     * Erstellt ein neues gepuffertes BufferedFile als gepufferten Ersatz eines {@link java.io.RandomAccessFile}.
     *
     * @param file       Datei
     * @param bufferSize Größe des Lese und Schreibpuffers in Byte
     *
     * @throws FileNotFoundException Falls Datei nicht gefunden
     */
    public BufferedRandomAccessFile(final File file, final int bufferSize) throws FileNotFoundException {
        this(file, "rw", bufferSize);
    }

    /**
     * Erstellt ein neues gepuffertes BufferedFile als gepufferten Ersatz eines {@link java.io.RandomAccessFile}.
     *
     * @param file Datei
     * @param mode "r" wenn nur gelesen werden soll, "rw" zum Lesen und schreiben. Siehe {@link java.io.RandomAccessFile}
     *
     * @throws java.io.FileNotFoundException Falls Datei nicht gefunden
     */
    public BufferedRandomAccessFile(final File file, final String mode) throws FileNotFoundException {
        this(file, mode, defaultBufferSize);
    }

    /**
     * Erstellt ein neues gepuffertes BufferedFile als gepufferten Ersatz eines {@link java.io.RandomAccessFile}.
     *
     * @param file       Datei
     * @param mode       "r" wenn nur gelesen werden soll, "rw" zum Lesen und schreiben. Siehe {@link java.io.RandomAccessFile}
     * @param bufferSize Größe des Lese und Schreibpuffers in Byte
     *
     * @throws FileNotFoundException Falls Datei nicht gefunden
     */
    public BufferedRandomAccessFile(final File file, final String mode, final int bufferSize) throws FileNotFoundException {
        this(new RandomAccessFile(file, mode), bufferSize);
    }

    private BufferedRandomAccessFile(final RandomAccessFile randomAccessFile, final int bufferSize) {
        super(bufferSize);
        if (bufferSize <= 0) {
            throw new IllegalArgumentException("Buffer size <= 0");
        }
        _channel = randomAccessFile.getChannel();
        _rawInStream = Channels.newInputStream(_channel);
        _rawOutStream = Channels.newOutputStream(_channel);
    }

    @Override
    protected DataOutputStream getDataOutStream() {
        flushInStream(); // Vor dem Schreiben Leseoperationen abschließen und Lesepuffer löschen (um FilePointer abschließend zu setzen)
        if (_dataOutStream == null) {
            _dataOutStream = new DataOutputStream(new BufferedOutputStream(_rawOutStream, _bufferSize));
        }
        return _dataOutStream;
    }

    @Override
    protected DataInputStream getDataInStream() throws IOException {
        flushOutStream(); // Vor dem Lesen Schreiboperationen abschließen und Schreibpuffer löschen (um FilePointer abschließend zu setzen)
        if (_dataInStream == null) {
            _dataInStream = new DataInputStream(new BufferedInputStream(_rawInStream, _bufferSize));
        }
        return _dataInStream;
    }

    @Override
    public boolean isOpen() {
        return _channel.isOpen();
    }

    @Override
    public FileChannel getChannel() {
        return _channel;
    }

    @Override
    public BufferedRandomAccessFile position(final long newPosition) throws IOException {
        super.position(newPosition);
        return this;
    }

    @Override
    public BufferedRandomAccessFile truncate(final long size) throws IOException {
        super.truncate(size);
        return this;
    }
}
