/*
 * Copyright 2024 by Kappich Systemberatung Aachen
 *
 * This file is part of de.bsvrz.dav.daf.
 *
 * de.bsvrz.dav.daf is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.dav.daf is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with de.bsvrz.dav.daf; If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Martin-Luther-Straße 14
 * 52062 Aachen, Germany
 * phone: +49 241 4090 436
 * mail: <info@kappich.de>
 */
package de.bsvrz.dav.daf.util;

import java.util.Objects;
import java.util.function.Supplier;

/**
 * Ersatz für Suppliers.memoize() von Guava
 *
 * @param <T>
 */
public class Memoizer<T> implements Supplier<T> {
	private static final Supplier<Object> SUCCESS =
			() -> {
				throw new IllegalStateException();
			};

	private T value;
	private volatile Supplier<T> supplier;

	/**
	 * Factory-Methode benutzen, um Instanzen zu erstellen
	 *
	 * @param supplier Delegate-Supplier
	 */
	Memoizer(Supplier<T> supplier) {
		this.supplier = Objects.requireNonNull(supplier);
	}

	@Override
	@SuppressWarnings("unchecked")
	public T get() {
		if (supplier != SUCCESS) {
			synchronized (this) {
				if (supplier != SUCCESS) {
					T t = supplier.get();
					value = t;
					supplier = (Supplier<T>) SUCCESS;
					return t;
				}
			}
		}
		return value;
	}

	@Override
	public String toString() {
		if (supplier == SUCCESS) {
			return "Memoizer{" + value + '}';
		}
		return "Memoizer{" + supplier + '}';
	}

	/**
	 * Erzeugt einen Supplier, der den übergeben Supplier als Delegate maximal einmal ausführt und sich den erhaltenen Wert merkt.
	 *
	 * @param delegate Delegate
	 * @param <T>      Typ des Supplier-Ergebnisses
	 * @return Supplier
	 */
	public static <T> Supplier<T> memoize(Supplier<T> delegate) {
		return new Memoizer<>(delegate);
	}
}
