/*
 * Copyright 2005 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.dataview;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientReceiverInterface;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.pat.sysbed.dataview.csv.CsvProgressDialogOnline;
import de.bsvrz.pat.sysbed.dataview.csv.CsvUtils;
import de.bsvrz.pat.sysbed.dataview.csv.PerpetualCsvConverter;
import de.bsvrz.pat.sysbed.dataview.csv.PostProcessor;
import de.bsvrz.pat.sysbed.dataview.filtering.FilterAttributeGroup;
import de.bsvrz.pat.sysbed.main.GenericTestMonitorApplication;
import de.bsvrz.pat.sysbed.plugins.api.settings.SettingsData;
import de.bsvrz.sys.funclib.debug.Debug;
import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Container;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.swing.WindowConstants;

/**
 * Ein Dialog, welcher die gelieferten Online- oder Archivdaten in Tabellenform dargestellt.
 *
 * @author Kappich Systemberatung
 * @see DataViewPanel
 */
public class DataViewFrame implements PrintFrame {

    /** hängt einen neuen Datensatz unten an die Dargestellten an */
    @SuppressWarnings("unused")
    private static final int ADD_BELOW = 0;

    /** fügt einen neuen Datensatz oben vor die Dargestellten ein */
    private static final int ADD_ABOVE = 1;

    /** stellt nur den neuesten Datensatz dar */
    private static final int ONLY_LATEST = 2;

    private final Debug _debug = Debug.getLogger();

    private final UnsubscribingJFrame _frame;

    private final DataViewModel _dataViewModel;

    private final DataViewPanel _dataViewPanel;

    private final ClientDavInterface _connection;

    private final List<SystemObject> _objects;

    private final FilterAttributeGroup _filterAttributeGroup;

    private final Aspect _aspect;

    private final DataDescription _dataDescription;

    private ReceiveOptions _receiveOptions = ReceiveOptions.normal();

    private ReceiverRole _receiverRole = ReceiverRole.receiver();

    private Component _parent;

//	/**
//	 * Konstruktor, der anhand der Datenidentifikation sich beim Datenverteiler anmeldet und die Daten in Tabellenform darstellt.
//	 *
//	 * @param connection        Verbindung zum Datenverteiler
//	 * @param objects           die zu betrachtenden Systemobjekte
//	 * @param filterAttributeGroup    die zu betrachtende Filter-Attributgruppe
//	 * @param aspect            der zu betrachtende Aspekt
//	 * @param simulationVariant die Simulationsvariante
//	 */
//	public DataViewFrame(
//			final ClientDavInterface connection,
//			final SystemObject[] objects,
//			final FilterAttributeGroup filterAttributeGroup,
//			final Aspect aspect,
//			int simulationVariant) {
//		this(connection, Arrays.asList(objects), filterAttributeGroup, aspect, simulationVariant);
//	}

    /**
     * Konstruktor, der anhand der Datenidentifikation sich beim Datenverteiler anmeldet und die Daten in Tabellenform darstellt.
     *
     * @param connection           Verbindung zum Datenverteiler
     * @param objects              die zu betrachtenden Systemobjekte
     * @param filterAttributeGroup die Filter-Attributgruppe
     * @param aspect               der zu betrachtende Aspekt
     * @param simulationVariant    die Simulationsvariante
     */
    public DataViewFrame(final ClientDavInterface connection, final List<SystemObject> objects, final FilterAttributeGroup filterAttributeGroup,
                         final Aspect aspect, int simulationVariant) {
        _connection = connection;
        _objects = objects;
        _filterAttributeGroup = filterAttributeGroup;
        _aspect = aspect;

        if (simulationVariant != -1) {
            _dataDescription = new DataDescription(_filterAttributeGroup.getAttributeGroup(), _aspect, (short) simulationVariant);
        } else {
            _dataDescription = new DataDescription(_filterAttributeGroup.getAttributeGroup(), _aspect);
        }

        System.getProperties().put("apple.laf.useScreenMenuBar", "true");

        _dataViewModel = new DataViewModel(_filterAttributeGroup);
        _dataViewPanel = new DataViewPanel(_dataViewModel);
        _dataViewModel.addDataViewListener(_dataViewPanel);

        _frame = new UnsubscribingJFrame(_connection, _objects, _dataDescription);
        _frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);

        MenuBar menuBar = new MenuBar(_frame, _dataViewModel, _filterAttributeGroup, _aspect, _dataViewPanel, _connection, _dataDescription, false);
        menuBar.createMenuBar(_dataViewPanel.getSelectionManager());

        Container pane = _frame.getContentPane();
        pane.setLayout(new BorderLayout());

        pane.add(_dataViewPanel, BorderLayout.CENTER);
        _frame.setSize(1000, 300);
    }

    /**
     * @param connection           die Datenverteiler-Verbindung
     * @param filterAttributeGroup die Filter-Attributgruppe
     * @param aspect               der Aspekt
     * @param simulationVariant    die Simualtionsvariante
     */
    public DataViewFrame(final ClientDavInterface connection, final FilterAttributeGroup filterAttributeGroup, final Aspect aspect,
                         int simulationVariant) {
        _connection = connection;
        _objects = new ArrayList<>();
        _filterAttributeGroup = filterAttributeGroup;
        _aspect = aspect;
        if (simulationVariant != -1) {
            _dataDescription = new DataDescription(_filterAttributeGroup.getAttributeGroup(), _aspect, (short) simulationVariant);
        } else {
            _dataDescription = new DataDescription(_filterAttributeGroup.getAttributeGroup(), _aspect);
        }

        System.getProperties().put("apple.laf.useScreenMenuBar", "true");

        _dataViewModel = new DataViewModel(_filterAttributeGroup);
        _dataViewPanel = new DataViewPanel(_dataViewModel);
        _dataViewModel.addDataViewListener(_dataViewPanel);
        _dataViewPanel.getSelectionManager().removeSelectionListeners();

        _frame = new UnsubscribingJFrame(_connection, _objects, _dataDescription);
        _frame.setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
        final Container pane = _frame.getContentPane();
        pane.setLayout(new BorderLayout());
    }

    /**
     * Gibt das UnsubscribingJFrame-Objekt zurück.
     *
     * @return das UnsubscribingJFrame-Objekt
     */
    @Override
    public UnsubscribingJFrame getFrame() {
        return _frame;
    }

    /**
     * Gibt das DataViewPanel zurück.
     *
     * @return das DataViewPanel
     */
    @Override
    public DataViewPanel getDataViewPanel() {
        return _dataViewPanel;
    }

    /**
     * Setzt die Empfängeroptionen.
     *
     * @param receiveOptions die Empfängeroptionen
     */
    public void setReceiveOptions(final ReceiveOptions receiveOptions) {
        _receiveOptions = receiveOptions;
    }

    /**
     * Setzt die Empfängerrolle.
     *
     * @param receiverRole die Empfängerrolle
     */
    public void setReceiverRole(final ReceiverRole receiverRole) {
        _receiverRole = receiverRole;
    }

    /**
     * Zeigt die in der Liste übergebenen konfigurierenden Daten an.
     *
     * @param configuringData die konfigurierenden Daten
     */
    public void showConfigurationData(final List<DataTableObject> configuringData) {
        _frame.setTitle(GenericTestMonitorApplication.getTitle(
            "Konfigurierende Daten (Attributgruppe: " + _filterAttributeGroup.getNameOrPidOrId() + ", Aspekt: " + _aspect.getNameOrPidOrId() + ")",
            _connection));
        _frame.setVisible(true);
        _dataViewModel.setDatasets(configuringData);
    }

    /**
     * Mit dieser Methode kann man eine Parent-Component setzen.
     *
     * @param parent
     */
    public void setParent(final Component parent) {
        _parent = parent;
    }

    /**
     * Zeigt die Onlinedaten der angemeldeten Datenidentifikation an. Der Parameter gibt an, an welcher Stelle neue Daten eingefügt werden sollen. Zur
     * Auswahl stehen: <ul> <li>0: unten anhängen</li> <li>1: Aktuelle Daten oben einfügen</li> <li>2: Nur aktuellste Daten anzeigen</li> <li>3: Daten
     * nur in einer CSV-Datei speichern</li></ul> Hierbei fällt die letzte Option natürlich aus dem Rahmen, da keine Daten angezeigt werden.
     *
     * @param displayOptions gibt an, wie neue Daten dargestellt werden sollen
     */
    public void showOnlineData(int displayOptions, String charsetName, SettingsData settingsData) {
        if (displayOptions < 3) {
            final String title;
            if (_filterAttributeGroup.getAtgFilter() == null) {
                title =
                    "Onlinetabelle (Attributgruppe: " + _filterAttributeGroup.getNameOrPidOrId() + ", Aspekt: " + _aspect.getNameOrPidOrId() + ")";
            } else {
                title = "Onlinetabelle (Attributgruppe: " + _filterAttributeGroup.getNameOrPidOrId() + ", Aspekt: " + _aspect.getNameOrPidOrId() +
                        ", Filter: " + _filterAttributeGroup.getAtgFilter().getName() + ")";
            }
            _frame.setTitle(GenericTestMonitorApplication.getTitle(title, _connection));
            _frame.setVisible(true);
            try {
                ClientReceiverInterface receiver = new DataViewReceiver(displayOptions);
                _connection.subscribeReceiver(receiver, _objects, _dataDescription, _receiveOptions, _receiverRole);
                _frame.setReceiver(receiver, _receiveOptions, _receiverRole);
            } catch (RuntimeException ex) {
                _frame.setVisible(false);
                _frame.dispose();
                _debug.error(
                    "Beim Öffnen einer neuen Onlinetabelle ist bei der Anmeldung der gewünschten Datenidentifikationen ein Fehler aufgetreten " +
                    "(siehe Exception)", ex);
                throw new IllegalStateException(ex.getMessage(), ex);
            }
        } else if (displayOptions == 3) {    // Daten nur in CSV-Datei speichern
            final File csvFile = CsvUtils.getCSVFileForExport(_parent);
            if (csvFile == null) {
                // Wenn der Benutzer keine Csv-Datei gewählt hat;
                // auf das Öffnen eines JOptionPanes wird hier bewusst verzichtet.
                return;
            }

            final CsvProgressDialogOnline progressDialog =
                new CsvProgressDialogOnline(csvFile.getName(), settingsData.getAttributeGroup(), settingsData.getAspect(),
                                            settingsData.getSimulationVariant(), settingsData.getObjects());
            DataExportReceiver receiver = new DataExportReceiver(csvFile, progressDialog, charsetName);
            try {
                receiver.openFile();
            } catch (FileNotFoundException ignore) {
                JOptionPane
                    .showMessageDialog(getFrame(), "Fehler: die Datei " + csvFile.getName() + " konnte nicht geöffnet werden.", "Fehlermeldung",
                                       JOptionPane.ERROR_MESSAGE);
                return;
            }
            progressDialog.addStopRequestListener(receiver);
            receiver.subscribe();
            progressDialog.setLocationRelativeTo(_parent);
            progressDialog.setVisible(true);
        }
    }

    @Override
    public String toString() {
        return "DataViewFrame{" + "_objects=" + _objects + ", _filterAttributeGroup=" + _filterAttributeGroup + ", _aspect=" + _aspect +
               ", _dataDescription=" + _dataDescription + ", _receiveOptions=" + _receiveOptions + ", _receiverRole=" + _receiverRole + '}';
    }

    /**
     * Die Klasse verarbeitet die Daten, die vom Datenverteiler gesandt werden, und kommt bei Darstellung der OnlineTabelle zum Zug
     */
    private class DataViewReceiver implements ClientReceiverInterface {

        private int _displayOptions;

        /**
         * Mit dem Konstruktor wird ein Parameter übergeben, der angibt, an welcher Stelle neue Daten eingefügt werden sollen. Zur Auswahl stehen:
         * <ul> <li>0: unten einfügen</li> <li>1: oben einfügen</li> <li>2: nur neueste Daten anzeigen</li> </ul>
         *
         * @param displayOptions gibt an, wie neue Daten dargestellt werden sollen
         */
        public DataViewReceiver(int displayOptions) {
            _displayOptions = displayOptions;
        }

        @Override
        public void update(ResultData[] results) {
            // aktueller Speicherverbrauch wird ausgegeben
            //			final LapStatistic statistic = new LapStatistic();
            //			statistic.printLapResultWithGc("UpdateDatasets (" + _counter++ + ")");

            if (_displayOptions == ADD_ABOVE) {
                for (final ResultData result : results) {
                    _dataViewModel.addDatasetAbove(new DataTableObject(result, _filterAttributeGroup));
                }
            } else if (_displayOptions == ONLY_LATEST) {
                List<DataTableObject> dataTableObjects = new LinkedList<>();
                for (final ResultData result : results) {
                    dataTableObjects.add(new DataTableObject(result, _filterAttributeGroup));
                }
                _dataViewModel.updateDatasets(dataTableObjects);
            } else {
                List<DataTableObject> dataTableObjects = new LinkedList<>();
                for (final ResultData result : results) {
                    dataTableObjects.add(new DataTableObject(result, _filterAttributeGroup));
                }
                _dataViewModel.addDatasetsBelow(dataTableObjects);
            }
        }
    }

    /**
     * Die Klasse verarbeitet die Daten, die vom Datenverteiler gesandt werden, und kommt beim CSV-Export der Daten zum Zug
     */
    public class DataExportReceiver implements ClientReceiverInterface, CsvProgressDialogOnline.StopRequestListener {

        private final File _csvFile;
        private final PerpetualCsvConverter _csvConverter;
        private final CsvProgressDialogOnline _progressDialog;
        private final String _charsetName;
        private OutputStreamWriter _fileWriter;
        private int _numberOfHeaderLines;
        private int _numberOfDataTableObjects;

        DataExportReceiver(final File csvFile, CsvProgressDialogOnline progressDialog, String charsetName) {
            _csvFile = csvFile;
            _csvConverter = new PerpetualCsvConverter(_filterAttributeGroup);
            _progressDialog = progressDialog;
            _charsetName = charsetName;
            _csvConverter.setDelimiter(";");
            _numberOfHeaderLines = 0;
            _numberOfDataTableObjects = 0;
        }

        public void openFile() throws FileNotFoundException {
            if (Charset.isSupported(_charsetName)) {
                _fileWriter = new OutputStreamWriter(new FileOutputStream(_csvFile), Charset.forName(_charsetName));
            } else {
                _fileWriter = new OutputStreamWriter(new FileOutputStream(_csvFile));
            }
            _numberOfHeaderLines = 0;
            _numberOfDataTableObjects = 0;
        }

        @Override
        public void update(final ResultData[] results) {
            if (null == _fileWriter) {
                return;
            }
            List<DataTableObject> dataTableObjects = new ArrayList<>(results.length);
            for (ResultData resultData : results) {
                final DataTableObject dataTableObject = new DataTableObject(resultData, _filterAttributeGroup);
                dataTableObjects.add(dataTableObject);
                SwingUtilities.invokeLater(() -> _progressDialog.newTimeStamp(dataTableObject.getDataTime()));
            }
            _numberOfDataTableObjects += dataTableObjects.size();
            SwingUtilities.invokeLater(() -> _progressDialog.setNumberOfDataTableObjects(_numberOfDataTableObjects));
            boolean columnStructureChanged = _csvConverter.setData(dataTableObjects, null);
            try {
                if (columnStructureChanged) {
                    _fileWriter.write(_csvConverter.getCsvHeaderLine(true));
                    // Wird dies mit false aufgerufen, so muss das PostProcessing geändert werden.
                    ++_numberOfHeaderLines;
                }
                _fileWriter.write(_csvConverter.getCsvLines(true));
            } catch (IOException e) {
                _debug.error("Das Schreiben der CSV-Datei schlug fehl (OutputStreamWriter.write()).", e);
            }
        }

        @Override
        public void stopRequested() {
            unsubscribe();
            try {
                _fileWriter.flush();
            } catch (IOException e) {
                _debug.error("Das Schreiben der CSV-Datei schlug fehl (OutputStreamWriter.flush().", e);
            }
            try {
                _fileWriter.close();
            } catch (IOException e) {
                _debug.error("Das Schließen der CSV-Datei schlug fehl (OutputStreamWriter.close().", e);
            }
            if (_numberOfHeaderLines > 1) { // Post-Processing-Bedingung
                _progressDialog.addPostProcessingLayout();

                PostProcessor postProcessor = new PostProcessor(_csvFile, _csvConverter, _progressDialog, _charsetName);
                postProcessor.execute();
            } else {
                _progressDialog.dispose();
            }
        }

        public void subscribe() {
            _connection.subscribeReceiver(this, _objects, _dataDescription, _receiveOptions, _receiverRole);
        }

        public void unsubscribe() {
            _connection.unsubscribeReceiver(this, _objects, _dataDescription);
        }
    }
}
