/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.dataview.csv;

import de.bsvrz.pat.sysbed.dataview.DataTableObject;
import de.bsvrz.pat.sysbed.dataview.DataTableObjectRenderer;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.regex.Pattern;

public abstract class AbstractCsvConverter implements CsvConverter {

    private static final Pattern QUOTE_SIGN_PATTERN = Pattern.compile("\"");
    private static final String TIME_FORMAT = "dd.MM.yyyy HH:mm:ss.SSS";    // ss.SSS: Punbkt statt Komma wg Excel
    /**
     * Enthält die benötigten Datensätze in der richtigen Reihenfolge
     */
    final List<DataTableObject> _dataTableObjects = new ArrayList<>();
    /**
     * Enthält die CSV-Spalten als Schlüssel und als Wert je eine HashMap, die wiederum als Schlüssel eine Datensatzkennung enthält und als Wert den
     * Text der Zelle.
     */
    final Map<CsvColumn, Map<DataTableObject.DataTableObjectId, String>> _csvColumnToText = new TreeMap<>();
    private String _delimiter = ";";

    /**
     * Gibt das Trennzeichen, das in der Header- und den Datenzeilen verwendet wird, zurück.
     *
     * @return das Trennzeichen
     */
    @Override
    public String getDelimiter() {
        return _delimiter;
    }

    /**
     * Setzt das Trennzeichen, das in der Header- und den Datenzeilen verwendet wird.
     *
     * @param delimiter ein Trennzeichen
     */
    @Override
    public void setDelimiter(final String delimiter) {
        _delimiter = delimiter;
    }

    /**
     * Gibt die Titelzeile der CSV-Datei zurück. Ist der übergebene Wert {@code true}, so werden auch die Spalten der Zeilenköpfe (Art, Zeit und
     * Objekt) aufgeführt.
     * <p>
     * Wurde im Konstruktor eine von {@code null} verschiedene Collection von CellKeys angegeben, so wirkt diese als Filter auf Spalten, Zeilen und
     * Zellen. Für die Header-Zeile heißt dies: es treten nur Spalten auf, für die mindestens ein selektierter CellKey existiert.
     *
     * @return die Header-Zeile
     */
    @Override
    public String getCsvHeaderLine(final boolean rowHeader) {
        final StringBuilder sb = new StringBuilder();
        if (rowHeader) {
            sb.append("Art").append(_delimiter).append("Zeit").append(_delimiter).append("Objekt").append(_delimiter);
        }
        for (CsvColumn column : _csvColumnToText.keySet()) {
            final String name = column.getName();
            if (!name.isEmpty()) {    // Der leere String entsteht nur bei den Keine-Daten-Datensätzen
                sb.append(name).append(_delimiter);
            }
        }
        sb.append("\n");
        return sb.toString();
    }

    /**
     * Gibt die Zeilen der CSV-Datei, die die Inhalte enthalten, zurück. Ist der übergebene Wert {@code true}, so werden auch die Inhalte der
     * Zeilenköpfe (Art, Zeit und Objekt) aufgeführt.
     * <p>
     * Wurde im Konstruktor eine von {@code null} verschiedene Collection von CellKeys angegeben, so wirkt diese als Filter auf Spalten, Zeilen und
     * Zellen. Für die Datenzeilen heißt dies, dass nur die Inhalte von Zellen mit selektierten CellKeys ausgegeben werden, und dass leere Spalten und
     * leere Zeilen gar nicht ausgeben werden.
     *
     * @return ein String mit allen Datenzeilen
     */
    @Override
    public String getCsvLines(final boolean rowHeader) {
        final StringBuilder buffer = new StringBuilder();
        for (DataTableObject dataTableObject : _dataTableObjects) {
            final StringBuffer csvLineForDataTableObject = getCsvLineForDataTableObject(dataTableObject, rowHeader);
            if (csvLineForDataTableObject != null) {
                buffer.append(csvLineForDataTableObject);
            }
        }
        return buffer.toString();
    }

    @Nullable
    private StringBuffer getCsvLineForDataTableObject(@NotNull final DataTableObject dataTableObject, final boolean rowHeader) {
        // Aus RFC 4180: The last field in the record must not be followed by a comma.
        final StringBuffer buffer = new StringBuffer();
        if (rowHeader) {
            buffer.append(encodeForCsv(DataTableObjectRenderer.getDatakindText(dataTableObject.getDataKind())));
            buffer.append(_delimiter);
            buffer.append(encodeForCsv(dataTableObject.getTimeText(TIME_FORMAT)));
            buffer.append(_delimiter);
            buffer.append(encodeForCsv(dataTableObject.getObject().getNameOrPidOrId()));
        }
        // Die Vorgehensweise ist nun wie folgt: wir haben ein DataTableObject, die _csvColumns, sowie die _selectedCellKeys.
        // Wir iterieren über _csvColumns und gucken, ob der entsprechende CellKey selektiert ist; wenn nicht, so gibt es einen
        // leeren Spalteneintrag. Wenn doch, so holen wir uns von dem DataTableObject den Wert und benutzen ihn.
        if (dataTableObject.getData() != null) {    // normaler Datensatz
            boolean lineIsRelevant = false;    // relevant = mind. ein CellKey wurde selektiert (false dürfte nicht mehr auftreten)
            for (final Map.Entry<CsvColumn, Map<DataTableObject.DataTableObjectId, String>> entry : _csvColumnToText.entrySet()) {
                final Map<DataTableObject.DataTableObjectId, String> theDataIndexSet = entry.getValue();
                final String cellText = theDataIndexSet.get(dataTableObject.getDataTableObjectId());
                buffer.append(_delimiter);
                if (cellText != null) {
                    buffer.append(encodeForCsv(cellText));
                    lineIsRelevant = true;
                }
            }
            if (lineIsRelevant) {
                buffer.append("\n");
                return buffer;
            } else {
                return null;
            }
        } else {
            buffer.append(_delimiter);
            buffer.append(encodeForCsv(DataTableObjectRenderer.getTextForState(dataTableObject.getDataState())));
            buffer.append("\n");
            return buffer;
        }
    }

    private String encodeForCsv(final String s) {
        String r = s;
        if (r.contains("\"")) {
            r = QUOTE_SIGN_PATTERN.matcher(r).replaceAll("\"\"");
        }
        if (r.contains(";") || r.contains(_delimiter) || r.contains("\n") || r.contains(" ") || r.contains("\"")) {
            r = "\"" + r + "\"";
        }
        return r;
    }
}
