/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.dataview.csv;

import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.pat.sysbed.plugins.api.DataIdentificationChoice;
import java.awt.BorderLayout;
import java.awt.Component;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JProgressBar;
import javax.swing.SpringLayout;
import javax.swing.WindowConstants;

/**
 * @author Kappich Systemberatung
 */
public class CsvProgressDialogArchive extends JFrame implements CsvProgressDialog {

    private final String _fileName;
    private final DataIdentificationChoice _dataIdentificationChoice;
    private JProgressBar _progressBar;
    private JLabel _numberOfDataTableObjects;
    private JPanel _panel;
    private JButton _stopButton;
    private JButton _interruptButton;
    private Set<StopRequestListener> _stopListeners;
    private Set<InterruptRequestListener> _interruptListeners;

    public CsvProgressDialogArchive(String fileName, final AttributeGroup attributeGroup, final Aspect aspect, final int simulationVariant,
                                    final List<SystemObject> objectList) {
        _fileName = fileName;
        _dataIdentificationChoice = new DataIdentificationChoice(attributeGroup, aspect, simulationVariant, objectList);
        initLayout();
        initLogic();
        setDefaultCloseOperation(WindowConstants.DISPOSE_ON_CLOSE);
    }

    private void initLayout() {
        setTitle("Archivdaten in " + _fileName + " speichern");

        _panel = new JPanel();
        _panel.setLayout(new BorderLayout());

        JLabel label1 = new JLabel("Fortschritt: ");
        JLabel label2 = new JLabel("Anzahl Datensätze: ");
        _progressBar = new JProgressBar(0, 100);
        _progressBar.setStringPainted(true);
        _numberOfDataTableObjects = new JLabel("0");
        _stopButton = new JButton("Beenden");
        _interruptButton = new JButton("Abbrechen");

        JPanel progressPanel = new JPanel();
        progressPanel.setLayout(new SpringLayout());
        progressPanel.setBorder(BorderFactory.createEmptyBorder(20, 10, 0, 0));
        progressPanel.add(label1);
        progressPanel.add(_progressBar);
        progressPanel.add(label2);
        progressPanel.add(_numberOfDataTableObjects);
        CsvUtils.makeCompactGrid(progressPanel, 2, 10, 20);

        JPanel stopButtonPanel = new JPanel();
        stopButtonPanel.add(Box.createHorizontalGlue());
        stopButtonPanel.add(_stopButton);
        stopButtonPanel.add(Box.createHorizontalGlue());

        _panel.add(_dataIdentificationChoice, BorderLayout.NORTH);
        _panel.add(progressPanel, BorderLayout.CENTER);
        _panel.add(stopButtonPanel, BorderLayout.SOUTH);

        add(_panel);
        pack();
//		setSize(new Dimension(450, 150));
    }

    private void initLogic() {
        _stopListeners = new HashSet<>();
        _stopButton.addActionListener(e -> {
            for (StopRequestListener listener : _stopListeners) {
                listener.stopRequested();
            }
            dispose();
        });
        _interruptListeners = new HashSet<>();
        _interruptButton.addActionListener(e -> {
            for (InterruptRequestListener listener : _interruptListeners) {
                listener.interruptRequested();
            }
        });
    }

    public void addPostProcessingLayout() {
        _stopButton.setEnabled(false);
        _panel.removeAll();
        _progressBar = new JProgressBar(0, 100);
        _progressBar.setStringPainted(true);

        JLabel label = new JLabel("Nachbearbeitung: ");

        JPanel progressPanel = new JPanel();
        progressPanel.setBorder(BorderFactory.createEmptyBorder(20, 10, 0, 0));
        progressPanel.add(label);
        progressPanel.add(_progressBar);

        JPanel stopButtonPanel = new JPanel();
        stopButtonPanel.add(Box.createHorizontalGlue());
        stopButtonPanel.add(_interruptButton);
        stopButtonPanel.add(Box.createHorizontalGlue());

        _panel.add(_dataIdentificationChoice, BorderLayout.NORTH);
        _panel.add(progressPanel, BorderLayout.CENTER);
        _panel.add(stopButtonPanel, BorderLayout.SOUTH);
        revalidate();
        repaint();
    }

    public void setProgress(final int progress, final int numberOfDataTableRecords) {
        _progressBar.setValue(Math.max(0, Math.min(progress, 100)));
        _numberOfDataTableObjects.setText(Integer.toString(numberOfDataTableRecords));
    }

    @Override
    public Component getComponent() {
        return this;
    }

    @Override
    public void disposeComponent() {
        dispose();
    }

    @Override
    public void setPostProcessingProgress(final int value) {
        _progressBar.setValue(Math.max(0, Math.min(value, 100)));
    }

    @Override
    public void addInterruptRequestListener(final InterruptRequestListener listener) {
        _interruptListeners.add(listener);
    }

    @Override
    public boolean removeInterruptRequestListener(final InterruptRequestListener listener) {
        return _interruptListeners.remove(listener);
    }

    public void addStopRequestListener(final StopRequestListener listener) {
        _stopListeners.add(listener);
    }

    @SuppressWarnings("unused")
    public boolean removeStopRequestListener(final StopRequestListener listener) {
        return _stopListeners.remove(listener);
    }

    @Override
    public String toString() {
        return "CsvProgressDialogArchive{" + "_fileName='" + _fileName + '\'' + '}';
    }

    public interface StopRequestListener {
        void stopRequested();
    }
}
