/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.dataview.csv;

import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.kappich.filechooser.AwtFileChooser;
import java.awt.Component;
import java.awt.Container;
import java.io.File;
import javax.swing.JFileChooser;
import javax.swing.JOptionPane;
import javax.swing.Spring;
import javax.swing.SpringLayout;
import javax.swing.filechooser.FileNameExtensionFilter;

/**
 * @author Kappich Systemberatung
 */
public final class CsvUtils {

    @Nullable
    public static File getCSVFileForExport(@Nullable final Component parent) {
        final JFileChooser fileChooser;
        if (System.getProperty("os.name").toLowerCase().startsWith("mac")) {
            fileChooser = AwtFileChooser.createFileChooser();
        } else {
            fileChooser = new JFileChooser();
        }
        FileNameExtensionFilter filter = new FileNameExtensionFilter("CSV-Datei", "csv");
        fileChooser.setFileFilter(filter);
        fileChooser.setDialogTitle("Daten nur in einer CSV-Datei speichern");

        File csvFile;
        while (true) {
//			int showSaveDialog = fileChooser.showOpenDialog(parent, "Auswählen");
            int showSaveDialog = fileChooser.showSaveDialog(parent);
            if (showSaveDialog == JFileChooser.APPROVE_OPTION) {
                File selectedFile = fileChooser.getSelectedFile();
                String path = selectedFile.getPath();

                if (!path.toLowerCase().endsWith(".csv")) {
                    path += ".csv";
                }
                csvFile = new File(path);

                if (csvFile.exists()) {
                    int n = JOptionPane.showConfirmDialog(parent,
                                                          "Die Datei '" + csvFile.getName() + "' existiert bereits." + System.lineSeparator() +
                                                          "Datei überschreiben?", "Warnung", JOptionPane.YES_NO_OPTION);
                    if (n == JOptionPane.YES_OPTION) {
                        break;
                    }
                } else {
                    break;
                }
            } else {
                return null;
            }
        }
        return csvFile;
    }

	/*
	Achtung: Die kommenden zwei Methoden sind aus der GND übernommen.
	 */

    private static SpringLayout.Constraints getConstraints(int row, int column, Container container, int columns) {
        SpringLayout layout = (SpringLayout) container.getLayout();
        Component c = container.getComponent(row * columns + column);
        return layout.getConstraints(c);
    }

    /**
     * Ordnet die Komponenten in einem Container in einem Grid so an, dass die Spaltenbreiten der maximalen bevorzugten Breite der enthaltenen
     * Komponenten entspricht und die Zeilenhöhen der maximalen bevorzugten Höhe der jeweils enthalten Komponenten entspricht. Die Containergröße wird
     * so eingestellt, dass alle Komponenten reinpassen.
     *
     * @param columns   Anzahl Spalten
     * @param columnGap Gewünschter Abstand zwischen den Spalten
     * @param rowGap    Gewünschter Abstand zwischen den Zeilen
     *
     * @throws ClassCastException    wenn der Container kein SpringLayout verwendet.
     * @throws IllegalStateException wenn die Anzahl der im Container enthaltenen Komponenten nicht ohne Rest durch die Anzahl Spalten teilbar ist.
     */
    @SuppressWarnings("SameParameterValue")
    public static void makeCompactGrid(Container container, int columns, int columnGap, int rowGap) {
        SpringLayout layout = (SpringLayout) container.getLayout();
        int numberOfComponents = container.getComponentCount();
        if (numberOfComponents % columns != 0) {
            throw new IllegalStateException(
                "Anzahl der Komponenten " + numberOfComponents + " ist nicht durch die Anzahl der Spalten (" + columns + ") teilbar");
        }
        int rows = numberOfComponents / columns;
        Spring x = Spring.constant(0);
        Spring y = Spring.constant(0);
        for (int column = 0; column < columns; ++column) {
            Spring width = Spring.constant(0);
            for (int row = 0; row < rows; ++row) {
                width = Spring.max(width, getConstraints(row, column, container, columns).getWidth());
            }
            for (int row = 0; row < rows; ++row) {
                SpringLayout.Constraints constraints = getConstraints(row, column, container, columns);
                constraints.setX(x);
                constraints.setWidth(width);
            }
            x = Spring.sum(x, Spring.sum(width, Spring.constant(columnGap)));
        }
        for (int row = 0; row < rows; ++row) {
            Spring height = Spring.constant(0);
            for (int column = 0; column < columns; ++column) {
                height = Spring.max(height, getConstraints(row, column, container, columns).getHeight());
            }
            for (int column = 0; column < columns; ++column) {
                SpringLayout.Constraints constraints = getConstraints(row, column, container, columns);
                constraints.setY(y);
                constraints.setHeight(height);
            }
            y = Spring.sum(y, Spring.sum(height, Spring.constant(rowGap)));
        }
        SpringLayout.Constraints containerConstraints = layout.getConstraints(container);
        containerConstraints.setConstraint(SpringLayout.SOUTH, y);
        containerConstraints.setConstraint(SpringLayout.EAST, x);
    }
}
