/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.dataview.csv;

import de.bsvrz.pat.sysbed.dataview.DataTableObject;
import de.bsvrz.pat.sysbed.dataview.filtering.FilterAttributeGroup;
import de.bsvrz.pat.sysbed.dataview.selectionManagement.CellKey;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * <p> Der PerpetualCsvConverter bietet eine Implementation des CsvConverters, die fortlaufend CSV-Zeilen exportiert. Dabei beachtet dieser Konverter
 * zur Bestimmung des Zeilenformats alle früher gesehenen {@link DataTableObject DataTableObjects}. Das Problem sind nämlich Arrays variierender
 * Größe; dieser Konverter stellt sicher, dass keine Spalten verschwinden, soll heißen: existierten Spalten in einem früher behandelten {@link
 * DataTableObject}, so hat jede neue CSV-Zeile auch diese Spalte, und sei es mit einem leeren Wert.</p> <p> Der PerpetualCsvConverter beachtet keine
 * Selektion von {@link CellKey CellKeys.}</p>
 */
public class PerpetualCsvConverter extends AbstractCsvConverter {

    private final FilterAttributeGroup _filterAttributeGroup;

    private int _rowCounter;

    private boolean _ensureStructureChangedArFirstLine;

    /**
     * Jeder PerpetualCsvConverter kann nur für eine Attributgruppe verwendet werden, die nicht veränderbar ist.
     *
     * @param filterAttributeGroup
     */
    public PerpetualCsvConverter(final FilterAttributeGroup filterAttributeGroup) {
        _filterAttributeGroup = filterAttributeGroup;
        _rowCounter = 0;
        _ensureStructureChangedArFirstLine = true;
    }

    @Override
    public boolean setData(final List<DataTableObject> dataTableObjects, @Nullable final Set<CellKey> selectedCellKeys) {
        synchronized (_dataTableObjects) {
            _dataTableObjects.clear();
            _dataTableObjects.addAll(dataTableObjects);
        }
        _rowCounter += _dataTableObjects.size();
        return updateCsvColumns();
    }

    public int getNumberOfCsvRows() {
        return _rowCounter;
    }

    ////////////////////////////////////////////////////////////////////////
    // private Methoden
    ////////////////////////////////////////////////////////////////////////

    private boolean updateCsvColumns() {
        for (Map.Entry<CsvColumn, Map<DataTableObject.DataTableObjectId, String>> entry : _csvColumnToText.entrySet()) {
            entry.getValue().clear();
        }
        boolean structureChanged = false;
        for (DataTableObject dataTableObject : _dataTableObjects) {
            for (CellKey cellKey : dataTableObject.getAllCellKeys()) {
                if (!cellKey.isSuperColumn()) {
                    final CsvColumn column = new CsvColumn(_filterAttributeGroup, cellKey);
                    final DataTableObject.DataTableObjectId dataTableObjectId = dataTableObject.getDataTableObjectId();
                    if (_csvColumnToText.containsKey(column)) {
                        _csvColumnToText.get(column).put(dataTableObjectId, cellKey.getCellText());
                    } else {
                        structureChanged = true;
                        final Map<DataTableObject.DataTableObjectId, String> newMap = new HashMap<>();
                        newMap.put(dataTableObjectId, cellKey.getCellText());
                        _csvColumnToText.put(column, newMap);
                    }
                }
            }
        }
        // Hier wird garantiert, dass die ersten Zeilen immer einen Rückgabewert true haben.
        // Das ist wichtig, wenn _dataTableObjects beim ersten Mal nur superColums enthält.
        if (_ensureStructureChangedArFirstLine && !_dataTableObjects.isEmpty()) {
            _ensureStructureChangedArFirstLine = false;
            return true;
        } else {
            return structureChanged;
        }
    }

    @Override
    public String toString() {
        return "PerpetualCsvConverter{}";
    }
}
