/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.dataview.csv;

import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ExecutionException;
import javax.swing.JOptionPane;
import javax.swing.SwingWorker;

/**
 * @author Kappich Systemberatung
 */
public class PostProcessor extends SwingWorker<String, Integer> implements CsvProgressDialog.InterruptRequestListener {

    private final Debug _debug = Debug.getLogger();

    private final File _csvFile;
    private final PerpetualCsvConverter _csvConverter;
    private final CsvProgressDialog _progressDialog;
    private final String _charsetName;
    private boolean _interruptRequested;

    public PostProcessor(final File csvFile, final PerpetualCsvConverter csvConverter, final CsvProgressDialog progressDialog,
                         final String charsetName) {
        _csvFile = csvFile;
        _csvConverter = csvConverter;
        _progressDialog = progressDialog;
        _charsetName = charsetName;
        _interruptRequested = false;
        _progressDialog.addInterruptRequestListener(this);
    }

    /**
     * Zerlegt eine Zeile in ein String-Array mit den einzelnen Einträgen
     *
     * @param line   Zeile
     * @param reader Reader (falls weitere Zeilen benötigt werden)
     *
     * @return String-Array
     *
     * @throws IOException IO-Fehler bei unerwartetem Dateiende
     */
    private static String[] splitLineToCells(final String line, final BufferedReader reader, final Character delimiter) throws IOException {
        // Vgl. CsvReader.splitLineToCells!
        char[] chars = line.toCharArray();
        boolean inQuote = false;

        final ArrayList<String> cells = new ArrayList<>();

        final StringBuilder cell = new StringBuilder();

        while (true) {
            for (int i = 0; i < chars.length; i++) {
                final char c = chars[i];
                if (inQuote) {
                    if (c == '\"') {
                        // Doppelte Anführungszeichen innerhalb Anführungszeichen durch eins ersetzen
                        if ((i + 1 < chars.length) && (chars[i + 1] == '\"')) {
                            cell.append('\"');
                            ++i;
                        } else {
                            inQuote = false;
                        }
                    } else {
                        cell.append(c);
                    }
                } else {
                    if (c == '\"') {
                        inQuote = true;
                    } else if (c == delimiter) {
                        cells.add(cell.toString());
                        cell.setLength(0);
                    } else {
                        cell.append(c);
                    }
                }
            }
            if (inQuote) {
                // Mehrzeiliger Text, weitere Zeile einlesen
                String tmp = reader.readLine();
                if (tmp == null) {
                    throw new IOException("Unerwartetes Dateiende");
                }
                chars = tmp.toCharArray();
                cell.append('\n');
            } else {
                break;
            }
        }

        cells.add(cell.toString());

        return cells.toArray(new String[0]);
    }

    @Nullable
    @Override
    public String doInBackground() {
        return postProcessing();
    }

    @Override
    public void done() {
        try {
            String result = get();
            if (result != null) {
                JOptionPane
                    .showMessageDialog(_progressDialog.getComponent(), "Fehler bei der Nachbearbeitung:" + System.lineSeparator() + result, "Fehler",
                                       JOptionPane.ERROR_MESSAGE);
            }
        } catch (InterruptedException | ExecutionException e) {
            _debug.error("Fehler bei der Nachbearbeitung", e);
        }
        _progressDialog.disposeComponent();
    }

    @Override
    protected void process(List<Integer> linesRead) {
        Integer lastLinesRead = linesRead.get(linesRead.size() - 1);
        int value = 100 * lastLinesRead / _csvConverter.getNumberOfCsvRows();
        _progressDialog.setPostProcessingProgress(value);
    }

    @Nullable
    private String postProcessing() {
        if (_interruptRequested) {
            return null;
        }
        final String finalHeaderLine = _csvConverter.getCsvHeaderLine(true);
        final String delimiter = _csvConverter.getDelimiter();
        String[] columnHeaders = finalHeaderLine.split(delimiter);

        File newFile;
        try {
            newFile = createNewFile();
            if (newFile == null) {
                return "Es konte keine temporäre Datei in " + _csvFile.getParent() + " angelegt werden.";
            }
        } catch (IOException ignore) {
            return "Beim Anlegen einer temporären Datei in " + _csvFile.getParent() + " ist ein Problem aufgetreten.";
        }

        Charset cs;
        if (Charset.isSupported(_charsetName)) {
            cs = Charset.forName(_charsetName);
        } else {
            cs = Charset.defaultCharset();
        }

        try (OutputStreamWriter tempFileWriter = new OutputStreamWriter(new FileOutputStream(newFile), cs);
             InputStreamReader fileReader = new InputStreamReader(new FileInputStream(_csvFile), cs);
             BufferedReader bufferedReader = new BufferedReader(fileReader)) {
            tempFileWriter.write(finalHeaderLine);

            final String rowHeaderColums = "Art" + delimiter + "Zeit" + delimiter + "Objekt" + delimiter;
            // Diese Erkennung der neuen Kopfzeilen innerhalb des CSV-Bodies geht nur, weil in update() die
            // Methode getCsvHeaderLine mit 'true' aufgerufen wurde.

            int linesRead = 0;
            final String firstLine = bufferedReader.readLine();
            ++linesRead;

            if (firstLine == null || !firstLine.startsWith(rowHeaderColums)) {
                return "Falsche Programmlogik!"; // das kann gar nicht auftreten
            }
            String[] currentColumnHeaders = firstLine.split(delimiter);
            String line = bufferedReader.readLine();
            ++linesRead;
            for (; line != null; line = bufferedReader.readLine(), ++linesRead) {
                if (_interruptRequested) {
                    break;
                }
                if (line.startsWith(rowHeaderColums)) {
                    currentColumnHeaders = line.split(delimiter);
                } else {
                    String[] cells = splitLineToCells(line, bufferedReader, delimiter.charAt(0));
                    tempFileWriter.write(getModifiedLine(cells, currentColumnHeaders, columnHeaders, delimiter));
                }
                if (linesRead % 10 == 0) {
                    publish(linesRead);
                }
            }
        } catch (FileNotFoundException ignore) {
            //noinspection ResultOfMethodCallIgnored
            newFile.delete();
            return "Es ist eine FileNotFoundException für beim Bearbeiten der Daten aufgetreten.";
        } catch (IOException ignore) {
            //noinspection ResultOfMethodCallIgnored
            newFile.delete();
            return "Es ist eine IOException für beim Bearbeiten der Daten aufgetreten.";
        }

        if (!_csvFile.delete()) {
            return "Das Löschen der Datei " + _csvFile.getName() + " ist fehlgeschlagen." + System.lineSeparator() +
                   "Die nachbereiteten Daten stehen in der Datei " + newFile.getName() + ".";
        }
        if (!newFile.renameTo(_csvFile)) {
            return "Die nachbereiteten Daten stehen in der Datei " + newFile.getName() + "." + System.lineSeparator() +
                   "Die Datei konnte nicht umbenannt werden.";
        }
        return null;
    }

    @Nullable
    private File createNewFile() throws IOException {
        if (_csvFile == null) {
            return null;
        }
        String path = _csvFile.getAbsolutePath();
        int i = 0;
        while (true) {
            String newPath = path + Integer.toString(i);
            File file = new File(newPath);
            if (file.createNewFile()) {
                return file;
            }
            ++i;
            if (i > 99) {
                return null;
            }
        }
    }

    private String getModifiedLine(final String[] cells, final String[] currentColumnHeaders, final String[] extendedColumnHeaders,
                                   final String delimiter) {
        StringBuilder sb = new StringBuilder();
        int eIndex = 0;

        for (int cIndex = 0; cIndex < Math.min(cells.length, currentColumnHeaders.length); ++cIndex, ++eIndex) {
            if (cIndex > 0) {
                sb.append(delimiter);
            }
            while (!currentColumnHeaders[cIndex].equals(extendedColumnHeaders[eIndex])) {
                ++eIndex;
                sb.append(delimiter);
            }
            sb.append(cells[cIndex]);
        }
        // Für "keine Daten"-Zeilen oder ähnliche kann der currentColumnHeader natürlich viel mehr Elemente haben als cells:
	    sb.append(String.valueOf(delimiter).repeat(Math.max(0, currentColumnHeaders.length - cells.length)));
        // Es gibt einen legalen Ausnahmefall, wo die Länge von cells die von currentColumnHeaders unterscheiden überschreitet;
        // wenn nämlich die erste(n) Datenlieferung(en) nur "keine Daten"-Zeilen oder ähnliche enthielt, so gibt es keinen
        // Header für die vierte Spalte, die aber noch hinzugefügt werden muss:
        if (cells.length == 4 && currentColumnHeaders.length == 3) {
            sb.append(delimiter).append(cells[3]);
        }
        // Weitere Ausnahmen sind Debug wert:
        if (cells.length > currentColumnHeaders.length && (cells.length != 4 || currentColumnHeaders.length != 3)) {
            String msg = "PostProcessor.getModifiedLine(), unbehandelter Ausnahmefall: ";
            msg += "Anzahl Zellen: " + cells.length + ", Anzahl Headerspalten: " + currentColumnHeaders.length;
            msg += ", Cells: " + Arrays.toString(cells) + ", CurrentColumnHeader: " + Arrays.toString(currentColumnHeaders);
            _debug.error(msg);
        }
        sb.append(System.lineSeparator());
        return sb.toString();
    }

    @Override
    public String toString() {
        return "PostProcessor{" + "_csvFile=" + _csvFile + ", _csvConverter=" + _csvConverter + '}';
    }

    @Override
    public void interruptRequested() {
        _interruptRequested = true;
    }
}
