/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.dataview.csv;

import de.bsvrz.pat.sysbed.dataview.DataTableObject;
import de.bsvrz.pat.sysbed.dataview.DataTableObject.DataTableObjectId;
import de.bsvrz.pat.sysbed.dataview.filtering.FilterAttributeGroup;
import de.bsvrz.pat.sysbed.dataview.selectionManagement.CellKey;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

/**
 * Der SingleUseCsvConverter übernimmt innerhalb dieses Pakets die Erstellung der Header- und Datenzeilen wie sie etwa in CSV-Dateien verwendet
 * werden, das heißt, es handelt sich um Zeilen mit Datenfeldern, die durch ein Trennzeichen voneinander separiert sind. Die Ausgangsdaten sind hier
 * in  einer Collection von DataTableObjects enthalten, und die Spalten und der Inhalt der Datenzeilen kann durch eine Collection von CellKeys
 * gefiltert werden. Eine wesentliche Aufgabe bei der Erstellung der Zeilen ist die Abbildung der Arrays: deren Inhalte werden in aufeinanderfolgenden
 * Spalten wiedergegeben.
 * <p>
 * Jeder SingleUseCsvConverter kann nur für eine Attributgruppe verwendet werden, die schon im Konstruktor endgültig festgelegt wird.
 *
 * @author Kappich Systemberatung
 */
public class SingleUseCsvConverter extends AbstractCsvConverter {

    /**
     * Jeder SingleUseCsvConverter kann nur für eine Attributgruppe verwendet werden, die schon im Konstruktor endgültig festgelegt wird.
     */
    private final FilterAttributeGroup _filterAttributeGroup;

    /**
     * Die im Konstruktor übergebene Menge selektierter CellkKeys bestimmt Spalten und diejenigen Einträge, die nicht leer sein können.
     */
    private Set<CellKey> _selectedCellKeys;

    /**
     * Jeder SingleUseCsvConverter kann nur für eine Attributgruppe verwendet werden, die nicht veränderbar ist.
     *
     * @param filterAttributeGroup eine FilterAttributeGroup
     */
    public SingleUseCsvConverter(final FilterAttributeGroup filterAttributeGroup) {
        _filterAttributeGroup = filterAttributeGroup;
    }

    /**
     * Bevor der SingleUseCsvConverter seine Arbeit erledigen kann, muss stets diese Methode aufgerufen werden. Mit ihr teilt man dem
     * SingleUseCsvConverter die möglicherweise selektierten {@code CellKeys} und in jedem Fall alle relevanten {@code DataTableObjects} mit.
     *
     * @param dataTableObjects die DataTableObjects
     * @param selectedCellKeys die selektierten CellKeys
     */
    @Override
    public boolean setData(final List<DataTableObject> dataTableObjects, @Nullable final Set<CellKey> selectedCellKeys) {
        final Map<String, DataTableObject> dataTableObjectMap = new HashMap<>();
        synchronized (_dataTableObjects) {
            _csvColumnToText.clear();
            _dataTableObjects.clear();
            if (selectedCellKeys != null) {    // eine Selektion
                _selectedCellKeys = new HashSet<>(selectedCellKeys);
                final Set<String> filterSet = new HashSet<>();
                for (CellKey cellKey : _selectedCellKeys) {
                    filterSet.add(cellKey.getPidOfTheDataTableObject() + ":" + cellKey.getDataIndex());
                }
                for (DataTableObject dataTableObject : dataTableObjects) {
                    final String s = dataTableObject.getObject().getPid() + ":" + dataTableObject.getDataIndex();
                    if (filterSet.contains(s)) {
                        _dataTableObjects.add(dataTableObject);
                        dataTableObjectMap.put(s, dataTableObject);
                    }
                }
            } else {
                // keine Selektion, nehme alle CellKeys aller DataTableObjects; ACHTUNG: wenn dataTableObjects keine Datensätze
                // mit Daten enthält, entstehen hier keine CellKeys, die Spalteninformationen haben, was auf die Initialisierung
                // in initCsvColumns Auswirkungen hat.
                _selectedCellKeys = new HashSet<>();
                for (DataTableObject dataTableObject : dataTableObjects) {
                    _dataTableObjects.add(dataTableObject);
                    final String s = dataTableObject.getObject().getPid() + ":" + dataTableObject.getDataIndex();
                    dataTableObjectMap.put(s, dataTableObject);
                    _selectedCellKeys.addAll(dataTableObject.getAllCellKeys());
                }
            }
        }
        return initCsvColumns(dataTableObjectMap);
    }

    ////////////////////////////////////////////////////////////////////////
    // private Methoden
    ////////////////////////////////////////////////////////////////////////

    private boolean initCsvColumns(final Map<String, DataTableObject> dataTableObjectMap) {
        // Enthält _dataTableObjects keine Datensätze mit richtigen Daten, so enthält _selectedCellKeys
        // nur Super-Spalten, weshalb _csvColumnToText leer bleibt. Im Moment fällt das nur bei der
        // Ausgabe, die auf getCsvHeaderLine zurückgreift auf, und ist wohl erträglich.
        // Richtiger wäre es, die folgende Art der Initialisierung nur für einen SingleUseCsvConverter mit
        // Selektion durchzuführen, und für einen CSV-Manager ohne Selektion die CSVColumns aus
        // der Attributgruppe zu initialisieren.

        // Eigentlich ist dieser Konverter nur zum einmaligen Gebrauch ausgelegt, doch hier treiben wir ein wenig
        // Aufwand, um ihn doch - nämlich sicherheitshalber - für mehrfachen Gebrauch zu rüsten. D.h. _csvColumnToText
        // wird nicht einfach geleert, sondern wir machen eine Kopie, damit der Rückgabewert richtig bestimmt werden kann.
        // (Das bedeutet keinen wesentlichen Aufwand, falls der Konverter nur einmal benutzt wird.)
        Map<CsvColumn, Map<DataTableObject.DataTableObjectId, String>> csvColumnToText = new TreeMap<>(_csvColumnToText);
        _csvColumnToText.clear();
        boolean structureChanged = false;
        for (CellKey cellKey : _selectedCellKeys) {
            if (!cellKey.isSuperColumn()) {
                final CsvColumn column = new CsvColumn(_filterAttributeGroup, cellKey);
                final String key = cellKey.getPidOfTheDataTableObject() + ":" + cellKey.getDataIndex();
                final DataTableObject dataTableObject = dataTableObjectMap.get(key);
                final DataTableObjectId dataTableObjectId = dataTableObject.getDataTableObjectId();
                if (_csvColumnToText.containsKey(column)) {
                    _csvColumnToText.get(column).put(dataTableObjectId, cellKey.getCellText());
                } else {
                    if (csvColumnToText.containsKey(column)) {
                        structureChanged = true;
                    }
                    final Map<DataTableObjectId, String> newMap = new HashMap<>();
                    newMap.put(dataTableObjectId, cellKey.getCellText());
                    _csvColumnToText.put(column, newMap);
                }
            }
        }
        return structureChanged;
    }

    @Override
    public String toString() {
        return "SingleUseCsvConverter{}";
    }
}
