/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.dataview.filtering;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.Attribute;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.util.ArrayList;
import java.util.List;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;
import javax.swing.tree.DefaultTreeModel;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

/**
 * Diese Klasse stellt ein Filter für eine Attributgruppe dar. Sie implementiert {@code DefaultTreeModel} für die Darstellung in {@link
 * AtgFilterDefDialog}.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("NonSerializableFieldInSerializableClass")
public class AtgFilter extends DefaultTreeModel {

    private static final String ATTRIBUTE_GROUP = "AttributGroup";
    private static final String LEAVES = "leaves";
    private static final String SUPPRESSED = "suppressed";
    private static final String PSEUDONYM = "pseudonym";
    private final AttributeGroup _attributeGroup;
    private String _name;
    private boolean _storeInPreferences = true;

    /**
     * Dieser Konstruktor ist ausschließlich dazu geeignet, einen Filter zu definieren, der nichts herausfiltert (weil er keine {@link AttributeGroup}
     * hat). Einen solchen Filter verwendet der {@link AtgFilterManager}.
     *
     * @param name der Name des Filters
     */
    public AtgFilter(@NotNull String name) {
        super(null);

        _name = name;
        _attributeGroup = null;
    }

    /**
     * Dieser Konstruktor dient zur Definition echter Filter.
     *
     * @param name der Name des Filters
     * @param atg  die Attributgruppe
     */
    public AtgFilter(@NotNull String name, @NotNull final AttributeGroup atg) {
        super(new AtgFilterNode(atg));

        _name = name;
        _attributeGroup = atg;
        ((AtgFilterNode) getRoot()).createChildren();
    }

    private static List<AtgFilterNode> getLeavesInDFS(TreeModel model) {
        List<AtgFilterNode> leaves = new ArrayList<>();
        addLeavesInDFS(model, model.getRoot(), leaves);
        return leaves;
    }

    private static void addLeavesInDFS(TreeModel model, @Nullable Object node, List<AtgFilterNode> leaves) {
        if (node == null) {
            return;
        }
        if (model.isLeaf(node)) {
            if (node instanceof AtgFilterNode) {
                leaves.add((AtgFilterNode) node);
            }
        } else {
            for (int index = 0; index < model.getChildCount(node); ++index) {
                Object child = model.getChild(node, index);
                if (child != null) {
                    addLeavesInDFS(model, child, leaves);
                }
            }
        }
    }

    /*
     * Für zwei AtgFilter zur selben Attributgruppe werden die Eigenschaften der AtgFilterNodes kopiert.
     */
    private static void copyProperties(final AtgFilter source, final AtgFilter target) {
        Object sourceRoot = source.getRoot();
        Object targetRoot = target.getRoot();
        if (sourceRoot instanceof AtgFilterNode && targetRoot instanceof AtgFilterNode) {
            AtgFilterNode.copyProperties((AtgFilterNode) sourceRoot, (AtgFilterNode) targetRoot);
            copyPropertiesOfNextLevel((AtgFilterNode) sourceRoot, (AtgFilterNode) targetRoot);
        }
    }

    private static void copyPropertiesOfNextLevel(final AtgFilterNode sourceNode, final AtgFilterNode targetNode) {
        if (sourceNode.getChildCount() == targetNode.getChildCount()) {
            for (int i = 0; i < sourceNode.getChildCount(); ++i) {
                Object sourceChild = sourceNode.getChildAt(i);
                Object targetChild = targetNode.getChildAt(i);
                if (sourceChild instanceof AtgFilterNode && targetChild instanceof AtgFilterNode) {
                    AtgFilterNode.copyProperties((AtgFilterNode) sourceChild, (AtgFilterNode) targetChild);
                    copyPropertiesOfNextLevel((AtgFilterNode) sourceChild, (AtgFilterNode) targetChild);
                }
            }
        }
    }

    /**
     * Initialisiert einen Filter aus en Präferenzen.
     *
     * @param connection ein ClientDavInterface
     * @param prefs      die Präferenzen
     *
     * @return der initialisierte Filter oder {@code null}
     */
    @Nullable
    static AtgFilter initFromPreferences(final ClientDavInterface connection, final Preferences prefs) {
        String atgPid = prefs.get(ATTRIBUTE_GROUP, "");
        if (atgPid.isEmpty()) {
            return null;
        }
        AttributeGroup atg = connection.getDataModel().getAttributeGroup(atgPid);
        if (atg == null) {
            return null;
        }
        AtgFilter atgFilter = new AtgFilter(prefs.name(), atg);

        Preferences leafNode = prefs.node(LEAVES);
        String[] children;
        try {
            children = leafNode.childrenNames();
        } catch (BackingStoreException ignore) {
            return null;
        }

        List<AtgFilterNode> leaves = getLeavesInDFS(atgFilter);

        if (leaves.size() != children.length) {
            return null;
        }

        for (int i = 0; i < leaves.size(); ++i) {
            Preferences leafPrefs = leafNode.node(Integer.toString(i));
            leaves.get(i).setSuppressed(leafPrefs.getBoolean(SUPPRESSED, false));
            leaves.get(i).setPseudonym(leafPrefs.get(PSEUDONYM, ""));
        }

        return atgFilter;
    }

    /**
     * Erstellt eine tiefe Kopie des Filters.
     *
     * @return die Kopie
     */
    public AtgFilter getCopy() {
        if (_attributeGroup == null) {
            return new AtgFilter(_name);
        } else {
            AtgFilter newFilter = new AtgFilter(_name, _attributeGroup);
            // kopieren der Eigenschaften der AtgFilterNodes
            if (root instanceof AtgFilterNode) {
                AtgFilterNode newRoot = ((AtgFilterNode) root).getCopy();
                newFilter.setRoot(newRoot);
                newRoot.createChildren();
                copyProperties(this, newFilter);
            }
            return newFilter;
        }
    }

    /**
     * Gibt den Namen des Filters zurück.
     *
     * @return den Namen des Filters
     */
    public String getName() {
        return _name;
    }

    /**
     * Setzt den Namen auf den übergebenen String.
     *
     * @param name der neue Name
     */
    public void setName(final String name) {
        _name = name;
    }

    /**
     * Gibt die {@link AttributeGroup} zurück.
     *
     * @return die Attributgruppe
     */
    public AttributeGroup getAttributeGroup() {
        return _attributeGroup;
    }

    /**
     * Gibt {@code true} zurück, falls der Filter in den Präferenzen des Benutzers gespeichert werden soll, und {@code false}, falls nicht.
     *
     * @return s.o.
     */
    boolean getStoreInPreferences() {
        return _storeInPreferences;
    }

    /**
     * Legt fest, ob der Filter in den Präferenzen des Benutzers gespeichert werden soll oder nicht.
     *
     * @param store der entsprechende Wert
     */
    void setStoreInPreferences(@SuppressWarnings("SameParameterValue") boolean store) {
        _storeInPreferences = store;
    }

    /**
     * Gibt an, ob es sich um einen echten Filter, also eine mit {@link AttributeGroup} handelt.
     *
     * @return s.o.
     */
    public boolean isValid() {
        return _attributeGroup != null;
    }

    @Override
    public String toString() {
        return _name;
    }

    @Nullable
    @Override
    public Object getChild(final Object parent, final int index) {
	    if (parent instanceof AtgFilterNode atgFilterNode) {
            return atgFilterNode.getChildAt(index);
        } else {
            return null;
        }
    }

    @Override
    public int getChildCount(final Object parent) {
	    if (parent instanceof AtgFilterNode atgFilterNode) {
            return atgFilterNode.getChildCount();
        }
        return 0;
    }

    @Override
    public boolean isLeaf(final Object node) {
	    if (node instanceof AtgFilterNode atgFilterNode) {
            return atgFilterNode.isLeaf();
        }
        return false;
    }

    @Override
    public void valueForPathChanged(final TreePath path, final Object newValue) {
        AtgFilterNode node = (AtgFilterNode) path.getLastPathComponent();
        if (newValue instanceof String) {
            node.setPseudonym((String) newValue);
        } else if (newValue instanceof Boolean) {
            node.setSuppressed((Boolean) newValue);
        }
        nodeChanged(node);
    }

    @Override
    public int getIndexOfChild(final Object parent, final Object child) {
	    if (parent instanceof AtgFilterNode atgFilterNode) {
            return atgFilterNode.getIndex((TreeNode) child);
        }
        return 0;
    }

    /**
     * Ein AtgFilter ist leer, wenn alles herausgefiltet wird.
     *
     * @return
     */
    public boolean isEmpty() {
        //noinspection SimplifiableIfStatement
        if (root != null && root instanceof AtgFilterNode) {
            return ((AtgFilterNode) root).isEmpty();
        } else {
            return true;
        }
    }

    /**
     * Gibt eine Liste von {@link AtgFilterNode AtgFilterNodes} zurück, die für die {@link Attribute} stehen, die den Filter passieren.
     *
     * @return s.o.
     */
    public List<AtgFilterNode> getFilterAttributes() {
        List<AtgFilterNode> attributes = new ArrayList<>();
        for (int i = 0; i < getChildCount(root); ++i) {
            addFilterAttributes(getChild(root, i), attributes);
        }
        return attributes;
    }

    private void addFilterAttributes(@Nullable Object node, List<AtgFilterNode> attributes) {
        if (node == null) {
            return;
        }
	    if (node instanceof AtgFilterNode atgFilterNode) {
            if (!atgFilterNode.isSuppressed()) { // ist automatisch ein Leaf
                Object userObject = atgFilterNode.getUserObject();
                if (userObject instanceof Attribute) {
                    attributes.add(atgFilterNode);
                }
            }
            for (int i = 0; i < getChildCount(node); ++i) {
                addFilterAttributes(getChild(node, i), attributes);
            }
        }
    }

    @Override
    public boolean equals(Object o) {
        return o instanceof AtgFilter && getName().equals(((AtgFilter) o).getName());
    }

    @Override
    public int hashCode() {
        return getName().hashCode();
    }

    /**
     * Speichert den Filter in den Präferenzen.
     *
     * @param prefs die Präferenzen
     *
     * @return {@code true}
     */
    boolean putPreferences(final Preferences prefs) {
        Preferences myPreferences = prefs.node(getName());
        myPreferences.put(ATTRIBUTE_GROUP, _attributeGroup.getPid());
        Preferences leafPreferences = myPreferences.node(LEAVES);

        List<AtgFilterNode> leaves = getLeavesInDFS(this);
        for (int i = 0; i < leaves.size(); ++i) {
            Preferences leafPrefs = leafPreferences.node(Integer.toString(i));
            leafPrefs.putBoolean(SUPPRESSED, leaves.get(i).isSuppressed());
            leafPrefs.put(PSEUDONYM, leaves.get(i).getPseudonym());
        }
        return true;
    }

    /**
     * Löscht den Filter mit diesem Namen aus den Präferenzen.
     *
     * @param prefs die Präferenzen
     *
     * @return {@code true} falls das Löschen erfolgreich war, {@code false} sonst
     */
    boolean removePreferences(final Preferences prefs) {
        Preferences myNode = prefs.node(getName());
        try {
            myNode.removeNode();
            return true;
        } catch (BackingStoreException ignore) {
            return false;
        }
    }

}
