/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.dataview.filtering;

import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Enumeration;
import java.util.EventObject;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.JTextField;
import javax.swing.JTree;
import javax.swing.SwingUtilities;
import javax.swing.tree.DefaultTreeCellEditor;
import javax.swing.tree.DefaultTreeCellRenderer;
import javax.swing.tree.TreeCellRenderer;
import javax.swing.tree.TreeNode;
import javax.swing.tree.TreePath;

/**
 * Dieser Dialog erlaubt die Definition eines Attributgruppen-Filters (s {@link AtgFilter}.
 *
 * @author Kappich Systemberatung
 */
public class AtgFilterDefDialog extends JDialog {

    private final Component _parent;
    private final JTextField _nameTextField = new JTextField(20);
    private CheckBoxTree _checkBoxTree;

    /**
     * Ein Konstruktor, um einen neuen Filter anzulegen.
     *
     * @param atg    die Attributgruppe
     * @param parent die Parent-Komponente oder {@code null}
     */
    public AtgFilterDefDialog(@NotNull final AttributeGroup atg, @Nullable final Component parent) {
        super();
        AtgFilter filter = new AtgFilter("Neuer Name", atg);
        _parent = parent;
        init(filter, true);
        _nameTextField.setText(filter.getName());
        _nameTextField.setEditable(true);
    }

    /**
     * Ein Konstruktor, um einen Filter zu bearbeiten.
     *
     * @param filter die Attributgruppe
     * @param parent die Parent-Komponente oder {@code null}
     */
    public AtgFilterDefDialog(final AtgFilter filter, @Nullable final Component parent) {
        super();
        _parent = parent;
        init(filter, false);
        _nameTextField.setText(filter.getName());
        _nameTextField.setEditable(false);
    }

    @SuppressWarnings("SameParameterValue")
    private static void expandToLevel(JTree tree, int level) {
        for (int l = 0; l < level; ++l) {
            for (int i = tree.getRowCount() - 1; i >= 0; --i) {
                tree.expandRow(i);
            }
        }
    }

    @SuppressWarnings("OverlyLongMethod")
    private void init(final AtgFilter filter, boolean isNew) {
        if (isNew) {
            setTitle("Einen neuen Filter anlegen");
        } else {
            setTitle("Einen Filter bearbeiten");
        }
        JPanel namePanel = new JPanel();
        namePanel.setLayout(new BoxLayout(namePanel, BoxLayout.X_AXIS));
        namePanel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
        JLabel nameLabel = new JLabel("Name: ");
        namePanel.add(nameLabel);
        namePanel.add(Box.createHorizontalStrut(5));
        namePanel.add(_nameTextField);
        namePanel.setMaximumSize(new Dimension(1200, 20));

        JPanel treePanel = new JPanel();
        treePanel.setLayout(new BoxLayout(treePanel, BoxLayout.X_AXIS));
        JTree tree = new JTree(filter);
        tree.setInvokesStopCellEditing(true);
        AtgFilterRenderer renderer = new AtgFilterRenderer();
        tree.setCellRenderer(renderer);
        tree.setEditable(true);
        tree.setCellEditor(new AtgFilterEditor(tree, renderer));
        expandToLevel(tree, 1);
        treePanel.add(new JScrollPane(tree));
        tree.addMouseListener(new MyMouseAdapter());
        tree.setRowHeight(20);

        _checkBoxTree = new CheckBoxTree(tree, (tree1, node) -> node instanceof AtgFilterNode && !((AtgFilterNode) node).isSuppressed());
        _checkBoxTree.setLeafsOnly(true);
        _checkBoxTree.suppressIcons();
        _checkBoxTree.addCheckBoxListener(node -> {
	        if (node instanceof AtgFilterNode atgFilterNode) {
                atgFilterNode.setSuppressed(!atgFilterNode.isSuppressed());
            }
        });

        JPanel buttonPanel = new JPanel();
        buttonPanel.setLayout(new BoxLayout(buttonPanel, BoxLayout.X_AXIS));
        buttonPanel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
        JButton deleteButton = new JButton("Löschen");
        deleteButton.setEnabled(!isNew);
        deleteButton.addActionListener(e -> {
            int n = JOptionPane.showConfirmDialog(this, "Diesen Filter unwiderruflich löschen?", "Löschen bestätigen", JOptionPane.OK_CANCEL_OPTION);
            if (n == JOptionPane.OK_OPTION) {
                boolean success = AtgFilterManager.getInstance().removeFilter(filter);
                if (!success) {
                    JOptionPane.showMessageDialog(this, "Dieser Filter kann nicht gelöscht werden!", "Fehlermeldung", JOptionPane.ERROR_MESSAGE);
                } else {
                    dispose();
                }
            }
        });
        JButton saveButton = new JButton("Speichern");
        saveButton.addActionListener(e -> {
            if (filter.isEmpty()) {
                JOptionPane
                    .showMessageDialog(this, "Dieser Filter blendet alles aus. Ein solcher Filter kann nicht gespeichert werden.", "Fehlermeldung",
                                       JOptionPane.ERROR_MESSAGE);
                return;
            }

            filter.setName(_nameTextField.getText());
            boolean success = AtgFilterManager.getInstance().addFilter(filter, isNew);
            if (success) {
                dispose();
            } else {
                JOptionPane.showMessageDialog(this, "Ein Filter mit diesem Namen existiert bereits!", "Fehlermeldung", JOptionPane.ERROR_MESSAGE);
            }
        });
        buttonPanel.add(deleteButton);
        buttonPanel.add(Box.createHorizontalStrut(5));
        buttonPanel.add(saveButton);
        buttonPanel.setMaximumSize(new Dimension(1200, 30));

        JPanel allPanel = new JPanel();
        allPanel.setLayout(new BorderLayout());
        allPanel.add(namePanel, BorderLayout.PAGE_START);
        allPanel.add(treePanel, BorderLayout.CENTER);
        allPanel.add(buttonPanel, BorderLayout.PAGE_END);

        add(allPanel);

        Dimension d = tree.getPreferredSize();
        setSize((int) Math.min(d.getWidth() + 200, 800), (int) Math.min(d.getHeight() + 200, 1200));
        if (null != _parent) {
            setLocationRelativeTo(_parent);
        }
    }

    @Override
    public String toString() {
        return "AtgFilterDefDialog{}";
    }

    /* Renderer */
    private static class AtgFilterRenderer extends DefaultTreeCellRenderer {

        @Nullable
        @Override
        public Component getTreeCellRendererComponent(final JTree tree, final Object value, final boolean isSelected, final boolean expanded,
                                                      final boolean leaf, final int row, final boolean hasTheFocus) {

	        if (value instanceof AtgFilterNode node) {
                return super.getTreeCellRendererComponent(tree, node.getNodeText(), isSelected, expanded, leaf, row, hasTheFocus);
            } else {
                return super.getTreeCellRendererComponent(tree, value, isSelected, expanded, leaf, row, hasTheFocus);
            }
        }
    }

    /* Editor */
    private static class AtgFilterEditor extends DefaultTreeCellEditor {

        TreeCellRenderer _renderer;

        public AtgFilterEditor(JTree theTree, DefaultTreeCellRenderer theRenderer) {
            super(theTree, theRenderer);
            _renderer = theRenderer;

        }

        @Override
        public Component getTreeCellEditorComponent(JTree theTree, Object value, boolean isSelected, boolean expanded, boolean leaf, int row) {
            if (value instanceof AtgFilterNode) {
                value = ((AtgFilterNode) value).getPseudonym();
            }
            return super.getTreeCellEditorComponent(tree, value, isSelected, expanded, leaf, row);
        }

        @Override
        public boolean isCellEditable(EventObject e) {
            return super.isCellEditable(e) && ((AtgFilterNode) lastPath.getLastPathComponent()).isLeaf();
        }
    }

    class MyMouseAdapter extends MouseAdapter {

        @Override
        public void mouseClicked(MouseEvent e) {
            if (SwingUtilities.isRightMouseButton(e)) {
                Object source = e.getSource();
	            if (!(source instanceof JTree tree)) {
                    return;
                }
                TreePath path = tree.getPathForLocation(e.getX(), e.getY());
                if (path == null) {
                    return;
                }
                Object node = path.getLastPathComponent();
	            if (!(node instanceof AtgFilterNode atgFilterNode)) {
                    return;
                }
                if (atgFilterNode.isLeaf()) {
                    return;
                }

                JPopupMenu popupMenu = new JPopupMenu();
                {
                    JMenuItem expandMenuItem = new JMenuItem("Vollständig aufklappen");
                    expandMenuItem.addActionListener(event -> expandAll(tree, path));
                    popupMenu.add(expandMenuItem);
                }
                {
                    JMenuItem selectAllLeavesMenuItem = new JMenuItem("Alle Blätter selektieren");
                    selectAllLeavesMenuItem.addActionListener(event -> {
                        selectAllLeaves(tree, path, false);
                        _checkBoxTree.reinitializeCheckBoxes();
                        expandAll(tree, path);
                    });
                    popupMenu.add(selectAllLeavesMenuItem);
                }
                {
                    JMenuItem deselectAllLeavesMenuItem = new JMenuItem("Alle Blätter deselektieren");
                    deselectAllLeavesMenuItem.addActionListener(event -> {
                        selectAllLeaves(tree, path, true);
                        _checkBoxTree.reinitializeCheckBoxes();
                        expandAll(tree, path);
                    });
                    popupMenu.add(deselectAllLeavesMenuItem);
                }
                popupMenu.show(e.getComponent(), e.getX(), e.getY());
            }
        }

        private void expandAll(JTree tree, TreePath parent) {
            TreeNode node = (TreeNode) parent.getLastPathComponent();
            if (node.getChildCount() >= 0) {
                for (@SuppressWarnings("rawtypes") Enumeration e = node.children(); e.hasMoreElements(); ) {
                    TreeNode n = (TreeNode) e.nextElement();
                    TreePath path = parent.pathByAddingChild(n);
                    expandAll(tree, path);
                }
            }
            tree.expandPath(parent);
        }

        private void selectAllLeaves(JTree tree, TreePath parent, final boolean suppress) {
            TreeNode node = (TreeNode) parent.getLastPathComponent();
            if (node instanceof AtgFilterNode && tree.getModel().isLeaf(node)) {
                ((AtgFilterNode) node).setSuppressed(suppress);
            }
            if (node.getChildCount() >= 0) {
                for (@SuppressWarnings("rawtypes") Enumeration e = node.children(); e.hasMoreElements(); ) {
                    TreeNode n = (TreeNode) e.nextElement();
                    TreePath path = parent.pathByAddingChild(n);
                    selectAllLeaves(tree, path, suppress);
                }
            }
        }
    }
}
