/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.dataview.filtering;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.util.ArrayList;
import java.util.List;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;

/**
 * Diese Klasse verwaltet alle Attributgruppen-Filter (s. {@link AtgFilter}. Es handelt sich um ein Singleton.
 *
 * @author Kappich Systemberatung
 */
@SuppressWarnings("NonSerializableFieldInSerializableClass")
public final class AtgFilterManager {

    private static final String NO_FILTER = "<kein Filter>";
    private static final Preferences PREFERENCES = Preferences.userRoot().node("/gtm").node("filters");

    private static AtgFilterManager _instance;
    private final ClientDavInterface _connection;

    @SuppressWarnings("NonSerializableFieldInSerializableClass")
    private final List<AtgFilter> _atgFilters = new ArrayList<>();
    private final List<AtgFilterListener> _listeners = new ArrayList<>();

    private AtgFilterManager(final ClientDavInterface connection) {
        _connection = connection;
        AtgFilter noFilter = new AtgFilter(NO_FILTER);
        noFilter.setStoreInPreferences(false);
        addFilter(noFilter, true);
        readPreferences();
    }

    /**
     * Diese {@code getInstance}-Methode muss unbedingt als erste aufgerufen werden, damit mit Hilfe des übergebenen {@link ClientDavInterface
     * ClientDavInterfaces} die existierenden Filter initialisiert werden.
     *
     * @param connection ein ClientDavInterface
     *
     * @return
     */
    public static AtgFilterManager getInstance(final ClientDavInterface connection) {
        if (_instance == null) {
            _instance = new AtgFilterManager(connection);
        }
        return _instance;
    }

    /**
     * Gibt die einzige Instanz zurück ohne eine Initialisierung vornehmen zu können.
     *
     * @return die einzige Instanz
     */
    public static AtgFilterManager getInstance() {
        return _instance;
    }

    private boolean readPreferences() {
        String[] children;
        try {
            children = PREFERENCES.childrenNames();
        } catch (BackingStoreException ignore) {
            return false;
        }
        for (final String aChildren : children) {
            Preferences childPrefs = PREFERENCES.node(aChildren);
            AtgFilter atgFilter = AtgFilter.initFromPreferences(_connection, childPrefs);
            if (atgFilter != null) {
                _atgFilters.add(atgFilter);
            }
        }
        return true;
    }

    /**
     * Versucht den übergebenen Filter hinzuzufügen. Ist dieser Filter new ({@code isNew} ist {@code true}), so findet eine Überprüfung statt, ob ein
     * Filter desselben Namens bereits existiert. Falls ja, so wird {@code false} zurückgegeben. Falls nein, so wird der Filter hinzugefügt. Ist der
     * Filter nicht neu ({@code isNew} ist {@code false}), so wird ein eventuell vorhandener Filter gleichen Namens gelöscht, und dann wird der
     * übergebene Filter hinzugefügt. Angemeldete {@link AtgFilterListener} werden entsprechend informiert.
     *
     * @param filter
     * @param isNew
     *
     * @return
     */
    public boolean addFilter(final AtgFilter filter, boolean isNew) {
        boolean removed = false;
        if (isNew) {
            for (AtgFilter atgFilter : _atgFilters) {
                if (atgFilter.getName().equals(filter.getName())) {
                    return false;
                }
            }
        } else {
            removed = removeFilter(filter, false);
        }
        if (filter.getStoreInPreferences()) {
            if (!filter.putPreferences(PREFERENCES)) {
                return false;
            }
        }
        _atgFilters.add(filter);
        if (removed) {
            notifyFilterChanged(filter);
        } else {
            notifyFilterAdded(filter);
        }
        return true;
    }

    private boolean removeFilter(final AtgFilter filter, boolean withNotify) {
        if (filter.getName().equals(NO_FILTER)) {
            return false;
        }
        for (AtgFilter atgFilter : _atgFilters) {
            if (filter.getName().equals(atgFilter.getName())) {
                _atgFilters.remove(atgFilter);
                atgFilter.removePreferences(PREFERENCES);
                if (withNotify) {
                    notifyFilterRemoved(atgFilter);
                }
                return true;
            }
        }
        return false;
    }

    /**
     * Entfernt den übergebenen Filter aus der Filter-Verwaltung. Informiert alle angemeldeten {@link AtgFilterListener}.
     *
     * @param filter ein Filter
     *
     * @return {@code true}, falls der Filter erfolgreich gelöscht wurde
     */
    public boolean removeFilter(final AtgFilter filter) {
        return removeFilter(filter, true);
    }

    private void notifyFilterAdded(AtgFilter newFilter) {
        for (AtgFilterListener listener : _listeners) {
            listener.filterAdded(newFilter);
        }
    }

    private void notifyFilterChanged(AtgFilter newFilter) {
        for (AtgFilterListener listener : _listeners) {
            listener.filterRemoved(newFilter);
        }
    }

    private void notifyFilterRemoved(AtgFilter oldFilter) {
        for (AtgFilterListener listener : _listeners) {
            listener.filterRemoved(oldFilter);
        }
    }

    /**
     * Gibt ein Array von Filtern zurück. Ist die übergebene {@link AttributeGroup} {@code null}, so enthält das Array alle Filter; andernfalls nur
     * die Filter zu dieser Attributgruppe.
     *
     * @param atg eine Attributgruppe oder {@code null}
     *
     * @return s.o.
     */
    public AtgFilter[] getAllFilters(@Nullable final AttributeGroup atg) {
        if (null == atg) {
            AtgFilter[] array = new AtgFilter[_atgFilters.size()];
            array = _atgFilters.toArray(array);
            return array;
        } else {
            List<AtgFilter> filters = new ArrayList<>();
            for (AtgFilter filter : _atgFilters) {
                if (filter.getAttributeGroup() == null || filter.getAttributeGroup().equals(atg)) {
                    filters.add(filter);
                }
            }
            AtgFilter[] array = new AtgFilter[filters.size()];
            array = filters.toArray(array);
            return array;
        }
    }

    /**
     * Gibt den Filter zu dem Namen zurück oder {@code null}, falls kein solcher Filter existiert.
     *
     * @param filterName ein Name
     *
     * @return ein Filter oder {@code null}
     */
    @Nullable
    public AtgFilter getFilter(@Nullable final String filterName) {
        if (filterName == null || filterName.isEmpty() || filterName.equals(NO_FILTER)) {
            return null;
        }
        for (AtgFilter filter : _atgFilters) {
            if (filter.getName().equals(filterName)) {
                return filter;
            }
        }
        return null;
    }

    /**
     * Meldet den übergebenen Listener an.
     *
     * @param listener ein AtgFilterListener
     */
    public void addListener(AtgFilterListener listener) {
        for (AtgFilterListener l : _listeners) {
            if (l.equals(listener)) {
                return;
            }
        }
        _listeners.add(listener);
    }

    /**
     * Meldet den übergebenen Listener ab.
     *
     * @param listener ein AtgFilterListener
     *
     * @return {@code true} falls er erfolgreich abgemeldet wurde und {@code false}, falls dieser gar nicht angemeldet war
     */
    public boolean removeListener(AtgFilterListener listener) {
        return _listeners.remove(listener);
    }

    /**
     * Ein Interface, um über Veränderungen in der Filter-Verwaltung informiert zu werden.
     */
    public interface AtgFilterListener {
        /**
         * Diese Methode wird aufgerufen, wenn der Filter hinzugefügt wird.
         *
         * @param newFilter
         */
        void filterAdded(AtgFilter newFilter);

        /**
         * Diese Methode wird aufgerufen, wenn der Filter geändert wird. Die Änderung eines Filters ist nicht auf die Änderung eines Objekts
         * beschränkt, sondern besagt nur, dass ein Filter durch einen Filter gleichen Namens ersetzt wurde. Der übergebene Filter ist die neue
         * Version.
         *
         * @param newFilter
         */
        void filterChanged(AtgFilter newFilter);

        /**
         * Diese Methode wird aufgrufen, wenn der Filter gelöscht wurde.
         *
         * @param oldFilter
         */
        void filterRemoved(AtgFilter oldFilter);
    }
}
