/*
 * Copyright 2005 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2006-2020 by Kappich Systemberatung, Aachen
 * Copyright 2022 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.pat.sysbed.main;

import de.bsvrz.dav.daf.accessControl.AccessControlManager;
import de.bsvrz.dav.daf.accessControl.UserInfo;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.*;
import de.bsvrz.pat.sysbed.help.GtmHelp;
import de.bsvrz.pat.sysbed.permissions.PermissionsState;
import de.bsvrz.pat.sysbed.plugins.api.ExternalModule;
import de.bsvrz.pat.sysbed.plugins.api.settings.SettingsData;
import de.bsvrz.pat.sysbed.preselection.lists.PreselectionLists;
import de.bsvrz.pat.sysbed.preselection.lists.PreselectionListsListener;
import de.bsvrz.pat.sysbed.preselection.lists.SystemObjectListRenderer;
import de.bsvrz.pat.sysbed.preselection.panel.PreselectionPanel;
import de.bsvrz.pat.sysbed.preselection.tree.TreeNodeObject;
import de.bsvrz.sys.funclib.application.StandardApplicationRunner;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.kappich.onlinehelp.HelpMenu;

import javax.swing.*;
import javax.swing.event.MenuEvent;
import javax.swing.event.MenuListener;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.image.BufferedImage;
import java.util.List;
import java.util.*;
import java.util.prefs.BackingStoreException;

/**
 * Diese Klasse erstellt das Fenster der Anwendung mit der {@link de.bsvrz.pat.sysbed.preselection.panel.PreselectionPanel Datenauswahl}, beliebigen
 * {@link de.bsvrz.pat.sysbed.plugins.api.ExternalModule Modulen} und dem Panel, welches die Einstellungen der Module verwaltet.
 * <p>
 * Damit die Module auch mit der Applikation kommunizieren können, implementiert diese Klasse das {@link ApplicationInterface}.
 *
 * @author Kappich Systemberatung
 */
public class GenericTestMonitorApplication implements ApplicationInterface {

    // Farben für Berechtigungs-Icons
    private static final Paint FILL_ALLOWED = new Color(0, 215, 3);
    private static final Paint STROKE_ALLOWED = new Color(0, 192, 3);
    private static final Paint FILL_ALLOWED_WRONG = new Color(159, 215, 0);
    private static final Paint STROKE_ALLOWED_WRONG = new Color(141, 192, 0);
    private static final Paint FILL_FORBIDDEN = new Color(215, 3, 0);
    private static final Paint STROKE_FORBIDDEN = new Color(191, 0, 0);

    /** Das Hauptfenster der Anwendung. */
    private final JFrame _frame;

    /** Die Verbindung zum Datenverteiler. */
    private final ClientDavInterface _connection;

    /** das Panel, worauf die Module angeordnet sind */
    private final JPanel _modulePanel;

    /** speichert die Bildschirmgröße */
    private final Dimension _screenSize;

    /** speichert den PreselectionPanel */
    private final PreselectionPanel _preselectionPanel;
    /** das Panel, worauf das Logo angeordnet ist */
    private final JPanel _logoPanel;
    /** speichert den Splitpane, welches das Fenster in der Horizontalen teilt */
    private final JSplitPane _splitPane;
    /** speichert das Panel mit den Einstellungen */
    private final SettingsHandler _settingsHandler;
    /** speichert das Panel mit der Datenidentifikationsauswahl und den Modulen */
    private final JPanel _dataSelectionPanel;
    /** Speichert die Objekte der Module. Anhand des Keys (Klassenname des Moduls) kann das Objekt ermittelt werden. */
    private final Map<String, ExternalModule> _moduleMap;
    /** speichert die ArgumentListe, die beim Aufruf der Applikation übergeben wurde */
    private List<String> _argumentList;
    /** Benutzer, dessen Berechtigungen angezeigt werden sollen */
    private SystemObject _selectedPermissionsUser;

    /** Ob Berechtigungen angezeigt werden sollen */
    private boolean _viewPermissions;

    private Map<PermissionsState, Icon> _iconCache = new HashMap<>();
    private Stroke stroke = new BasicStroke(3.0f);

    /**
     * Konstruktor. Die Applikation erhält eine {@link de.bsvrz.dav.daf.main.ClientDavInterface Verbindung zum Datenverteiler} und den für den {@link
     * de.bsvrz.pat.sysbed.preselection.tree.PreselectionTree Filterbaum} benötigten Parameter zur Spezifizierung der Vorauswahl, bestehend aus
     * System- und {@link de.bsvrz.pat.sysbed.preselection.tree.TreeNodeObject Knotenobjekten}.
     *
     * @param title      der Titel des Fensters
     * @param connection Verbindung zum Datenverteiler
     * @param treeNodes  bestehend aus System- und {@link TreeNodeObject Knotenobjekten}
     */
    public GenericTestMonitorApplication(String title, final ClientDavInterface connection, final Collection<Object> treeNodes) {
        _moduleMap = new HashMap<>();

        _connection = connection;
        _selectedPermissionsUser = _connection.getLocalUser();
        _screenSize = Toolkit.getDefaultToolkit().getScreenSize();

        _frame = new JFrame();
        _frame.setTitle(getTitle(title, connection));
        _frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        _frame.addWindowListener(new WindowAdapter() {
            @Override
            public void windowClosing(WindowEvent e) {
                _connection.disconnect(false, "Applikation wurde beendet.");
            }
        });

        Container pane = _frame.getContentPane();
        pane.setLayout(new BorderLayout());

        // hier werden PreselectionPanel, Logo und Modul-Buttons zusammengestellt
        _preselectionPanel = new PreselectionPanel(_connection, treeNodes);
        _preselectionPanel.showSimulationVariant();
        try {
            _preselectionPanel.getPreselectionTree().setSelectedTreePath("alles");
        } catch (Exception ignored) {
        }

        JPanel buttonAndLogoPanel = new JPanel(new BorderLayout());
        _logoPanel = new JPanel(new BorderLayout());
        buttonAndLogoPanel.add(_logoPanel, BorderLayout.NORTH);

        _modulePanel = new JPanel();
        _modulePanel.setLayout(new BoxLayout(_modulePanel, BoxLayout.Y_AXIS));
        _modulePanel.setBorder(BorderFactory.createEmptyBorder(0, 5, 0, 5));

        buttonAndLogoPanel.add(_modulePanel, BorderLayout.CENTER);

        _dataSelectionPanel = new JPanel(new BorderLayout());
        _dataSelectionPanel.add(_preselectionPanel, BorderLayout.CENTER);
        _dataSelectionPanel.add(buttonAndLogoPanel, BorderLayout.EAST);

        // dataSelectionPanel mit settingsPanel in ein SplitPane
        _settingsHandler = new SettingsHandler(this, _preselectionPanel);
        _settingsHandler.setMaximumNumberOfLastUsedSettings(20);
        _splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT, true, _dataSelectionPanel, _settingsHandler.getSettingsPanel());
        _splitPane.setOneTouchExpandable(true);
        pane.add(_splitPane, BorderLayout.CENTER);

        // Menüleiste
        JMenuBar menuBar = new JMenuBar();
        menuBar.add(createFileMenu());
        menuBar.add(createViewMenu());
        menuBar.add(HelpMenu.createHelpMenu(new GtmHelp(), "bedienung.html", _frame));
        _frame.setJMenuBar(menuBar);
    }

    private static JMenu createFileMenu() {
        JMenu result = new JMenu("Datei");
        result.add(new AbstractAction("Neue Verbindung...") {
            @Override
            public void actionPerformed(final ActionEvent e) {
                StandardApplicationRunner.restart(new GenericTestMonitor(), new String[] {});
            }
        });
        result.addSeparator();
        result.add(new AbstractAction("Beenden") {
            @Override
            public void actionPerformed(final ActionEvent e) {
                System.exit(0);
            }
        });
        return result;
    }

    public static String getTitle(final String title, final ClientDavInterface connection) {
        if (connection == null) {
            return title;
        }
        String address = connection.getClientDavParameters().getDavCommunicationAddress();
        int port = connection.getClientDavParameters().getDavCommunicationSubAddress();
        return title + " - " + address + ":" + port;
    }

    public PreselectionLists getPreselectionLists() {
        return _preselectionPanel.getPreselectionLists();
    }

    private JMenu createViewMenu() {
        JMenu result = new JMenu("Ansicht");
        JMenu permissionsMenu = new JMenu("Berechtigungen");
        result.add(permissionsMenu);
        JCheckBoxMenuItem viewPermissionsCheckBox = new JCheckBoxMenuItem("Berechtigungen anzeigen");
        permissionsMenu.add(viewPermissionsCheckBox);
        viewPermissionsCheckBox.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(final ActionEvent e) {
//				viewPermissionsCheckBox.setSelected(!viewPermissionsCheckBox.isSelected());
                _viewPermissions = viewPermissionsCheckBox.isSelected();
                refreshPermissionsView();
            }
        });
        JMenu userMenu = new JMenu("Berechtigungen für Benutzer");
        permissionsMenu.add(userMenu);
        userMenu.addMenuListener(new MenuListener() {
            @Override
            public void menuSelected(final MenuEvent e) {
                for (SystemObject object : _connection.getDataModel().getType("typ.benutzer").getObjects()) {
                    JRadioButtonMenuItem item = new JRadioButtonMenuItem(object.getName(), Objects.equals(object, _selectedPermissionsUser));
                    userMenu.add(item);
                    item.addActionListener(e1 -> {
                        _selectedPermissionsUser = object;
                        refreshPermissionsView();
                    });
                }
            }

            @Override
            public void menuDeselected(final MenuEvent e) {
                userMenu.removeAll();
            }

            @Override
            public void menuCanceled(final MenuEvent e) {
            }
        });
        return result;
    }

    private void refreshPermissionsView() {
        AccessControlManager manager = _connection.getAccessControlManager();

        if (_selectedPermissionsUser == null || !_viewPermissions) {
            _preselectionPanel.getPreselectionLists().setObjectListRenderer(null);
            return;
        }

        UserInfo userPermissions = manager.getUserPermissions(_selectedPermissionsUser);

        SystemObjectListRenderer renderer = new SystemObjectListRenderer() {
            @Override
            public String getText(final SystemObject obj) {
                return obj.toString();
            }

            @Nullable
            @Override
            public Icon getIcon(final SystemObject obj) {
                AttributeGroupUsage usage = getAttributeGroupUsage();
                if (usage == null) {
                    return null;
                }
                return getPermissionsIcon(userPermissions, obj, usage);
            }

            @Nullable
            @Override
            public String getTooltip(final SystemObject object) {
                AttributeGroupUsage usage = getAttributeGroupUsage();
                if (usage == null) {
                    return null;
                }
                PermissionsState state = new PermissionsState(userPermissions, object, usage);
                return "Sender: " + formatValue(state.getSender()) + "<br>Empfänger: " + formatValue(state.getReceiver()) + "<br>Quelle: " +
                       formatValue(state.getSource()) + "<br>Senke: " + formatValue(state.getDrain());
            }

            private String formatValue(final PermissionsState.InnerState state) {
	            return switch (state) {
		            case Allowed -> "<b>Erlaubt</b>";
		            case Forbidden -> "<b>Verboten</b>";
		            case AllowedWrongUsage -> "<b>Erlaubt (für Testzwecke)</b>";
		            default -> "-";
	            };
            }

            @Nullable
            public AttributeGroupUsage getAttributeGroupUsage() {
                List<AttributeGroup> atgs = _preselectionPanel.getPreselectionLists().getSelectedAttributeGroups();
                List<Aspect> aspects = _preselectionPanel.getPreselectionLists().getSelectedAspects();
                if (atgs.size() != 1 || aspects.size() != 1) {
                    return null;
                }
                AttributeGroup attributeGroup = atgs.get(0);
                Aspect aspect = aspects.get(0);
                AttributeGroupUsage usage = attributeGroup.getAttributeGroupUsage(aspect);
                if (usage == null || usage.isConfigurating()) {
                    return null;
                }
                return usage;
            }
        };
        _preselectionPanel.getPreselectionLists().setObjectListRenderer(renderer);
    }

    private Icon getPermissionsIcon(final UserInfo userPermissions, final SystemObject obj, final AttributeGroupUsage usage) {
        PermissionsState state = new PermissionsState(userPermissions, obj, usage);
        return _iconCache.computeIfAbsent(state, this::drawIcon);
    }

    private Icon drawIcon(final PermissionsState permissionsState) {
        BufferedImage icon = new BufferedImage(35, 16, BufferedImage.TYPE_4BYTE_ABGR);
        Graphics2D graphics = icon.createGraphics();
        graphics.setRenderingHints((Map<?, ?>) Toolkit.getDefaultToolkit().getDesktopProperty("awt.font.desktophints"));
        graphics.setRenderingHint(RenderingHints.KEY_ANTIALIASING, RenderingHints.VALUE_ANTIALIAS_ON);
        graphics.setRenderingHint(RenderingHints.KEY_STROKE_CONTROL, RenderingHints.VALUE_STROKE_PURE);
        graphics.translate(.5, .5);
        graphics.setStroke(stroke);
        drawCircle(graphics, 6, 8, permissionsState.getSource(), true);
        drawCircle(graphics, 28, 8, permissionsState.getDrain(), true);
        drawCircle(graphics, 6, 8, permissionsState.getSender(), false);
        drawCircle(graphics, 28, 8, permissionsState.getReceiver(), false);
        graphics.setStroke(new BasicStroke(1.0f));
        graphics.setPaint(Color.darkGray);
        graphics.drawLine(14, 8, 20, 8);
        graphics.drawLine(20, 8, 16, 4);
        graphics.drawLine(20, 8, 16, 12);

        graphics.dispose();
        return new ImageIcon(icon);
    }

    private void drawCircle(final Graphics2D graphics, final int x, final int y, final PermissionsState.InnerState state, final boolean fill) {
        Paint color = Color.lightGray;
        switch (state) {
            case Allowed:
                if (fill) {
                    color = FILL_ALLOWED;
                } else {
                    color = STROKE_ALLOWED;
                }
                break;
            case AllowedWrongUsage:
                if (fill) {
                    color = FILL_ALLOWED_WRONG;
                } else {
                    color = STROKE_ALLOWED_WRONG;
                }
                break;
            case Forbidden:
                if (fill) {
                    color = FILL_FORBIDDEN;
                } else {
                    color = STROKE_FORBIDDEN;
                }
                break;
            case Ignore:
                return;
        }
        graphics.setPaint(color);
        if (fill) {
            graphics.fillOval(x - 4, y - 4, 8, 8);
        } else {
            graphics.drawOval(x - 4, y - 4, 8, 8);
        }
    }

    /**
     * Mit dieser Methode wird ein {@link ExternalModule Modul} zur Applikation hinzugefügt.
     *
     * @param module ein Modul
     */
    public void addModule(ExternalModule module) {
        // Eintrag in die Hashmap
        _moduleMap.put(module.getClass().getName(), module);

        ModuleButton button = new ModuleButton(module);
        button.setMaximumSize(new Dimension(_screenSize.width, button.getMaximumSize().height));
        _modulePanel.add(button);
        _modulePanel.add(Box.createVerticalStrut(5));
        _preselectionPanel.addPreselectionListener(button);
        _frame.revalidate();
        _frame.repaint();
    }

    /** Fügt zwischen die Buttons, mit denen die Module gestartet werden können, einen optischen Abstandshalter ein. */
    public void addSeparator() {
        _modulePanel.add(Box.createVerticalStrut(15));
    }

    /**
     * Fügt über die Buttons, mit denen die Module gestartet werden können, ein Logo ein.
     *
     * @param icon das Logo
     */
    public void addLogo(final Icon icon) {
        JLabel label = new JLabel(icon);
        label.setBorder(BorderFactory.createEmptyBorder(1, 1, 1, 1));
        _logoPanel.add(label);
        _logoPanel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
    }

    /** Stellt die Anwendung dar und lädt die Einstellungen aus den Preferences. */
    public void start() {
        _frame.pack();
        _frame.setLocation((_screenSize.width - _frame.getSize().width) / 2, (_screenSize.height - _frame.getSize().height) / 2);
        _frame.setVisible(true);
        try {
            _settingsHandler.loadAllSettings();
        } catch (BackingStoreException ex) {
            ex.printStackTrace();
        }
    }

    /**
     * Gibt die Verbindung zum Datenverteiler zurück.
     *
     * @return die Verbindung zum Datenverteiler
     */
    @Override
    public ClientDavInterface getConnection() {
        return _connection;
    }

    /**
     * Nimmt die Einstellungsdaten der Module entgegen und übergibt sie an das Panel, welches die Einstellungen verwaltet.
     *
     * @param settingsData die Einstellungsdaten
     */
    @Override
    public void saveSettings(final SettingsData settingsData) {
        _settingsHandler.saveSettings(settingsData);
    }

    /**
     * Gibt die Argumente zurück, die beim Aufruf der {@code main}-Methode übergeben wurden.
     *
     * @return die Liste der Argumente
     */
    @Override
    @Deprecated
    public List<String> getArgumentList() {
        return _argumentList;
    }

    /**
     * Es wird die Argumentliste gesetzt, die beim Starten der Anwendung als Parameter angegeben wurde.
     *
     * @param argumentList die Argumentliste
     */
    @Deprecated
    public void setArgumentList(List<String> argumentList) {
        _argumentList = argumentList;
    }

    /**
     * Gibt das Hauptfenster der Anwendung zurück.
     *
     * @return das Hauptfenster
     */
    @Override
    public Window getParent() {
        return _frame;
    }

    /**
     * Gibt die Parameter für die Vorauswahl (Baum) zurück. Die Collection enthält Systemobjekte und Knotenobjekte. Anhand der Objekte wird der Baum
     * für die Vorauswahl erzeugt.
     *
     * @return die Sammlung von System- und Knotenobjekten
     */
    @Override
    public Collection<Object> getTreeNodes() {
        return _preselectionPanel.getPreselectionTree().getTreeNodes();
    }

    /**
     * Ermittelt anhand des Modulnamens das zugehörige Objekt und gibt es zurück.
     *
     * @param moduleName der Name des gesuchten Moduls
     *
     * @return das Objekt des gesuchten Moduls, {@code null}, wenn kein passendes Modul gespeichert ist
     */
    public ExternalModule getExternalModule(String moduleName) {
        return _moduleMap.get(moduleName);
    }

    /**
     * Fügt ein weiteres Modul in die Liste der Applikation ein.
     *
     * @param externalModule neues Modul
     */
    public void setExternalModule(final ExternalModule externalModule) {
        externalModule.setApplication(this);
        String className = externalModule.getClass().getName();
        if (_moduleMap.containsKey(className)) {
            _moduleMap.put(className, externalModule);
        }
    }

    /**
     * Diese Klasse erstellt für ein {@link ExternalModule Modul} einen Button für die Applikation. Der Button erhält vom Modul einen Text und einen
     * Tooltip. Weiterhin wird das Modul über Änderungen in der {@link PreselectionLists PreselectionList} informiert.
     */
    private class ModuleButton extends JButton implements PreselectionListsListener {

        /** speichert das Modul */
        private ExternalModule _module;

        /** speichert die Datenidentifikation (Objekttypen, Attributgruppe, Aspekt und Objekte) in den Einstellungsdaten */
        private SettingsData _settingsData;

        private PreselectionLists _preselectionLists;

        /**
         * Konstruktor. Erstellt anhand der Informationen aus dem {@link ExternalModule Modul} einen Button mit Tooltip.
         *
         * @param module ein Modul
         */
        public ModuleButton(final ExternalModule module) {
            _module = module;
            _module.setApplication(GenericTestMonitorApplication.this);
            setText(module.getButtonText());
            setToolTipText(module.getTooltipText());
            addActionListener(new ActionListener() {
                @Override
                public void actionPerformed(ActionEvent e) {
                    _settingsData.setTreePath(_preselectionPanel.getPreselectionTree().getSelectedTreePath());
                    _settingsData.setSimulationVariant(_preselectionLists.getSimulationVariant());
                    _module.startModule(_settingsData);
                }
            });
            listSelectionChanged(_preselectionPanel.getPreselectionLists()); // Enabled-Status setzen
        }

        /**
         * Verarbeitet die in der {@link de.bsvrz.pat.sysbed.preselection.lists.PreselectionLists PreselectionList} ausgewählten Elemente.
         *
         * @param preselectionLists die Auswahllisten für die Datenidentifikation
         */
        @Override
        public void listSelectionChanged(final PreselectionLists preselectionLists) {
            _preselectionLists = preselectionLists;
            List<SystemObjectType> objectTypes = preselectionLists.getSelectedObjectTypes();
            List<AttributeGroup> atgs = preselectionLists.getSelectedAttributeGroups();
            AttributeGroup atg = null;
            if (atgs.size() == 1) {
                atg = atgs.get(0);
            }
            List<Aspect> asps = preselectionLists.getSelectedAspects();
            Aspect asp = null;
            if (asps.size() == 1) {
                asp = asps.get(0);
            }
            List<SystemObject> objects = preselectionLists.getSelectedObjects();
            _settingsData = new SettingsData(objectTypes, atg, asp, objects);
            setEnabled(_module.isPreselectionValid(_settingsData));
            setToolTipText(_module.getTooltipText());
        }
    }
}
