/*
 * Copyright 2005 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.plugins.api;

import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.KeyStroke;

/**
 * Diese Klasse stellt ein Panel mit den Buttons "Hilfe", "Speichern unter ...", "Abbrechen" und "OK" dar. Eine Klasse, welche das Interface {@link
 * DialogInterface} implementiert, implementiert dadurch die Funktionalität dieser drei Buttons.
 *
 * @author Kappich Systemberatung
 * @see DialogInterface
 */
public class ButtonBar extends JPanel {

    /** speichert den "OK" - Button */
    private final JButton _acceptButton;

    /** speichert ein Objekt, welches das Interface {@link DialogInterface} implementiert hat */
    private final DialogInterface _dialog;

    /**
     * Dem Konstruktor wird ein Objekt übergeben, welches das Interface {@link DialogInterface} implementiert hat. Dieses Objekt liefert die
     * Funktionalität der Buttons "Speichern...", "Abbrechen" und "OK" dieser Klasse.
     *
     * @param dialog ein Objekt, welches das Interface {@code DialogInterface} implementiert
     */
    public ButtonBar(DialogInterface dialog) {
        _dialog = dialog;

        /* speichert den "Hilfe" - Button */
        final JButton helpButton = new JButton("Hilfe");
        helpButton.addActionListener(e -> dialog.openHelp());
        /* speichert den "Speichern..." - Button */
        final JButton saveButton = new JButton("Speichern unter ...");
        saveButton.addActionListener(e -> {
            String title = JOptionPane.showInputDialog("Bitte einen Namen vergeben: ");
            if (title != null) {
                _dialog.doSave(title);
            }
        });
        /* speichert den "Abbrechen" - Button */
        final CancelButton cancelButton = new CancelButton("Abbrechen");
        cancelButton.addActionListener(e -> _dialog.doCancel());
        _acceptButton = new JButton("OK");          // bei Return aktivieren
        _acceptButton.addActionListener(e -> _dialog.doOK());

        setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
        setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
        add(helpButton);
        add(Box.createHorizontalGlue());
        add(saveButton);
        add(Box.createHorizontalStrut(5));
        add(cancelButton);
        add(Box.createHorizontalStrut(5));
        add(_acceptButton);
    }

    /**
     * Gibt den "OK" - Button des Panels zurück.
     *
     * @return der "OK" - Button
     */
    public JButton getAcceptButton() {
        return _acceptButton;
    }

    @Override
    public String toString() {
        return "ButtonBar{}";
    }

    /** Die Klasse {@code CancelButton} erweitert den {@code JButton}, damit dieser auf die Taste ESC reagiert. */
    private static class CancelButton extends JButton {

        /**
         * Dem Konstruktor wird ein Titel des Buttons übergeben. Zusätzlich wird die ESC-Taste an diesen Button gebunden. D.h. wenn die ESC-Taste
         * betätigt wird, wird dieser Button ausgeführt.
         *
         * @param title Titel des Buttons
         */
        public CancelButton(String title) {
            super(title);
            ActionListener al = event -> {
                String cmd = event.getActionCommand();
                if (cmd.equals("PressedESCAPE")) {
                    doClick();
                }
            };
            registerKeyboardAction(al, "PressedESCAPE", KeyStroke.getKeyStroke(KeyEvent.VK_ESCAPE, 0), JComponent.WHEN_IN_FOCUSED_WINDOW);
        }
    }
}
