/*
 * Copyright 2018-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.plugins.archiveinfo;

import com.google.common.base.Joiner;
import com.google.common.collect.ImmutableRangeSet;
import com.google.common.collect.Range;
import com.google.common.collect.RangeSet;
import de.bsvrz.dav.daf.main.archive.ArchiveDataSpecification;
import de.bsvrz.dav.daf.main.archive.ArchiveInformationResult;
import de.bsvrz.dav.daf.main.archive.ArchiveRequestManager;
import de.bsvrz.dav.daf.main.archive.TimingType;
import de.bsvrz.pat.sysbed.main.ApplicationInterface;
import de.bsvrz.pat.sysbed.plugins.api.settings.SettingsData;
import de.bsvrz.pat.sysbed.plugins.archiverequest.StreamBasedArchiveRequestModule;
import java.awt.BorderLayout;
import java.awt.Dimension;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.format.FormatStyle;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;
import java.util.stream.Collectors;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JButton;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JViewport;

/**
 * Fenster für das Ergebnis einer Info-Anfrage
 *
 * @author Kappich Systemberatung
 */
public final class ArchiveInfoResultDialog extends JDialog {

    public static final DateTimeFormatter DATE_TIME_FORMATTER = DateTimeFormatter.ofLocalizedDateTime(FormatStyle.SHORT);
    private final String _archivePid;
    private JLabel _selectionLabel;
    private DataView _dataView;
    private JButton _backup;
    private JButton _delete;
    private JButton _deleteLater;
    private JButton _requestData;
    private JButton _request;
    private JButton _restore;

    public ArchiveInfoResultDialog(final ArchiveInfoRequester requester, final Map<ArchiveDataSpecification, List<ArchiveInformationResult>> data,
                                   final String archivePid, final ApplicationInterface application) {
        super(requester.getParent(), "Archiv-Informationsanfrage");
        _archivePid = archivePid;
        TreeSet<Long> timestamps = new TreeSet<>();
        for (List<ArchiveInformationResult> list : data.values()) {
            for (ArchiveInformationResult item : list) {
                timestamps.add(item.getIntervalStart());
                timestamps.add(item.getIntervalEnd());
            }
        }
        _selectionLabel = new JLabel(formatSelection(ImmutableRangeSet.of()));
        JPanel view = new JPanel(new BorderLayout());
        JScrollPane jScrollPane = new JScrollPane(view);
        jScrollPane.getViewport().setScrollMode(JViewport.SIMPLE_SCROLL_MODE);
        _dataView = new DataView(timestamps, data, jScrollPane.getViewport()) {
            @Override
            protected void selectionChanged() {
                RangeSet<Long> selectionInterval = _dataView.getSelectionInterval();
                _selectionLabel.setText(formatSelection(selectionInterval));
                _delete.setEnabled(!selectionInterval.isEmpty());
                _deleteLater.setEnabled(!selectionInterval.isEmpty());
                _requestData.setEnabled(!selectionInterval.isEmpty());
                _request.setEnabled(!selectionInterval.isEmpty());
                _restore.setEnabled(!selectionInterval.isEmpty());
            }
        };
        view.add(_dataView, BorderLayout.CENTER);
        DataRowHeader dataRowHeader = new DataRowHeader(data.keySet());
        view.add(dataRowHeader, BorderLayout.WEST);
        dataRowHeader.validate();
        JPanel contentPane = new JPanel(new BorderLayout());
        contentPane.add(jScrollPane, BorderLayout.CENTER);
        jScrollPane.setMinimumSize(new Dimension(640, 400));
        jScrollPane.setPreferredSize(new Dimension(640, 400));
        JPanel bottomBar = new JPanel();
        bottomBar.setLayout(new BoxLayout(bottomBar, BoxLayout.PAGE_AXIS));
        JPanel buttonBar = new JPanel();
        bottomBar.add(_selectionLabel);
        bottomBar.add(buttonBar);
        _selectionLabel.setAlignmentX(0.0f);
        buttonBar.setAlignmentX(0.0f);
        buttonBar.setLayout(new BoxLayout(buttonBar, BoxLayout.LINE_AXIS));
        buttonBar.setBorder(BorderFactory.createEmptyBorder(8, 8, 8, 8));
        _backup = new JButton("Alles Sichern");
        _backup.addActionListener(e -> requester.executeAsync(ArchiveRequestManager::savePersistentData));
        _restore = new JButton("Wiederherstellen");
        _restore.addActionListener(e -> requester.executeAsync(it -> it.restorePersistentData(_dataView.getSelection())));
        _delete = new JButton("Sofort Löschen");
        _delete.addActionListener(e -> requester.executeAsync(it -> it.deleteData(_dataView.getSelection(), true)));
        _deleteLater = new JButton("Später Löschen");
        _deleteLater.addActionListener(e -> requester.executeAsync(it -> it.deleteData(_dataView.getSelection(), false)));
        _requestData = new JButton("Nachfordern");
        _requestData.addActionListener(e -> requester.executeAsync(it -> it.requestData(_dataView.getSelection(), Collections.emptyList())));
        _request = new JButton("Archiv-Anfrage");
        _request.addActionListener(e -> {
            SettingsData settingsData = new SettingsData();
            HashMap<String, String> map = new HashMap<>();
            map.put("archivePid", _archivePid);
            map.put("timing",
                    data.keySet().stream().map(i -> i.getTimeSpec().getTimingType()).findAny().get() == TimingType.DATA_TIME ? "Datenzeitstempel" :
                    "Archivzeitstempel");
            Range<Long> span = _dataView.getSelectionInterval().span();
            map.put("from", span.lowerEndpoint().toString());
            map.put("to", span.upperEndpoint().toString());
            settingsData.setSettingsMap(map);
            settingsData.setObjects(data.keySet().stream().map(ArchiveDataSpecification::getObject).distinct().collect(Collectors.toList()));
            settingsData.setAttributeGroup(data.keySet().stream().map(i -> i.getDataDescription().getAttributeGroup()).findAny().get());
            settingsData.setAspect(data.keySet().stream().map(i -> i.getDataDescription().getAspect()).findAny().get());
            StreamBasedArchiveRequestModule module = new StreamBasedArchiveRequestModule("");
            module.setApplication(application);
            module.change(settingsData);
        });
        buttonBar.add(_backup);
        buttonBar.add(Box.createHorizontalGlue());
        buttonBar.add(Box.createHorizontalStrut(8));
        buttonBar.add(_restore);
        buttonBar.add(Box.createHorizontalStrut(8));
        buttonBar.add(_delete);
        buttonBar.add(Box.createHorizontalStrut(8));
        buttonBar.add(_deleteLater);
        buttonBar.add(Box.createHorizontalStrut(8));
        buttonBar.add(_requestData);
        buttonBar.add(Box.createHorizontalStrut(8));
        buttonBar.add(_request);
        contentPane.add(bottomBar, BorderLayout.SOUTH);
        setContentPane(contentPane);
        pack();
        setLocationRelativeTo(requester.getParent());
        _dataView.selectionChanged();
    }

    private static String formatSelection(final RangeSet<Long> selectionInterval) {
        String join = Joiner.on("; ").join(selectionInterval.asRanges().stream().map(ArchiveInfoResultDialog::formatRange).iterator());
        if (join.isEmpty()) {
            return "Mit der Maus einen Bereich markieren um einen Zeitbereich auswählen";
        }
        return join;
    }

    private static String formatRange(final Range<Long> it) {
        return formatDate(it.lowerEndpoint()) + " – " + formatDate(it.upperEndpoint());
    }

    private static String formatDate(final Long epochMilli) {
        return DATE_TIME_FORMATTER.format(Instant.ofEpochMilli(epochMilli).atZone(ZoneId.systemDefault()));
    }
}
