/*
 * Copyright 2018-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.plugins.archiveinfo;

import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.archive.ArchiveAvailabilityListener;
import de.bsvrz.dav.daf.main.archive.ArchiveRequestManager;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.pat.sysbed.plugins.api.ButtonBar;
import de.bsvrz.pat.sysbed.plugins.api.DataIdentificationChoice;
import de.bsvrz.pat.sysbed.plugins.api.DialogInterface;
import de.bsvrz.pat.sysbed.plugins.api.settings.SettingsData;
import de.bsvrz.sys.funclib.kappich.collections.CollatorComparator;
import de.bsvrz.sys.funclib.kappich.collections.SortedListModel;
import java.awt.BorderLayout;
import java.awt.Window;
import java.util.Locale;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;

/**
 * Einstellungsdialog für Archiv-Informationsanfragen
 *
 * @author Kappich Systemberatung
 */
public final class ArchiveInfoSettingsDialog extends JDialog implements DialogInterface, ArchiveAvailabilityListener {

    private final ArchiveInfoModule _archiveInfoModule;
    private final ArchiveInfoSettings _settings;
    private final ButtonBar _buttonBar;
    private ArchiveRequestManager _archive;
    private JComboBox<SystemObject> _archiveCombo;
    private JCheckBox _archiveTimeCheckBox;
    private JCheckBox _orderCheckBox;

    public ArchiveInfoSettingsDialog(final ArchiveInfoModule archiveInfoModule, final Window parent, final ArchiveInfoSettings settings) {
        super(parent, "Archivsystem - Informationsanfrage (Experimentell)");
        _archiveInfoModule = archiveInfoModule;
        _settings = settings;
        JPanel contentPane = new JPanel();
        contentPane.setLayout(new BoxLayout(contentPane, BoxLayout.Y_AXIS));

        DataIdentificationChoice dataIdentificationChoice =
            new DataIdentificationChoice(null, _archiveInfoModule.getConnection().getDataModel().getBaseTypes());
        contentPane.add(dataIdentificationChoice);
        dataIdentificationChoice.setDataIdentification(settings.getObjectTypes(), settings.getDataDescription().getAttributeGroup(),
                                                       settings.getDataDescription().getAspect(), settings.getObjects(),
                                                       settings.getDataDescription().getSimulationVariant());

        dataIdentificationChoice.addChoiceListener(newAtg -> {
            settings.setDataDescription(new DataDescription(dataIdentificationChoice.getAttributeGroup(), dataIdentificationChoice.getAspect(),
                                                            (short) dataIdentificationChoice.getSimulationVariant()));
            settings.getObjects().clear();
            settings.getObjects().addAll(dataIdentificationChoice.getObjects());
        });

        JPanel archiveSelectPanel = new JPanel();
        archiveSelectPanel.setLayout(new BoxLayout(archiveSelectPanel, BoxLayout.X_AXIS));
        JLabel label = new JLabel("Archivsystem:");
        _archiveCombo = new JComboBox<>();
        SortedListModel<SystemObject> model = new SortedListModel<>(new CollatorComparator<>(SystemObject::getNameOrPidOrId, Locale.GERMAN));
        model.setElements(archiveInfoModule.getConnection().getDataModel().getType("typ.archiv").getElements());
        _archiveCombo.setModel(model);
        if (settings.getArchive().isEmpty()) {
            _archiveCombo.setSelectedItem(archiveInfoModule.getConnection().getLocalConfigurationAuthority());
        } else {
            _archiveCombo.setSelectedItem(archiveInfoModule.getConnection().getDataModel().getObject(settings.getArchive()));
        }
        label.setLabelFor(_archiveCombo);
        archiveSelectPanel.add(Box.createHorizontalStrut(7));
        archiveSelectPanel.add(label);
        archiveSelectPanel.add(Box.createHorizontalStrut(5));
        archiveSelectPanel.add(_archiveCombo);
        archiveSelectPanel.add(Box.createHorizontalStrut(7));

        contentPane.add(Box.createVerticalStrut(5));
        contentPane.add(archiveSelectPanel);
        contentPane.add(Box.createVerticalStrut(5));

        JPanel group = new JPanel();
        group.setBorder(BorderFactory.createTitledBorder("Zeitbereich"));
        group.setLayout(new BorderLayout());
        group.add(new TimeSpecificationPanel(settings), BorderLayout.CENTER);
        contentPane.add(group);

        _archiveTimeCheckBox = new JCheckBox("Abfrage nach Archivzeit");
        _orderCheckBox = new JCheckBox("Nachgelieferte Daten nach Datenzeit einsortieren");
        contentPane.add(_archiveTimeCheckBox);
        contentPane.add(_orderCheckBox);
        _archiveTimeCheckBox.setSelected(settings.isQueryByArchiveTime());
        _orderCheckBox.setSelected(settings.isSortDelayedByDataTime());
        _archiveTimeCheckBox.addChangeListener(e -> _settings.setQueryByArchiveTime(_archiveTimeCheckBox.isSelected()));
        _orderCheckBox.addChangeListener(e -> _settings.setSortDelayedByDataTime(_orderCheckBox.isSelected()));

        _buttonBar = new ButtonBar(this);
        contentPane.add(_buttonBar);
        setContentPane(contentPane);
        pack();
        setLocationRelativeTo(parent);

        _archiveCombo.addActionListener(e -> refreshArchive());
        refreshArchive();
    }

    private void refreshArchive() {
        SystemObject selectedItem = (SystemObject) _archiveCombo.getSelectedItem();
        if (selectedItem == null) {
            return;
        }
        if (_archive != null) {
            _archive.removeArchiveAvailabilityListener(ArchiveInfoSettingsDialog.this);
        }
        _archive = _archiveInfoModule.getConnection().getArchive(selectedItem);
        _archive.addArchiveAvailabilityListener(ArchiveInfoSettingsDialog.this);
        _buttonBar.getAcceptButton().setEnabled(_archive.isArchiveAvailable());
        _settings.setArchive(selectedItem.getPid());
    }

    @Override
    public void doOK() {
        SettingsData settings = _settings.getSettings("");
        _archiveInfoModule.startSettings(settings);
        doCancel();
        _archiveInfoModule.saveSettings(settings);
    }

    @Override
    public void doCancel() {
        setVisible(false);
        dispose();
    }

    @Override
    public void doSave(final String title) {
        SettingsData settings = _settings.getSettings(title);
        _archiveInfoModule.saveSettings(settings);
    }

    @Override
    public void archiveAvailabilityChanged(final ArchiveRequestManager archive) {
        _buttonBar.getAcceptButton().setEnabled(archive.isArchiveAvailable());
    }
}
