/*
 * Copyright 2018-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.plugins.archiveinfo;

import static java.awt.GridBagConstraints.BASELINE_LEADING;
import static java.awt.GridBagConstraints.EAST;
import static java.awt.GridBagConstraints.HORIZONTAL;
import static java.awt.GridBagConstraints.NONE;


import java.awt.GridBagConstraints;
import java.awt.GridBagLayout;
import java.awt.Insets;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.FocusAdapter;
import java.awt.event.FocusEvent;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.Date;
import javax.swing.ButtonGroup;
import javax.swing.DefaultComboBoxModel;
import javax.swing.JComboBox;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JSpinner;
import javax.swing.SpinnerDateModel;
import javax.swing.SpinnerNumberModel;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

/**
 * Panel für die Angabe einer Zeitspezifikation
 *
 * @author Kappich Systemberatung
 */
public final class TimeSpecificationPanel extends JPanel implements ChangeListener, ActionListener {
    private final ArchiveInfoSettings _settings;
    private final JRadioButton _fromRelative;
    private final JRadioButton _fromAbsolute;
    private final JRadioButton _fromAll;
    private final JSpinner _fromAbsoluteDate;
    private final JComboBox<Object> _fromRelativeTimingOption;
    private final JSpinner _fromRelativeNumber;
    private final JSpinner _toDate;
    private final JRadioButton _toNow;
    private final JRadioButton _toAbsolute;
    private final JSpinner _toAbsoluteDate;

    public TimeSpecificationPanel(final ArchiveInfoSettings settings) {
        super(new GridBagLayout());
        _settings = settings;

        GridBagConstraints c = new GridBagConstraints();

        c.gridx = 0;
        c.gridy = 0;
        c.anchor = BASELINE_LEADING;
        c.insets = new Insets(0, 0, 5, 5);

        _fromAll = new JRadioButton("Alle Daten");
        _fromRelative = new JRadioButton("Mindestens");
        _fromAbsolute = new JRadioButton("Von Datum");

        _toNow = new JRadioButton("Jetzt");
        _toAbsolute = new JRadioButton("Datum");

        ButtonGroup buttonGroup = new ButtonGroup();
        buttonGroup.add(_fromAll);
        buttonGroup.add(_fromRelative);
        buttonGroup.add(_fromAbsolute);

        ButtonGroup buttonGroup2 = new ButtonGroup();
        buttonGroup2.add(_toNow);
        buttonGroup2.add(_toAbsolute);

        c.gridwidth = 3;
        add(_fromAll, c);
        c.gridwidth = 1;
        c.gridy = 1;
        add(_fromRelative, c);
        c.gridy = 2;
        add(_fromAbsolute, c);

        c.gridy = 1;
        c.gridx = 1;
        c.weightx = 1.0;
        c.fill = HORIZONTAL;
        JSpinner dummy = new JSpinner(new SpinnerNumberModel(1, 1, 100, 1));
        _fromRelativeNumber = new JSpinner(new SpinnerNumberModel((Long) 10L, (Long) 1L, (Long) Long.MAX_VALUE, (Long) 1L));
        _fromRelativeNumber.setMinimumSize(dummy.getMinimumSize());
        _fromRelativeNumber.setPreferredSize(dummy.getPreferredSize());
        add(_fromRelativeNumber, c);
        c.weightx = 0;
        c.gridx = 2;
        c.fill = NONE;
        _fromRelativeTimingOption = new JComboBox<>();
        _fromRelativeTimingOption.setModel(new DefaultComboBoxModel<>(new TimingOption[] {TimingOption.DataSets,
                                                                                          TimingOption.Minutes,
                                                                                          TimingOption.Hours,
                                                                                          TimingOption.Days,
                                                                                          TimingOption.Weeks,
                                                                                          TimingOption.Months,
                                                                                          TimingOption.Years}));
        add(_fromRelativeTimingOption, c);
        c.gridx = 1;
        c.gridy = 2;
        c.gridwidth = 2;
        _fromAbsoluteDate = new JSpinner(new SpinnerDateModel());
        add(_fromAbsoluteDate, c);

        c.gridx = 3;
        c.gridy = 0;
        c.gridheight = 3;
        c.gridwidth = 1;
        c.insets = new Insets(10, 10, 15, 15);
        c.anchor = EAST;
        add(new JLabel("bis"), c);
        c.anchor = BASELINE_LEADING;
        c.insets = new Insets(0, 0, 5, 5);

        c.gridx = 4;
        c.gridy = 1;
        c.gridheight = 1;
        c.gridwidth = 2;

        add(_toNow, c);
        c.gridy = 2;
        c.gridwidth = 1;
        add(_toAbsolute, c);
        c.gridx = 5;
        _toAbsoluteDate = new JSpinner(new SpinnerDateModel());
        _toDate = _toAbsoluteDate;
        add(_toDate, c);

        linkFocus(_fromRelativeNumber, _fromRelative);

        _fromRelativeTimingOption.addFocusListener(new FocusAdapter() {
            @Override
            public void focusGained(final FocusEvent e) {
                _fromRelative.setSelected(true);
            }
        });

        linkFocus(_fromAbsoluteDate, _fromAbsolute);
        linkFocus(_toAbsoluteDate, _toAbsolute);

        _fromRelativeTimingOption.setSelectedItem(TimingOption.Days);

        LocalDate now = LocalDate.now();
        Instant start = now.atStartOfDay().atZone(ZoneId.systemDefault()).toInstant();

        _fromAbsoluteDate.setValue(Date.from(start));
        _toAbsoluteDate.setValue(Date.from(start));

        switch (settings.getTimeOption()) {
            case DataSets:
            case Minutes:
            case Hours:
            case Days:
            case Weeks:
            case Months:
            case Years:
                _fromRelative.setSelected(true);
                _fromRelativeTimingOption.setSelectedItem(settings.getTimeOption());
                _fromRelativeNumber.setValue(settings.getTimeFrom());
                break;
            case Everything:
                _fromAll.setSelected(true);
                break;
            case Absolute:
                _fromAbsolute.setSelected(true);
                _fromAbsoluteDate.setValue(new Date(_settings.getTimeFrom()));
                break;
        }

        if (settings.getTimeTo() == -1) {
            _toNow.setSelected(true);
        } else {
            _toAbsolute.setSelected(true);
            _toAbsoluteDate.setValue(new Date(settings.getTimeTo()));
        }

        _fromAll.addChangeListener(this);
        _fromRelative.addChangeListener(this);
        _fromAbsolute.addChangeListener(this);
        _toNow.addChangeListener(this);
        _toAbsolute.addChangeListener(this);

        _fromRelativeTimingOption.addActionListener(this);
        _fromRelativeNumber.addChangeListener(this);
        _fromAbsoluteDate.addChangeListener(this);
        _toAbsoluteDate.addChangeListener(this);

    }

    private static void linkFocus(final JSpinner jSpinner, final JRadioButton jRadioButton) {
        ((JSpinner.DefaultEditor) jSpinner.getEditor()).getTextField().addFocusListener(new FocusAdapter() {
            @Override
            public void focusGained(final FocusEvent e) {
                jRadioButton.setSelected(true);
            }
        });
    }

    @Override
    public void stateChanged(final ChangeEvent e) {
        saveValue();
    }

    @Override
    public void actionPerformed(final ActionEvent e) {
        saveValue();
    }

    private void saveValue() {
        if (_fromAll.isSelected()) {
            _settings.setTimeOption(TimingOption.Everything);
            _settings.setTimeFrom(-1);
        } else if (_fromRelative.isSelected()) {
            _settings.setTimeOption((TimingOption) _fromRelativeTimingOption.getSelectedItem());
            _settings.setTimeFrom((Long) _fromRelativeNumber.getValue());
        } else if (_fromAbsolute.isSelected()) {
            _settings.setTimeOption(TimingOption.Absolute);
            _settings.setTimeFrom(((Date) _fromAbsoluteDate.getValue()).getTime());
        }

        if (_toNow.isSelected()) {
            _settings.setTimeTo(-1L);
        } else {
            _settings.setTimeTo(((Date) _toAbsoluteDate.getValue()).getTime());
        }
    }
}
