/*
 * Copyright 2005 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.pat.sysbed.plugins.archiverequest;

import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.DataState;
import de.bsvrz.dav.daf.main.archive.ArchiveAvailabilityListener;
import de.bsvrz.dav.daf.main.archive.ArchiveData;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKindCombination;
import de.bsvrz.dav.daf.main.archive.ArchiveDataQueryResult;
import de.bsvrz.dav.daf.main.archive.ArchiveDataSpecification;
import de.bsvrz.dav.daf.main.archive.ArchiveDataStream;
import de.bsvrz.dav.daf.main.archive.ArchiveOrder;
import de.bsvrz.dav.daf.main.archive.ArchiveQueryPriority;
import de.bsvrz.dav.daf.main.archive.ArchiveRequestManager;
import de.bsvrz.dav.daf.main.archive.ArchiveRequestOption;
import de.bsvrz.dav.daf.main.archive.ArchiveTimeSpecification;
import de.bsvrz.dav.daf.main.archive.TimingType;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.AttributeGroupUsage;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.pat.sysbed.dataview.ArchiveDataTableView;
import de.bsvrz.pat.sysbed.dataview.DataTableObject;
import de.bsvrz.pat.sysbed.dataview.csv.CsvProgressDialogArchive;
import de.bsvrz.pat.sysbed.dataview.csv.CsvUtils;
import de.bsvrz.pat.sysbed.dataview.csv.PerpetualCsvConverter;
import de.bsvrz.pat.sysbed.dataview.csv.PostProcessor;
import de.bsvrz.pat.sysbed.dataview.filtering.FilterAttributeGroup;
import de.bsvrz.pat.sysbed.help.GtmHelp;
import de.bsvrz.pat.sysbed.plugins.api.ButtonBar;
import de.bsvrz.pat.sysbed.plugins.api.DataIdentificationChoice;
import de.bsvrz.pat.sysbed.plugins.api.DialogInterface;
import de.bsvrz.pat.sysbed.plugins.api.ExternalModuleAdapter;
import de.bsvrz.pat.sysbed.plugins.api.FilterPanel;
import de.bsvrz.pat.sysbed.plugins.api.settings.SettingsData;
import de.bsvrz.sys.funclib.concurrent.Semaphore;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.awt.Container;
import java.awt.Dimension;
import java.awt.EventQueue;
import java.awt.FlowLayout;
import java.awt.GridLayout;
import java.awt.Window;
import java.awt.event.ItemEvent;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.lang.reflect.InvocationTargetException;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.ButtonGroup;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JComboBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JRadioButton;
import javax.swing.JSpinner;
import javax.swing.SpinnerDateModel;
import javax.swing.SpinnerModel;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingUtilities;

/**
 * Diese Klasse implementiert das Modul für eine streambasierte Archivanfrage. Mit Hilfe eines Dialogs werden alle benötigten Parameter eingestellt,
 * die die streambasierte Archivanfrage braucht. Hierzu gehört die Priorität, der (Zeit-)Bereich und die Art der Archivanfrage. Falls erforderlich muß
 * die Sortierreihenfolge der als nachgeliefert gekennzeichneten Archivdatensätzen angegeben werden. Weiterhin muß angegeben werden, ob es sich um
 * eine Zustands- oder Deltaanfrage handelt. Zur Darstellung in einer Tabelle kann die Sortierreihenfolge der Archivdaten angegeben werden. Zwei
 * Möglichkeiten stehen hierfür zur Verfügung. Es ist möglich nach der Zeit oder nach den Datenidentifikationen zu sortieren.
 *
 * @author Kappich Systemberatung
 */
public class StreamBasedArchiveRequestModule extends ExternalModuleAdapter {

    /** DebugLogger für Debug-Ausgaben */
    private static final Debug _debug = Debug.getLogger();
    /** speichert den Dialog des Archivmoduls */
    private static StreamBasedArchiveRequestDialog _moduleDialog;
    /** speichert den Wert für die Warnung bei großen Anfragen */
    private final String _warningLimitString;
    /** speichert den Text des Tooltips */
    private String _tooltipText;

    /**
     * Der Konstruktor mit einer Grenze für Warnungen bei großen Anfragen.
     */
    public StreamBasedArchiveRequestModule(final String warningLimit) {
        _warningLimitString = warningLimit;
    }

    /* ################### Methoden ########### */

    /**
     * Gibt den Namen des Moduls zurück.
     *
     * @return der Name des Moduls
     */
    @Override
    public String getModuleName() {
        return "Archivanfrage Stream";
    }

    /**
     * Gibt des Text des Buttons zurück.
     *
     * @return Text des Buttons
     */
    @Override
    public String getButtonText() {
        return "Archivanfrage (Stream)";
    }

    /**
     * Gibt den Text des Tooltips zurück.
     *
     * @return Text des Tooltips
     */
    @Override
    public String getTooltipText() {
        return _tooltipText;
    }

    /**
     * Diese Methode erhält alle ausgewählten Parameter und startet den {@link StreamBasedArchiveRequestDialog Dialog} zur Auswahl weiterer
     * Einstellungen der Archivanfrage.
     *
     * @param settingsData enthält die ausgewählte Datenidentifikation
     */
    @Override
    public void startModule(final SettingsData settingsData) {
        _moduleDialog = new StreamBasedArchiveRequestDialog(_warningLimitString, getApplication().getParent());
        _moduleDialog.setDataIdentification(settingsData);
    }

    /**
     * Diese Methode erhält alle Einstellungen für die Archivanfrage und startet diese ohne den Dialog anzuzeigen.
     *
     * @param settingsData die Einstellungsdaten
     */
    @Override
    public void startSettings(final SettingsData settingsData) {
        _moduleDialog = new StreamBasedArchiveRequestDialog(_warningLimitString, getApplication().getParent());
        _moduleDialog.startRequest(settingsData);
    }

    /**
     * Diese Methode erhält alle Einstellungen für die Archivanfrage und startet den {@link StreamBasedArchiveRequestDialog Dialog} und füllt ihn
     * entsprechend mit den Einstellungen.
     *
     * @param settingsData die Einstellungsdaten
     */
    @Override
    public void change(final SettingsData settingsData) {
        _moduleDialog = new StreamBasedArchiveRequestDialog(_warningLimitString, getApplication().getParent());
        _moduleDialog.setSettings(settingsData);
    }

    /**
     * Überprüft, ob die Voraussetzungen für das Modul gegeben sind.
     *
     * @param settingsData enthält die ausgewählte Datenidentifikation
     *
     * @return gibt zurück, ob die Voraussetzungen für das Modul gegeben sind
     */
    @Override
    public boolean isPreselectionValid(final SettingsData settingsData) {
        if (!super.isPreselectionValid(settingsData)) {
            _tooltipText = "Genau eine Attributgruppe, ein Aspekt und mindestens ein Objekt müssen ausgewählt sein.";
            return false;
        }

        // ATGV prüfen
        final AttributeGroupUsage atgUsage = settingsData.getAttributeGroup().getAttributeGroupUsage(settingsData.getAspect());
        if (atgUsage == null || atgUsage.isConfigurating()) {
            _tooltipText = "Es muss eine Online-Attributgruppenverwendung ausgewählt werden.";
            return false;
        }
        _tooltipText = "Auswahl übernehmen";
        return true;
    }

    /* ################ Klasse ArchiveRequestDialog ############# */

    @Override
    public String toString() {
        return "StreamBasedArchiveRequestModule{" + "_tooltipText='" + _tooltipText + '\'' + '}';
    }

    private static class SpinnerLongModel extends SpinnerNumberModel {

        public SpinnerLongModel(long value, long minimum, long maximum, long stepSize) {
            super(value, minimum, maximum, stepSize);
        }
    }

    /**
     * Stellt einen Dialog dar, womit Parameter für die Archivanfrage eingestellt werden können. Diese Einstellungen können gespeichert werden. Durch
     * betätigen des "OK"-Buttons werden die Einstellungen übernommen, die Archivanfrage gestartet und der Dialog geschlossen. Durch betätigen des
     * "Speichern unter ..."-Buttons werden die Einstellungen gespeichert.
     */
    private class StreamBasedArchiveRequestDialog implements DialogInterface {

        /** Gibt die möglichen Prioritäten für eine Archivanfrage an. */
        private final String[] _priorityUnit = {"Hoch", "Mittel", "Niedrig"};
        /** Gibt die möglichen (Zeit-)Bereiche an. */
        private final String[] _timingUnit = {"Datenzeitstempel", "Archivzeitstempel", "Datenindex"};
        /** Gibt die möglichen Sortierreihenfolgen der als nachgeliefert gekennzeichneten Archivdatensätze an. */
        private final String[] _sortOfDataUnit = {"Datenindex", "Datenzeitstempel"};
        /** Betrifft die Sortierung der Datensätze bei der Darstellung in einer Tabelle. */
        private final String[] _sortViewUnit = {"Zeitstempel", "Datenidentifikation"};
        /** gibt an, welche Zeichenkodierungen zur Verfügung stehen */
        private final String[] _encodingOptions = {"ISO-8859-1", "UTF-8", "MacRoman"};
        /** eine Parent-Komponente */
        private final Window _parent;
        /** speichert den Dialog */
        private JDialog _dialog;
        /** Speichert den Zugriff auf das Archivsystem. */
        private ArchiveRequestManager _currentArchiveRequestManager;
        /** speichert die Datenidentifikationsauswahl */
        private DataIdentificationChoice _dataIdentificationChoice;
        /** Das Panel für die (Zeit-)Bereichsanfrage. */
        private JPanel _domainPanel;
        /** Die Auswahlbox für die Prioritäten der Archivanfrage. */
        private JComboBox<String> _priorityComboBox;
        /** Die Auswahlbox für die möglichen (Zeit-)Bereiche der Archivanfrage. */
        private JComboBox<String> _timingComboBox;
        /** Gibt den Startwert des Zeitbereichs der Archivanfrage an. */
        private JSpinner _startTimeSpinner;
        /** Gibt den Endzeitpunkt des Zeitbereichs der Archivanfrage an. */
        private JSpinner _endTimeSpinner;
        /** Gibt den Startwert der Anfrage für des Datenindex an. */
        private JSpinner _startIndexSpinner;
        /** Gibt den Endwert der Anfrage für den Datenindex an. */
        private JSpinner _endIndexSpinner;
        /** Gibt an, wieviele Datensätze vor dem Endwert der Archivanfrage übertragen werden sollen. */
        private JSpinner _numberOfDatasetsSpinner;
        /** Das Ankreuzfeld für die Art der Archivanfrage: aktuelle Daten */
        private JCheckBox _oaDataCheckBox;
        /** Das Ankreuzfeld für die Art der Archivanfrage: nachgefordert-aktuelle Daten */
        private JCheckBox _naDataCheckBox;
        /** Das Ankreuzfeld für die Art der Archivanfrage: nachgelieferte Daten */
        private JCheckBox _onDataCheckBox;
        /** Das Ankreuzfeld für die Art der Archivanfrage: nachgefordert-nachgelieferte Daten */
        private JCheckBox _nnDataCheckBox;
        /** Die Auswahlbox für die Sortierreihenfolge der als nachgeliefert gekennzeichneten Archivdatensätze. */
        private JComboBox<String> _sortSequenceComboBox;
        /** Die Auswahlbox für die Sortierung der Archivdatensätze zur Darstellung in einer Tabelle. */
        private JComboBox<String> _viewSortComboBox;
        /** Die Checkbox zur Festlegung, ob nur in eine CSV-Datei gespeichert werden soll. */
        private JCheckBox _onlyCsvCheckBox;
        /** hier kann die Zeichenkodierung für den CSV-Export festgelegt werden */
        private JComboBox<String> _encodingComboBox;
        /** zur Auswahl, ob historische Objekte berücksichtigt werden sollen */
        private JCheckBox _useOldObjectsBox;
        /** Auswahlschaltfläche, ob alle Datensätze in dem spezifizierten Zeitraum übergeben werden sollen. */
        private JRadioButton _stateRadioButton;
        /** Auswahlschaltfläche, ob nur Archivdatensätze übermittelt werden, die sich auch vom vorhergehenden Datensatz unterscheiden. */
        private JRadioButton _deltaRadioButton;
        /** Stellt die untere Buttonleiste dar mit den Buttons "Speichern unter", "Abbrechen" und "OK" */
        private ButtonBar _buttonBar;
        /** Gibt an, ob der ausgewählte Bereich relativ oder absolut ist. */
        private JCheckBox _relativeBox;
        /* hier kann ein Filter ausgewählt und bearbeitet werden */
        private FilterPanel _filterPanel;
        /** die Filterattributgruppe */
        private FilterAttributeGroup _filterAttributeGroup;
        /** das Limit, ab dem der Benutzer vor einer langen Abfrage gewarnt wird */
        private int _warningLimit = Integer.MAX_VALUE;
        /** zeigt an, ob ein Archivsystem zur Verfügung steht */
        private boolean _isArchiveAvailable;

        private JComboBox<SystemObject> _archiveBox;
        private ArchiveListener _archiveListener;

        /* ################# Methoden ################### */

        public StreamBasedArchiveRequestDialog(final String warningLimitString, @Nullable final Window parent) {
            if (warningLimitString != null) {
                try {
                    _warningLimit = Integer.parseInt(warningLimitString);
                } catch (NumberFormatException ignore) {
                }
            }
            _parent = parent;
        }

        /**
         * Mit dieser Methode können die Datenidentifikationsdaten übergeben werden.
         *
         * @param data enthält die ausgewählte Datenidentifikation
         */
        public void setDataIdentification(final SettingsData data) {
            if (_dialog == null) {
                createDialog();
            }
            _dataIdentificationChoice.setDataIdentification(data.getObjectTypes(), data.getAttributeGroup(), data.getAspect(), data.getObjects(),
                                                            data.getSimulationVariant());
            _dataIdentificationChoice.showTree(getApplication().getTreeNodes(), getApplication().getConnection(), data.getTreePath());
            _filterPanel.setAttributeGroup(data.getAttributeGroup());
            _filterPanel.setParent(_dialog);
            _dataIdentificationChoice.addChoiceListener(_filterPanel);
            showDialog();
        }

        /**
         * Diese Methode zeigt den Dialog an und trägt die Einstellungsdaten in die entsprechenden Felder ein.
         *
         * @param data Einstellungsdaten
         */
        @SuppressWarnings("OverlyLongMethod")
        public void setSettings(final SettingsData data) {
            if (_dialog == null) {
                createDialog();
            }
            _dataIdentificationChoice.setDataIdentification(data.getObjectTypes(), data.getAttributeGroup(), data.getAspect(), data.getObjects(),
                                                            data.getSimulationVariant());
            _dataIdentificationChoice.showTree(getApplication().getTreeNodes(), getApplication().getConnection(), data.getTreePath());
            _filterPanel.setAttributeGroup(data.getAttributeGroup());
            _filterPanel.setParent(_dialog);
            _dataIdentificationChoice.addChoiceListener(_filterPanel);

            String timing = "";
            String relative = "";
            String from = "";
            String to = "";
            Map<String, String> settingsMap = data.getSettingsMap();
            for (final Map.Entry<String, String> entry : settingsMap.entrySet()) {
                String value = entry.getValue();
                if (entry.getKey().equals("archivePid")) {
                    setArchivePid(value);
                }
                switch (entry.getKey()) {
                    case "priority":
                        setPriority(value);
                        break;
                    case "timing":
                        timing = value;
                        setTimingType(timing);
                        break;
                    case "relative":
                        relative = value;
                        setRelative(relative);
                        break;
                    case "from":
                        from = value;
                        break;
                    case "to":
                        to = value;
                        break;
                    case "archivetype":
                        setArchiveType(value);
                        break;
                    case "sortsequence":
                        setSortSequence(value);
                        break;
                    case "requestview":
                        setRequestView(value);
                        break;
                    case "viewsort":
                        setViewSort(value);
                        break;
                    case "oldobj":
                        setUseOldObjects(value);
                        break;
                    case "onlycsv":
                        setOnlyCsv(value);
                        break;
                    case "csvencoding":
                        setEncoding(value);
                        break;
                    case "filter":
                        setFilter(value);
                        break;
                }
            }
            setFrom(timing, from, relative);
            setTo(timing, to);

            showDialog();
        }

        private void setEncoding(final String value) {
            _encodingComboBox.setSelectedItem(value);
        }

        private void setFilter(final String value) {
            _filterPanel.setFilterByName(value);
        }

        /**
         * Startet die Archivanfrage anhand der Einstellungsdaten.
         *
         * @param settingsData die Einstellungsdaten
         */
        @SuppressWarnings({"OverlyCoupledMethod", "OverlyLongMethod"})
        public void startRequest(final SettingsData settingsData) {
            ArchiveQueryPriority archiveQueryPriority = ArchiveQueryPriority.LOW;
            TimingType timingType = TimingType.ARCHIVE_TIME;
            boolean startRelative = false;
            boolean useOldObjects = false;
            long intervalStart = 0;
            long intervalEnd = 0;
            ArchiveDataKindCombination archiveDataKindCombination = new ArchiveDataKindCombination(ArchiveDataKind.ONLINE);
            ArchiveOrder archiveOrder = ArchiveOrder.BY_INDEX;
            ArchiveRequestOption archiveRequestOption = ArchiveRequestOption.NORMAL;
            SystemObject archive = getConnection().getDataModel().getConfigurationAuthority();
            String filterName = null;
            boolean onlyCsv = false;
            String encoding = "UTF-8";

            Map<String, String> settingsMap = settingsData.getSettingsMap();
            for (final Map.Entry<String, String> entry : settingsMap.entrySet()) {
                String value = entry.getValue();
                if (entry.getKey().equals("archivePid")) {
                    archive = getConnection().getDataModel().getObject(value);
                }
                switch (entry.getKey()) {
                    case "priority":
	                    archiveQueryPriority = switch (value) {
		                    case "Hoch" -> ArchiveQueryPriority.HIGH;
		                    case "Mittel" -> ArchiveQueryPriority.MEDIUM;
		                    case "Niedrig" -> ArchiveQueryPriority.LOW;
		                    default -> archiveQueryPriority;
	                    };
                        break;
                    case "timing":
	                    timingType = switch (value) {
		                    case "Datenzeitstempel" -> TimingType.DATA_TIME;
		                    case "Archivzeitstempel" -> TimingType.ARCHIVE_TIME;
		                    case "Datenindex" -> TimingType.DATA_INDEX;
		                    default -> timingType;
	                    };
                        break;
                    case "relative":
                        startRelative = value.equals("true");
                        break;
                    case "from":
                        intervalStart = Long.parseLong(value);
                        break;
                    case "to":
                        intervalEnd = Long.parseLong(value);
                        break;
                    case "archivetype":
                        String[] types = value.split(" ");
                        ArchiveDataKind[] dataKinds = new ArchiveDataKind[types.length];
                        for (int i = 0; i < types.length; i++) {
                            String type = types[i];
                            switch (type) {
                                case "oa":
                                    dataKinds[i] = ArchiveDataKind.ONLINE;
                                    break;
                                case "on":
                                    dataKinds[i] = ArchiveDataKind.ONLINE_DELAYED;
                                    break;
                                case "na":
                                    dataKinds[i] = ArchiveDataKind.REQUESTED;
                                    break;
                                case "nn":
                                    dataKinds[i] = ArchiveDataKind.REQUESTED_DELAYED;
                                    break;
                            }
                        }
                        int length = dataKinds.length;
	                    archiveDataKindCombination = switch (length) {
		                    case 1 -> new ArchiveDataKindCombination(dataKinds[0]);
		                    case 2 -> new ArchiveDataKindCombination(dataKinds[0], dataKinds[1]);
		                    case 3 -> new ArchiveDataKindCombination(dataKinds[0], dataKinds[1], dataKinds[2]);
		                    case 4 ->
				                    new ArchiveDataKindCombination(dataKinds[0], dataKinds[1], dataKinds[2], dataKinds[3]);
		                    default -> new ArchiveDataKindCombination(ArchiveDataKind.ONLINE);
	                    };
                        break;
                    case "sortsequence":
                        if (value.equals("Datenindex")) {
                            archiveOrder = ArchiveOrder.BY_INDEX;
                        } else {     // nach Datenzeitstempel
                            archiveOrder = ArchiveOrder.BY_DATA_TIME;
                        }
                        break;
                    case "requestview":
                        if (value.equals("state")) {
                            archiveRequestOption = ArchiveRequestOption.NORMAL;
                        } else if (value.equals("delta")) {
                            archiveRequestOption = ArchiveRequestOption.DELTA;
                        }
                        break;
                    case "viewsort":      // Sortierung bei mehreren Datenidentifikationen
                        _debug.fine("Sortierung bei mehreren Datenidentifikationen noch nicht implementiert");
                        break;
                    case "oldobj":
                        useOldObjects = value.equals("true");
                        break;
                    case "filter":
                        filterName = value;
                        break;
                    case "onlycsv":
                        onlyCsv = value.equals("true");
                        break;
                    case "csvencoding":
                        encoding = value;
                        break;
                }
            }
            DataDescription dataDescription =
                new DataDescription(settingsData.getAttributeGroup(), settingsData.getAspect(), (short) settingsData.getSimulationVariant());

            ArchiveTimeSpecification archiveTimeSpecification = new ArchiveTimeSpecification(timingType, startRelative, intervalStart, intervalEnd);

            List<ArchiveDataSpecification> archiveDataSpecifications = new LinkedList<>();
            List<SystemObject> objects = settingsData.getObjects();
            boolean usePidQuery = useOldObjects;
            for (SystemObject systemObject : objects) {
                ArchiveDataSpecification ads =
                    new ArchiveDataSpecification(archiveTimeSpecification, archiveDataKindCombination, archiveOrder, archiveRequestOption,
                                                 dataDescription, systemObject);
                if (usePidQuery) {
                    try {
                        ads.setQueryWithPid();
                    } catch (NoSuchMethodError ignore) {
                        usePidQuery = false;
                    }
                }
                archiveDataSpecifications.add(ads);
            }
            if (useOldObjects && !usePidQuery) {
                _debug.warning("Archivanfrage kann historische Objekte nicht berücksichtigen, bitte DAF-Bibliothek aktualisieren.");
            }
            //noinspection ConstantConditions
            _filterAttributeGroup = new FilterAttributeGroup(settingsData.getAttributeGroup(), filterName);

            final ArchiveDataTableView dataTableView;   // Entweder wird die Tabelle initialisiert
            final File csvFile;                         // oder das CSV-File und der -Fortschrittsdialog.
            final CsvProgressDialogArchive progressDialog;
            if (!onlyCsv) {
                // Erzeugen des Ausgabefensters mit der Online-Tabelle
                dataTableView = new ArchiveDataTableView(settingsData, getConnection(), dataDescription);
                csvFile = null;
                progressDialog = null;
            } else {
                dataTableView = null;
                // Csv-Export vorbereiten
                csvFile = CsvUtils.getCSVFileForExport(_dialog);
                if (csvFile == null) {
                    // Wenn der Benutzer keine Csv-Datei gewählt hat;
                    // auf das Öffnen eines JOptionPanes wird hier bewusst verzichtet.
                    return;
                }
                progressDialog = new CsvProgressDialogArchive(csvFile.getName(), settingsData.getAttributeGroup(), settingsData.getAspect(),
                                                              settingsData.getSimulationVariant(), settingsData.getObjects());
            }

            ArchiveRequestManager archiveRequestManager = getConnection().getArchive(archive);

            // Anfrage starten
            final ArchiveDataQueryResult queryResult = archiveRequestManager.request(archiveQueryPriority, archiveDataSpecifications);
            Charset cs;
            if (Charset.isSupported(encoding)) {
                cs = Charset.forName(encoding);
            } else {
                cs = Charset.defaultCharset();
            }
            Thread archiveThread =
                new Thread(new ArchiveRequest(queryResult, dataTableView, timingType, csvFile, cs, progressDialog), "GTM-Archivanfrage");
            archiveThread.start();
        }

        /** Erstellt den Dialog. Bestandteil ist die Datenidentifikation, die Archivoptionen und die Darstellungsoptionen. */
        @SuppressWarnings("OverlyLongMethod")
        private void createDialog() {
            _dialog = new JDialog(_parent);
            _dialog.setTitle(getButtonText());
            _dialog.setResizable(false);

            Container pane = _dialog.getContentPane();
            pane.setLayout(new BoxLayout(pane, BoxLayout.Y_AXIS));

            // Datenidentifikationsauswahl-Panel
            final List<SystemObjectType> types = new LinkedList<>();
            DataModel configuration = getConnection().getDataModel();
            types.add(configuration.getType("typ.konfigurationsObjekt"));
            types.add(configuration.getType("typ.dynamischesObjekt"));
            _dataIdentificationChoice = new DataIdentificationChoice(null, types);
            pane.add(_dataIdentificationChoice);

            // Archivoptionen
            JPanel archivePanel = new JPanel();
            archivePanel.setBorder(BorderFactory.createTitledBorder("Archivoptionen"));
            archivePanel.setLayout(new BoxLayout(archivePanel, BoxLayout.Y_AXIS));

            JLabel archiveLabel = new JLabel("Archiv: ");
            archiveLabel.setLabelFor(_archiveBox);
            _archiveBox = new JComboBox<>();
            JPanel archiveBoxPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
            archiveBoxPanel.add(archiveLabel);
            archiveBoxPanel.add(Box.createHorizontalStrut(5));
            archiveBoxPanel.add(_archiveBox);
            archivePanel.add(archiveBoxPanel);

            // Priorität
            JPanel priorityPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
            JLabel priorityLabel = new JLabel("Priorität der Anfrage: ");
            _priorityComboBox = new JComboBox<>(_priorityUnit);
            priorityPanel.add(priorityLabel);
            priorityPanel.add(Box.createHorizontalStrut(5));
            priorityPanel.add(_priorityComboBox);
            archivePanel.add(priorityPanel);

            _domainPanel = new JPanel();
            _domainPanel.setLayout(new BoxLayout(_domainPanel, BoxLayout.Y_AXIS));
            _domainPanel.setBorder(BorderFactory.createTitledBorder("(Zeit-)Bereich der Anfrage"));

            _relativeBox = new JCheckBox("relative Angabe");
            _relativeBox.addItemListener(event -> {
                final JPanel domainPanel = createDomainPanel(_timingComboBox.getSelectedIndex(), event.getStateChange() == ItemEvent.SELECTED);
                _domainPanel.remove(1);
                _domainPanel.add(domainPanel);
                _dialog.repaint();
                _dialog.pack();
            });

            JPanel domainPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
            JLabel domainLabel = new JLabel("Bereich: ");
            _timingComboBox = new JComboBox<>(_timingUnit);
            _timingComboBox.setSelectedIndex(0);
            _timingComboBox.addItemListener(event -> {
                final JPanel domainPanel1 = createDomainPanel(_timingComboBox.getSelectedIndex(), _relativeBox.isSelected());
                _domainPanel.remove(1);
                _domainPanel.add(domainPanel1);
                _dialog.repaint();
                _dialog.pack();
            });

            domainPanel.add(domainLabel);
            domainPanel.add(_timingComboBox);
            domainPanel.add(Box.createHorizontalStrut(10));
            domainPanel.add(_relativeBox);

            // Bereich hinzufügen bzw. wegnehmen
            _domainPanel.add(domainPanel);
            _domainPanel.add(createDomainPanel(_timingComboBox.getSelectedIndex(), _relativeBox.isSelected()));
            archivePanel.add(_domainPanel);

            // Art der Archivanfrage
            JPanel archiveTypePanel = new JPanel();
            archiveTypePanel.setBorder(BorderFactory.createTitledBorder("Art der Archivanfrage"));
            archiveTypePanel.setLayout(new BoxLayout(archiveTypePanel, BoxLayout.Y_AXIS));

            _oaDataCheckBox = new JCheckBox("aktuelle Daten", true);
            _naDataCheckBox = new JCheckBox("nachgefordert-aktuelle Daten", false);
            _onDataCheckBox = new JCheckBox("nachgelieferte Daten", false);
            _nnDataCheckBox = new JCheckBox("nachgefordert-nachgelieferte Daten", false);
            final JPanel sortSequencePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
            final JLabel sortSequenceLabel = new JLabel("Sortierreihenfolge der nachgelieferten Daten: ");
            sortSequenceLabel.setEnabled(false);
            _sortSequenceComboBox = new JComboBox<>(_sortOfDataUnit);
            _sortSequenceComboBox.setEnabled(false);
            sortSequencePanel.add(sortSequenceLabel);
            sortSequencePanel.add(_sortSequenceComboBox);

            _useOldObjectsBox = new JCheckBox("Historische Objekte mit gleicher Pid berücksichtigen");

            _oaDataCheckBox.addItemListener(e -> {
                if (_oaDataCheckBox.isSelected()) {
                    // OK-Button enablen
                    _buttonBar.getAcceptButton().setEnabled(_isArchiveAvailable);
                } else {
                    // ist jetzt keiner mehr ausgewählt? -> OK-Button disablen
                    if (!_naDataCheckBox.isSelected() && !_onDataCheckBox.isSelected() && !_nnDataCheckBox.isSelected()) {
                        _buttonBar.getAcceptButton().setEnabled(false);
                    }
                }
            });
            _naDataCheckBox.addItemListener(e -> {
                if (_naDataCheckBox.isSelected()) {
                    _buttonBar.getAcceptButton().setEnabled(_isArchiveAvailable);
                } else {
                    if (!_oaDataCheckBox.isSelected() && !_onDataCheckBox.isSelected() && !_nnDataCheckBox.isSelected()) {
                        _buttonBar.getAcceptButton().setEnabled(false);
                    }
                }
            });

            _onDataCheckBox.addItemListener(event -> {
                if (_onDataCheckBox.isSelected()) {    // -> Sortierreihenfolge aktivieren
                    _buttonBar.getAcceptButton().setEnabled(_isArchiveAvailable);
                    sortSequenceLabel.setEnabled(true);
                    _sortSequenceComboBox.setEnabled(true);
                    sortSequencePanel.validate();
                } else {        // deaktivieren, wenn die andere Box (nachgefordert-nachgelieferte Daten) auch nicht selektiert ist
                    if (!_oaDataCheckBox.isSelected() && !_naDataCheckBox.isSelected() && !_nnDataCheckBox.isSelected()) {
                        _buttonBar.getAcceptButton().setEnabled(false);
                    }
                    if (!_nnDataCheckBox.isSelected()) {
                        sortSequenceLabel.setEnabled(false);
                        _sortSequenceComboBox.setEnabled(false);
                        sortSequencePanel.validate();
                    }
                }
            });

            _nnDataCheckBox.addItemListener(e -> {
                if (_nnDataCheckBox.isSelected()) {
                    _buttonBar.getAcceptButton().setEnabled(_isArchiveAvailable);
                    sortSequenceLabel.setEnabled(true);
                    _sortSequenceComboBox.setEnabled(true);
                    sortSequencePanel.validate();
                } else {
                    if (!_oaDataCheckBox.isSelected() && !_naDataCheckBox.isSelected() && !_onDataCheckBox.isSelected()) {
                        _buttonBar.getAcceptButton().setEnabled(false);
                    }
                    if (!_onDataCheckBox.isSelected()) {
                        sortSequenceLabel.setEnabled(false);
                        _sortSequenceComboBox.setEnabled(false);
                        sortSequencePanel.validate();
                    }
                }
            });
            JPanel gridPanel = new JPanel(new GridLayout(4, 1));
            gridPanel.add(_oaDataCheckBox);
            gridPanel.add(_onDataCheckBox);
            gridPanel.add(_naDataCheckBox);
            gridPanel.add(_nnDataCheckBox);

            archiveTypePanel.add(gridPanel);
            archiveTypePanel.add(sortSequencePanel);
            archivePanel.add(archiveTypePanel);

            // Zustands- oder Deltaanfrage
            JPanel requestPanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
            ButtonGroup requestButtonGroup = new ButtonGroup();
            _stateRadioButton = new JRadioButton("Zustandsanfrage", true);
            _deltaRadioButton = new JRadioButton("Deltaanfrage", false);
            requestButtonGroup.add(_stateRadioButton);
            requestButtonGroup.add(_deltaRadioButton);
            requestPanel.add(_stateRadioButton);
            requestPanel.add(_deltaRadioButton);
            archivePanel.add(requestPanel);

            // Historische Objekte berücksichtigen
            JPanel box = new JPanel(new FlowLayout(FlowLayout.LEFT));
            box.add(_useOldObjectsBox);
            archivePanel.add(box);

            // Sortierung der Darstellung
            JPanel displayOptionsPanel = new JPanel();
            displayOptionsPanel.setLayout(new BoxLayout(displayOptionsPanel, BoxLayout.Y_AXIS));
            displayOptionsPanel.setBorder(BorderFactory.createTitledBorder("Darstellungsoptionen"));

            JPanel firstLinePanel = new JPanel();
            firstLinePanel.setLayout(new BoxLayout(firstLinePanel, BoxLayout.X_AXIS));
            _onlyCsvCheckBox = new JCheckBox("Daten nur in CSV-Datei speichern");
            _onlyCsvCheckBox.setSelected(false);
            _onlyCsvCheckBox.addActionListener(e -> _encodingComboBox.setEnabled(_onlyCsvCheckBox.isSelected()));
            JLabel encodingLabel = new JLabel("Zeichenkodierung: ");
            _encodingComboBox = new JComboBox<>(_encodingOptions);
            _encodingComboBox.setEnabled(_onlyCsvCheckBox.isSelected());
            String defaultCharsetName = Charset.defaultCharset().displayName();
            if (defaultCharsetName.equals("UTF-8")) {
                _encodingComboBox.setSelectedIndex(1);
            } else {
                _encodingComboBox.setSelectedIndex(0);
            }
            firstLinePanel.add(_onlyCsvCheckBox);
            firstLinePanel.add(Box.createHorizontalStrut(15));
            firstLinePanel.add(encodingLabel);
            firstLinePanel.add(_encodingComboBox);

            JPanel secondLinePanel = new JPanel(new FlowLayout(FlowLayout.LEFT));
            JLabel viewSortLabel = new JLabel("Daten sortieren nach: ");
            _viewSortComboBox = new JComboBox<>(_sortViewUnit);
            _viewSortComboBox.setSelectedIndex(1);
            _viewSortComboBox.setEnabled(false);
            secondLinePanel.add(viewSortLabel);
            secondLinePanel.add(Box.createRigidArea(new Dimension(10, 0)));
            secondLinePanel.add(_viewSortComboBox);

            displayOptionsPanel.add(firstLinePanel);
            displayOptionsPanel.add(Box.createVerticalStrut(5));
            displayOptionsPanel.add(secondLinePanel);

            // Filterauswahl
            _filterPanel = new FilterPanel();
            _dataIdentificationChoice.addChoiceListener(_filterPanel);

            pane.add(archivePanel);
            pane.add(displayOptionsPanel);
            pane.add(_filterPanel);

            // untere Buttonleiste
            _buttonBar = new ButtonBar(this);     // brauche noch Übergabeparameter
            _dialog.getRootPane().setDefaultButton(_buttonBar.getAcceptButton());
            pane.add(_buttonBar);

            for (SystemObject object : getConnection().getDataModel().getType("typ.archiv").getElements()) {
                _archiveBox.addItem(object);
            }

            _archiveListener = new ArchiveListener(_buttonBar.getAcceptButton());

            _archiveBox.addActionListener(e -> {
                if (_currentArchiveRequestManager != null) {
                    // Alten Listener entfernen
                    _currentArchiveRequestManager.removeArchiveAvailabilityListener(_archiveListener);
                }
                _currentArchiveRequestManager = null;
                SystemObject selectedItem = (SystemObject) _archiveBox.getSelectedItem();
                if (selectedItem != null) {
                    _currentArchiveRequestManager = getConnection().getArchive(selectedItem);
                    _currentArchiveRequestManager.addArchiveAvailabilityListener(_archiveListener);
                }
                _archiveListener.archiveAvailabilityChanged(_currentArchiveRequestManager);
            });

            _archiveBox.setSelectedItem(getConnection().getDataModel().getConfigurationAuthority());

        }

        /**
         * Das Panel für den Bereich, der angefragt wird, wird erstellt und zurückgegeben.
         *
         * @param domain     Datenzeitstempel, Archivzeitstempel oder Datenindex
         * @param isRelative gibt an, ob es sich um eine relative Bereichsangabe handelt
         *
         * @return Panel für den anzufragenden Bereich
         */
        @SuppressWarnings("OverlyLongMethod")
        private JPanel createDomainPanel(int domain, boolean isRelative) {
            if (domain == 0 || domain == 1) {    // Zeitbereich
                final JPanel domainPanel = new JPanel();
                domainPanel.setLayout(new FlowLayout(FlowLayout.LEFT));
                domainPanel.setBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2));
                final JLabel startLabel;
                long time = System.currentTimeMillis();     // Werte für die Zeitbereiche minutengenau einstellen
                long temp = time % 60000;
                time = time - temp;
                if (isRelative) {                // relative Angabe
                    startLabel = new JLabel("Anzahl Datensätze: ");
                    if (_numberOfDatasetsSpinner == null) {
                        final SpinnerModel numberModel = new SpinnerNumberModel(0, 0, Integer.MAX_VALUE, 1);
                        _numberOfDatasetsSpinner = new JSpinner(numberModel);
                        _numberOfDatasetsSpinner.setPreferredSize(new Dimension(100, _numberOfDatasetsSpinner.getPreferredSize().height));
                    }
                } else {                        // absolute Bereichsangabe
                    startLabel = new JLabel("Von: ");
                    if (_startTimeSpinner == null) {
                        final SpinnerDateModel startTimeModel = new SpinnerDateModel();
                        startTimeModel.setValue(new Date(time));
                        _startTimeSpinner = new JSpinner(startTimeModel);
                        _startTimeSpinner.addChangeListener(e -> {
                            if (((Date) _startTimeSpinner.getModel().getValue()).getTime() >
                                ((Date) _endTimeSpinner.getModel().getValue()).getTime()) {
                                _endTimeSpinner.getModel().setValue(_startTimeSpinner.getModel().getValue());
                            }
                        });
                    }
                }
                final JLabel endLabel = new JLabel("Bis: ");
                if (_endTimeSpinner == null) {
                    final SpinnerDateModel endTimeModel = new SpinnerDateModel();
                    endTimeModel.setValue(new Date(time));
                    _endTimeSpinner = new JSpinner(endTimeModel);
                    _endTimeSpinner.addChangeListener(e -> {
                        if (((Date) _endTimeSpinner.getModel().getValue()).getTime() < ((Date) _startTimeSpinner.getModel().getValue()).getTime()) {
                            _startTimeSpinner.getModel().setValue(_endTimeSpinner.getModel().getValue());
                        }
                    });
                }
                domainPanel.add(startLabel);
                if (isRelative) {
                    domainPanel.add(_numberOfDatasetsSpinner);
                } else {
                    domainPanel.add(_startTimeSpinner);
                }
                domainPanel.add(Box.createHorizontalStrut(10));
                domainPanel.add(endLabel);
                domainPanel.add(_endTimeSpinner);
                return domainPanel;
            } else {                            // Datenindex
                final JPanel domainPanel = new JPanel();
                domainPanel.setLayout(new FlowLayout(FlowLayout.LEFT));
                domainPanel.setBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2));
                final JLabel startLabel;
                if (isRelative) {               // relative Angabe
                    startLabel = new JLabel("Anzahl Datensätze: ");
                    if (_numberOfDatasetsSpinner == null) {
                        final SpinnerModel numberModel = new SpinnerNumberModel(0, 0, Integer.MAX_VALUE, 1);
                        _numberOfDatasetsSpinner = new JSpinner(numberModel);
                        _numberOfDatasetsSpinner.setPreferredSize(new Dimension(100, _numberOfDatasetsSpinner.getPreferredSize().height));
                    }
                } else {                        // absolute Bereichsangabe
                    startLabel = new JLabel("Von: ");
                    if (_startIndexSpinner == null) {
                        SpinnerModel startIndexModel = new SpinnerLongModel(0, 0, Long.MAX_VALUE, 1);
                        _startIndexSpinner = new JSpinner(startIndexModel);
                        _startIndexSpinner.setPreferredSize(new Dimension(180, _startIndexSpinner.getPreferredSize().height));
                        _startIndexSpinner.addChangeListener(e -> {
                            if ((Long) _startIndexSpinner.getModel().getValue() > (Long) _endIndexSpinner.getModel().getValue()) {
                                _endIndexSpinner.getModel().setValue(_startIndexSpinner.getModel().getValue());
                            }
                        });
                    }
                }
                final JLabel endLabel = new JLabel("Bis: ");
                if (_endIndexSpinner == null) {
                    SpinnerModel endIndexModel = new SpinnerLongModel(0, 0, Long.MAX_VALUE, 1);
                    _endIndexSpinner = new JSpinner(endIndexModel);
                    _endIndexSpinner.setPreferredSize(new Dimension(180, _endIndexSpinner.getPreferredSize().height));
                    _endIndexSpinner.addChangeListener(e -> {
                        if ((Long) _endIndexSpinner.getModel().getValue() < (Long) _startIndexSpinner.getModel().getValue()) {
                            _startIndexSpinner.getModel().setValue(_endIndexSpinner.getModel().getValue());
                        }
                    });
                }
                domainPanel.add(startLabel);
                if (isRelative) {
                    domainPanel.add(_numberOfDatasetsSpinner);
                } else {
                    domainPanel.add(_startIndexSpinner);
                }
                domainPanel.add(Box.createHorizontalStrut(10));
                domainPanel.add(endLabel);
                domainPanel.add(_endIndexSpinner);
                return domainPanel;
            }
        }

        /** Durch diese Methode wird der Dialog angezeigt. */
        private void showDialog() {
            _dialog.setLocation(50, 50);
            _dialog.pack();
            _dialog.setVisible(true);
        }

        /**
         * Gibt die Priorität der Archivanfrage zurück.
         *
         * @return die Priorität ("Hoch", "Mittel", "Niedrig")
         */
        private String getPriority() {
            return (String) _priorityComboBox.getSelectedItem();
        }

        /**
         * Setzt die Priorität der Archivanfrage im Dialog.
         *
         * @param priority die Priorität ("Hoch", "Mittel", "Niedrig")
         */
        private void setPriority(String priority) {
            _priorityComboBox.setSelectedItem(priority);
        }

        /**
         * Gibt den (Zeit-)Bereich der Archivanfrage zurück.
         *
         * @return der (Zeit-)Bereich ("Datenzeitstempel", "Archivzeitstempel", "Datenindex")
         */
        private String getTimingType() {
            return (String) _timingComboBox.getSelectedItem();
        }

        /**
         * Setzt den (Zeit-)Bereich der Archivanfrage im Dialog.
         *
         * @param timingType der (Zeit-)Bereich ("Datenzeitstempel", "Archivzeitstempel", "Datenindex")
         */
        private void setTimingType(String timingType) {
            _timingComboBox.setSelectedItem(timingType);
        }

        /**
         * Gibt zurück, ob die Zeit- / Index-Angabe relativ oder absolut gemacht wurde. Wobei relativ bedeutet, dass der eine Wert die Anzahl der
         * Datensätze bestimmt, die vor dem zweiten Wert liegen.
         *
         * @return ob relativ oder absolut
         */
        private String getRelative() {
            return String.valueOf(_relativeBox.isSelected());
        }

        /**
         * Das Flag des Moduls wird gesetzt, wo unterschieden werden kann, ob die Zeit-/Index-Angabe relativ oder absolut ist.
         *
         * @param relative gibt an, ob die Zeit-/Index-Angabe relativ oder absolut ist
         */
        private void setRelative(String relative) {
            _relativeBox.setSelected(relative.equals("true"));
        }

        /**
         * Gibt den Startpunkt des (Zeit-)Bereichs zurück.
         *
         * @return Startpunkt des (Zeit-)Bereichs
         */
        private String getFrom() {
            if (_relativeBox.isSelected()) {    // relative Angabe
                Integer i = (Integer) _numberOfDatasetsSpinner.getModel().getValue();
                return String.valueOf(i);
            } else {            // absolute Angabe
                if (_timingComboBox.getSelectedIndex() == 0 || _timingComboBox.getSelectedIndex() == 1) {    // Zeitbereich
                    Date date = (Date) _startTimeSpinner.getModel().getValue();
                    return String.valueOf(date.getTime());
                } else {    // Datenindex
                    Long l = (Long) _startIndexSpinner.getModel().getValue();
                    return String.valueOf(l.longValue());
                }
            }
        }

        /**
         * Setzt den Startpunkt des (Zeit-)Bereichs im Dialog.
         *
         * @param timing   der (Zeit-)Bereich ("Datenzeitstempel", "Archivzeitstempel", "Datenindex")
         * @param from     der Startpunkt
         * @param relative ob die Parameter relativ sind
         */
        private void setFrom(String timing, String from, String relative) {
            if (relative.equals("true")) {
                Integer i = Integer.valueOf(from);
                _numberOfDatasetsSpinner.getModel().setValue(i);
            } else {
                if (timing.equals(_timingUnit[0]) || timing.equals(_timingUnit[1])) {    // Zeitbereich
                    Date date = new Date(Long.parseLong(from));
                    _startTimeSpinner.getModel().setValue(date);
                } else {
                    _startIndexSpinner.getModel().setValue(Long.parseLong(from));
                }
            }
        }

        /**
         * Gibt den Endpunkt des (Zeit-)Bereichs zurück.
         *
         * @return Endpunkt des Zeitbereichs oder Anzahl der Datensätze, die ausgegeben werden sollen
         */
        private String getTo() {
            if (_timingComboBox.getSelectedIndex() == 0 || _timingComboBox.getSelectedIndex() == 1) {    // Zeitbereich
                Date date = (Date) _endTimeSpinner.getModel().getValue();
                return String.valueOf(date.getTime());
            } else {
                Long l = (Long) _endIndexSpinner.getModel().getValue();
                return String.valueOf(l.longValue());
            }
        }

        /**
         * Setzt den Wert für das Feld "Bis:" bzw. "Anzahl vor dem Index:", je nachdem welche Timingangabe angegeben wird.
         *
         * @param timing "Datenzeitstempel", "Archivzeitstempel" oder "Datenindex"
         * @param to     entweder ein Zeitstempel oder die Anzahl Datensätze vor dem Index
         */
        private void setTo(String timing, String to) {
            if (timing.equals(_timingUnit[0]) || timing.equals(_timingUnit[1])) {    // Zeitbereich
                Date date = new Date(Long.parseLong(to));
                _endTimeSpinner.getModel().setValue(date);
            } else {    // Datenindex
                _endIndexSpinner.getModel().setValue(Long.valueOf(to));
            }
        }

        /**
         * Gibt zurück, welche Arten der Archivanfrage ausgewählt wurden.
         *
         * @return z.B. "oa on nn"
         */
        private String getArchiveType() {
            String result = "";
            if (_oaDataCheckBox.isSelected()) {
                result += "oa ";    // aktuelle Daten
            }
            if (_onDataCheckBox.isSelected()) {
                result += "on ";    // nachgelieferte Daten
            }
            if (_naDataCheckBox.isSelected()) {
                result += "na ";    // nachgefordert-aktuelle Daten
            }
            if (_nnDataCheckBox.isSelected()) {
                result += "nn ";    // nachgefordert-nachgelieferte Daten
            }
            result = result.substring(0, result.length() - 1);
            return result;
        }

        /**
         * Setzt die Checkboxen, welche Art der Archivanfrage gesetzt sein soll.
         *
         * @param archiveType z.B. "oa on nn";
         */
        private void setArchiveType(String archiveType) {
            String[] types = archiveType.split(" ");
            _oaDataCheckBox.setSelected(false);        // alle deselektieren
            for (String type : types) {
                switch (type) {
                    case "oa":
                        _oaDataCheckBox.setSelected(true);
                        break;
                    case "na":
                        _naDataCheckBox.setSelected(true);
                        break;
                    case "on":
                        _onDataCheckBox.setSelected(true);
                        break;
                    case "nn":
                        _nnDataCheckBox.setSelected(true);
                        break;
                }
            }
        }

        /**
         * Gibt zurück, wie die nachgelieferten Archivdatensätze einsortiert werden sollen.
         *
         * @return "Datenindex" oder "Datenzeitstempel"
         */
        private String getSortSequence() {
            return (String) _sortSequenceComboBox.getSelectedItem();
        }

        /**
         * Setzt den Parameter für die Sortierreihenfolge der als nachgeliefert gekennzeichneten Archivdatensätze.
         *
         * @param sortSequence Sortierreihenfolge der nachgeliegerten Archivdatensätze
         */
        private void setSortSequence(String sortSequence) {
            _sortSequenceComboBox.setSelectedItem(sortSequence);
        }

        /**
         * Gibt zurück, ob es sich um eine Zustands- oder Deltaanfrage handelt.
         *
         * @return "state" oder "delta"
         */
        private String getRequestView() {
            if (_stateRadioButton.isSelected()) {
                return "state";
            } else {
                return "delta";
            }
        }

        /**
         * Setzt den Parameter, ob es sich um eine Zustands- oder Deltaanfrage handelt.
         *
         * @param requestView Parameter: "state" oder "delta"
         */
        private void setRequestView(String requestView) {
            if (requestView.equals("state")) {
                _stateRadioButton.setSelected(true);
                _deltaRadioButton.setSelected(false);
            } else {
                _stateRadioButton.setSelected(false);
                _deltaRadioButton.setSelected(true);
            }
        }

        /**
         * Parameter wird abgefragt, wie die Daten in der Tabelle angezeigt werden sollen, sortiert nach der Zeit oder nach der Datenidentifikation.
         *
         * @return "Zeitstempel" oder "Datenidentifikation"
         */
        private String getViewSort() {
            return (String) _viewSortComboBox.getSelectedItem();
        }

        /**
         * Parameter wird gesetzt, wie die Daten in der Tabelle angezeigt werden sollen, sortiert nach der Zeit oder nach der Datenidentifikation.
         *
         * @param viewSort Parameter: "Zeitstempel" oder "Datenidentifikation"
         */
        private void setViewSort(String viewSort) {
            _viewSortComboBox.setSelectedItem(viewSort);
        }

        /**
         * Gibt zurück, historische Objekte mit gleicher Pid berücksichtigt werden sollen
         *
         * @return
         */
        private String getUseOldObjects() {
            return String.valueOf(_useOldObjectsBox.isSelected());
        }

        /**
         * Das Flag des Moduls wird gesetzt, wenn historische Objekte mit gleicher Pid berücksichtigt werden sollen
         *
         * @param relative
         */
        private void setUseOldObjects(String relative) {
            _useOldObjectsBox.setSelected(relative.equals("true"));
        }

        /**
         * Erstellt die Einstellungsdaten.
         *
         * @param title der Name für die Einstellungen
         *
         * @return die Einstellungsdaten
         */
        private SettingsData getSettings(String title) {
            Class<?> moduleClass = StreamBasedArchiveRequestModule.class;
            List<SystemObjectType> objectTypes = _dataIdentificationChoice.getObjectTypes();
            AttributeGroup atg = _dataIdentificationChoice.getAttributeGroup();
            Aspect asp = _dataIdentificationChoice.getAspect();
            List<SystemObject> objects = _dataIdentificationChoice.getObjects();

            SettingsData settingsData = new SettingsData(getModuleName(), moduleClass, objectTypes, atg, asp, objects);
            settingsData.setTitle(title);
            settingsData.setSimulationVariant(_dataIdentificationChoice.getSimulationVariant());
            settingsData.setTreePath(_dataIdentificationChoice.getTreePath());
            settingsData.setSettingsMap(getSettingsMap());

            return settingsData;
        }

        /**
         * Sammelt alle Parameter des Dialogs.
         *
         * @return Map aller Parameter des Dialogs
         */
        private Map<String, String> getSettingsMap() {
            Map<String, String> map = new HashMap<>();
            map.put("priority", getPriority());
            map.put("timing", getTimingType());
            map.put("relative", getRelative());
            map.put("from", getFrom());
            map.put("to", getTo());
            map.put("archivetype", getArchiveType());
            if (_onDataCheckBox.isSelected() || _nnDataCheckBox.isSelected()) {
                map.put("sortsequence", getSortSequence());
            }
            map.put("requestview", getRequestView());
            map.put("viewsort", getViewSort());
            map.put("oldobj", getUseOldObjects());
            map.put("archivePid", getArchivePid());
            map.put("onlycsv", getOnlyCsv());
            map.put("csvencoding", (String) _encodingComboBox.getSelectedItem());
            map.put("filter", _filterPanel.getFilterName());

            return map;
        }

        private String getArchivePid() {
            return ((SystemObject) _archiveBox.getSelectedItem()).getPid();
        }

        private void setArchivePid(final String value) {
            _archiveBox.setSelectedItem(null);
            _archiveBox.setSelectedItem(getConnection().getDataModel().getObject(value));
        }

        private String getOnlyCsv() {
            return Boolean.toString(_onlyCsvCheckBox.isSelected());
        }

        private void setOnlyCsv(final String value) {
            _onlyCsvCheckBox.setSelected(value.equals("true"));
            _encodingComboBox.setEnabled(value.equals("true"));
        }

        /**
         * Durch betätigen des "OK"-Buttons wird die Archivanfrage mit den eingestellten Parametern in einem neuen Fenster gestartet und dieser Dialog
         * wird geschlossen. Die Parameter werden gespeichert.
         */
        @Override
        public void doOK() {
            SettingsData settingsData = getSettings("");

            Map<String, String> settingsMap = settingsData.getSettingsMap();
            String timingType = settingsMap.get("timing");
            Integer factor = null;
            if (timingType != null) {
                switch (timingType) {
                    case "Datenzeitstempel":
                    case "Archivzeitstempel":
                        String startTimeString = settingsMap.get("from");
                        String endTimeString = settingsMap.get("to");
                        if (startTimeString != null && endTimeString != null) {
                            Long startTime = Long.parseLong(startTimeString) / 1000;    // Millis zu Sekunden
                            Long endTime = Long.parseLong(endTimeString) / 1000;        // Millis zu Sekunden
                            Long factorAsLong = Math.abs(endTime - startTime) / 60;     // in Minuten
                            factor = factorAsLong.intValue();
                        }
                        break;
                    case "Datenindex":
                        break;
                }
            }
            if (factor == null) {
                factor = 1;
            }

            if (!_relativeBox.isSelected() && settingsData.getObjects().size() * factor > _warningLimit) {
                String message;
                if (factor != 1) {
                    message = "Umfangreiche Aktion: es werden Daten zu " + settingsData.getObjects().size() + " Objekten für " + factor +
                              " Minuten angefordert.";
                } else {
                    message = "Umfangreiche Aktion: es werden Daten zu " + settingsData.getObjects().size() + " Objekten angefordert.";
                }
                Object[] objects = {"Durchführen", "Abbrechen"};
                int n = JOptionPane
                    .showOptionDialog(_dialog, message, "Warnung", JOptionPane.OK_CANCEL_OPTION, JOptionPane.WARNING_MESSAGE, null, objects,
                                      objects[1]);
                if (n != 0) {    // n kann 0,1 und -1 sein!
                    return;
                }
            }
            try {
                startRequest(settingsData);
                doCancel();
                saveSettings(settingsData);
            } catch (Exception ex) {
                String message = ex.getMessage();
                _debug.error("Aufgrund einer unerwarteten Ausnahme konnte die gewünschte Archivanfrage nicht gestellt werden (siehe exception)", ex);
                JOptionPane.showMessageDialog(_dialog, message, "Archivanfrage kann nicht gestellt werden.", JOptionPane.ERROR_MESSAGE);
            }
        }

        /** Durch betätigen des "Abbrechen"-Buttons wird der Dialog geschlossen. */
        @Override
        public void doCancel() {
            _dialog.setVisible(false);
            _dialog.dispose();
        }

        /**
         * Durch betätigen des "Speichern unter ..."-Buttons werden die Einstellungen gespeichert.
         *
         * @param title Titel der Einstellungen
         */
        @Override
        public void doSave(String title) {
            SettingsData settingsData = getSettings(title);
            saveSettings(settingsData);
        }

        /**
         * Durch Betätigen des "Hilfe"-Buttons wird die Kontexthilfe geöffnet.
         */
        @Override
        public void openHelp() {
            GtmHelp.openHelp("#Archivanfrage");
        }

        /** Diese Klasse verarbeitet die erhaltenen Streams einer Archivanfrage und stellt die Datensätze in einer Online-Tabelle dar. */
        private final class ArchiveRequest implements Runnable, CsvProgressDialogArchive.StopRequestListener {

            private final ArchiveDataQueryResult _queryResult;
            private final ArchiveDataTableView _dataTableView;
            private final TimingType _timingType;
            private final File _csvFile;
            private final Charset _charset;
            private final CsvProgressDialogArchive _progressDialog;
            private ArchiveDataStream[] _archiveDataStreams;
            private OutputStreamWriter _fileWriter;

            private PerpetualCsvConverter _csvConverter;

            private boolean _stopRequested;

            private int _numberOfHeaderLines;

            private int _numberOfDataTableObjects;

            public ArchiveRequest(ArchiveDataQueryResult queryResult, @Nullable ArchiveDataTableView dataTableView, TimingType timingType,
                                  @Nullable File csvFile, Charset charset, @Nullable CsvProgressDialogArchive progressDialog) {
                _queryResult = queryResult;
                _dataTableView = dataTableView;
                _timingType = timingType;
                _csvFile = csvFile;
                _charset = charset;
                _progressDialog = progressDialog;
                _stopRequested = false;
                if (_progressDialog != null) {
                    _progressDialog.addStopRequestListener(this);
                }
                _numberOfHeaderLines = 0;
                _numberOfDataTableObjects = 0;
            }

            @SuppressWarnings({"OverlyLongMethod", "OverlyNestedMethod"})
            @Override
            public void run() {
                _numberOfHeaderLines = 0;
                _numberOfDataTableObjects = 0;
                try {
                    if (_queryResult.isRequestSuccessful()) {
                        _debug.info("Archivanfrage konnte erfolgreich bearbeitet werden.");
                        _archiveDataStreams = _queryResult.getStreams();
                    } else {
                        _debug.warning("Eine Archivanfrage konnte nicht bearbeitet werden, Fehler: " + _queryResult.getErrorMessage());
                        SwingUtilities.invokeLater(() -> JOptionPane
                            .showMessageDialog(_dialog, "Die Archivanfrage konnte nicht bearbeitet werden.", "Archivanfrage fehlerhaft",
                                               JOptionPane.ERROR_MESSAGE));
                        return;
                    }
                } catch (InterruptedException ex) {
                    _debug.warning("Die Übertragung der Archivdaten wurde aufgrund eines Übertragungsfehlers unterbrochen (siehe exception)", ex);
                    SwingUtilities.invokeLater(() -> JOptionPane
                        .showMessageDialog(_dialog, "Übertragung der Archivdaten wurde unterbrochen.", "Übertragungsfehler",
                                           JOptionPane.ERROR_MESSAGE));
                    return;
                }

                if (_csvFile != null) {
                    try {
                        _fileWriter = new OutputStreamWriter(new FileOutputStream(_csvFile), _charset);
                    } catch (FileNotFoundException e) {
                        JOptionPane
                            .showMessageDialog(_dialog, "Die Datei " + _csvFile.getAbsolutePath() + " konnte nicht zum Schreiben geöffnet werden.",
                                               "Datei nicht gefunden", JOptionPane.ERROR_MESSAGE);
                    }
                    _csvConverter = new PerpetualCsvConverter(_filterAttributeGroup);
                    _csvConverter.setDelimiter(";");
                    if (_progressDialog != null) {
                        SwingUtilities.invokeLater(() -> _progressDialog.setVisible(true));
                    }
                }

                // Dieses Lock sorgt dafür, dass die beiden Threads abwechselnd zugreifen.
                // Swing nimmt sich einen Datensatz (solange ist take gesperrt), dann darf take wieder einen Datensatz ablegen
                final Semaphore lock = new Semaphore(1);
                int archiveStreamCounter = 0;
                for (final ArchiveDataStream _archiveDataStream : _archiveDataStreams) {
                    final int archiveStreamCounterFinal = archiveStreamCounter;
                    try {
                        // Es soll ein take ausgeführt werden, also muss ein Lock angefordert werden
                        lock.acquire();
                        ArchiveData archiveData = _archiveDataStream.take();
                        int archiveDataCounter = 0;
                        while (archiveData != null && !_stopRequested) {
                            final int archiveDataCounterFinal = archiveDataCounter;
                            if (_dataTableView != null && _dataTableView.isDisposed()) {
                                _debug.finer("Archivanfrage wurde abgebrochen", _archiveDataStream.getDataSpecification().toString());
                                _archiveDataStream.abort();
                                break;
                            } else {
                                // Datensatz erzeugen:
                                final ArchiveData helper = archiveData; // STH: Es wird eine final Variable benötigt.
                                Runnable runner = () -> {
                                    DataTableObject dataTableObject =
                                        new DataTableObject(helper.getObject(), _timingType, helper.getArchiveTime(), helper.getDataTime(),
                                                            helper.getDataIndex(), helper.getDataType(), helper.getDataKind(), helper.getData(),
                                                            _filterAttributeGroup);
                                    // Datensatz weiterreichen:
                                    if (_dataTableView != null) {
                                        _dataTableView.addDataset(dataTableObject);
                                    } else {
                                        List<DataTableObject> dataTableObjects = new ArrayList<>(1);
                                        dataTableObjects.add(dataTableObject);
                                        ++_numberOfDataTableObjects;
                                        boolean columnStructureChanged = _csvConverter.setData(dataTableObjects, null);
                                        try {
                                            if (columnStructureChanged) {
                                                _fileWriter.write(_csvConverter.getCsvHeaderLine(true));
                                                // Wird dies mit false aufgerufen, so muss das PostProcessing geändert werden.
                                                ++_numberOfHeaderLines;
                                            }
                                            _fileWriter.write(_csvConverter.getCsvLines(true));
                                        } catch (IOException ignore) {
                                            _debug.finer("CSV-Export wurde abgebrochen", _archiveDataStream.getDataSpecification().toString());
                                            _archiveDataStream.abort();
                                        }
                                        if (_progressDialog != null && helper.getDataType() != DataState.END_OF_ARCHIVE) {
                                            SwingUtilities.invokeLater(new Runnable() {
                                                @Override
                                                public void run() {
                                                    _progressDialog.setProgress(
                                                        getProgress(helper, _archiveDataStream.getDataSpecification().getTimeSpec(),
                                                                    archiveStreamCounterFinal, archiveDataCounterFinal), _numberOfDataTableObjects);
                                                }
                                            });
                                        }
                                    }
                                    lock.release();
                                };
                                EventQueue.invokeLater(runner);

                                // Es sollen Datensätze angefordert werden, dies darf nur geschehen, wenn der Swingthread
                                // fertig ist
                                lock.acquire();
                                // nächsten Datensatz vom Archivsystem holen
                                archiveData = _archiveDataStream.take();
                            }
                            ++archiveDataCounter;
                        }
                    } catch (InterruptedException ex) {
                        _debug.error("Die Übertragung der Archivdaten wurde aufgrund eines Übertragungsfehlers unterbrochen (siehe exception)", ex);
                        JOptionPane.showMessageDialog(_dialog, "Übertragung der Archivdaten wurde unterbrochen.", "Übertragungsfehler",
                                                      JOptionPane.ERROR_MESSAGE);
                    } catch (IOException ex) {
                        _debug.error("Übertragungsfehler zum Datenverteiler oder zum Archiv (siehe exception)", ex);
                        JOptionPane.showMessageDialog(_dialog, "Problem mit Datenverteiler oder Archivsystem.", "Übertragungsfehler",
                                                      JOptionPane.ERROR_MESSAGE);
                    } finally {
                        lock.release();
                        _debug.info("Keine weiteren Archivdatensätze für diese Datenidentifikation.");
                    }
                    ++archiveStreamCounter;
                }
                try {
                    if (_fileWriter != null) {
                        _fileWriter.flush();
                        _fileWriter.close();
                    }
                } catch (IOException e) {
                    _debug.error("Fehler beim Beenden des CSV-Exports.", e);
                    SwingUtilities.invokeLater(new Runnable() {
                        @Override
                        public void run() {
                            JOptionPane.showMessageDialog(_dialog, "Der CSV-Export ist möglicherweise unvollständig.", "Fehler beim CSV-Export.",
                                                          JOptionPane.ERROR_MESSAGE);
                        }
                    });
                }
                _debug.info("Alle Datenidentifikationen abgearbeitet. Keine weiteren Daten vorhanden.");

                if (_csvFile != null && _progressDialog != null) {
                    if (!_stopRequested && _numberOfHeaderLines > 1) {   // Bedingung für eine Nachbehandlung
                        try {
                            SwingUtilities.invokeAndWait(new Runnable() {
                                @Override
                                public void run() {
                                    _progressDialog.addPostProcessingLayout();
                                }
                            });
                        } catch (InterruptedException | InvocationTargetException e) {
                            _debug.error("Das Ändern des Layouts des Nachbearbeitungsdialogs schlug fehl.", e);
                            _progressDialog.dispose();
                            return;
                        }

                        PostProcessor postProcessor = new PostProcessor(_csvFile, _csvConverter, _progressDialog, _charset.displayName());
                        postProcessor.execute();    // Hier wird das Schließen des Dialogs übernommen.
                    } else {
                        _progressDialog.dispose();
                    }
                } else if (_progressDialog != null) {
                    _progressDialog.dispose();
                }
            }

            @Override
            public void stopRequested() {
                _stopRequested = true;
            }

            public int getProgress(final ArchiveData archiveData, final ArchiveTimeSpecification timeSpecification, final int archiveStreamCounter,
                                   final int archiveDataCounter) {
                Double progress = (double) archiveStreamCounter / _archiveDataStreams.length;
                if (timeSpecification.isStartRelative()) {
                    long start = timeSpecification.getIntervalStart();
                    progress += Math.min((double) (archiveDataCounter + 1) / start, 1.) / _archiveDataStreams.length;
                } else {
                    long start = timeSpecification.getIntervalStart();
                    long end = timeSpecification.getIntervalEnd();
                    if (end - start > 0) {
                        long value = 0;
                        TimingType timingType = timeSpecification.getTimingType();
                        if (timingType == TimingType.ARCHIVE_TIME) {
                            value = archiveData.getArchiveTime();
                        } else if (timingType == TimingType.DATA_INDEX) {
                            value = archiveData.getDataIndex();
                        } else if (timingType == TimingType.DATA_TIME) {
                            value = archiveData.getDataTime();
                        }
                        if (value > start) {
                            progress += (double) (value - start) / ((end - start) * _archiveDataStreams.length);
                        }
                    }
                }
                progress *= 100.;
                return progress.intValue();
            }
        }

        private class ArchiveListener implements ArchiveAvailabilityListener {

            private final JButton _okButton;

            public ArchiveListener(JButton okButton) {
                _okButton = okButton;
            }

            @Override
            public void archiveAvailabilityChanged(ArchiveRequestManager archive) {
                if (_currentArchiveRequestManager != null && _currentArchiveRequestManager.isArchiveAvailable()) {
                    _okButton.setEnabled(true);
                    _okButton.setToolTipText(null);
                    _isArchiveAvailable = true;
                } else {
                    _okButton.setEnabled(false);
                    _okButton.setToolTipText("Das Archivsystem ist nicht verfügbar.");
                    _isArchiveAvailable = false;
                }
            }
        }
    }
}
