/*
 * Copyright 2011-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.pat.sysbed.plugins.subscriptions;

import de.bsvrz.dav.daf.main.ClientConnectionState;
import de.bsvrz.dav.daf.main.ClientDavConnection;
import de.bsvrz.dav.daf.main.ClientSubscriptionInfo;
import de.bsvrz.dav.daf.main.SubscriptionState;
import de.bsvrz.dav.daf.main.config.AttributeGroupUsage;
import de.bsvrz.dav.daf.main.config.DavApplication;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.kappich.selectionlist.SelectionList;
import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Dimension;
import java.awt.Font;
import java.io.IOException;
import java.util.Collections;
import java.util.List;
import javax.swing.BorderFactory;
import javax.swing.DefaultListCellRenderer;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.SwingConstants;
import javax.swing.border.TitledBorder;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

/**
 * @author Kappich Systemberatung
 * @version $Revision: 11467 $
 */
public class SubscriptionsPanel extends JPanel {

    private final JSplitPane _jSplitPaneH;

    private final JSplitPane _jSplitPaneV;

    private final JLabel _label;

    private final SelectionList<ClientSubscriptionInfo.ClientSendingSubscription> _senderList;

    private final SelectionList<ClientSubscriptionInfo.ClientReceivingSubscription> _receiverList;

    private final SelectionList<ClientSubscriptionInfo.DavInformation> _potDavList;

    public SubscriptionsPanel(final ClientDavConnection connection, final SystemObject object, final AttributeGroupUsage usage,
                              final short simulationVariant, final DavApplication dav) {
        super(new BorderLayout());
        _jSplitPaneH = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
        _jSplitPaneV = new JSplitPane(JSplitPane.VERTICAL_SPLIT);
        ClientSubscriptionInfo subscriptionInfo;
        try {
            subscriptionInfo = connection.getSubscriptionInfo(dav, object, usage, simulationVariant);
        } catch (IOException e) {
            subscriptionInfo = null;
            e.printStackTrace();
            JOptionPane.showMessageDialog(this, "Konnte die Anmeldungen nicht auflisten. " + e.getMessage());
        }
        final TitledBorder labelBorder = BorderFactory.createTitledBorder("Details");

        final JComponent paneSend = new JPanel(new BorderLayout());
        _senderList = new SelectionList<>("Sende-Anmeldungen", "Sende-Anmeldungen",
                                          (pattern, id, o) -> (pattern != null && pattern.matcher(o.toString()).find()) ||
                                                              o.getApplicationId() == id || o.getUserId() == id,
                                          SelectionList.FilterStyle.HiddenFilter, null);
        _senderList.setElements(subscriptionInfo == null ? Collections.emptyList() : subscriptionInfo.getSenderSubscriptions());
        _senderList.setListRenderer(IconRenderer.getInstance());
        paneSend.add(_senderList, BorderLayout.CENTER);

        final JComponent paneReceive = new JPanel(new BorderLayout());
        _receiverList = new SelectionList<>("Empfangs-Anmeldungen", "Empfangs-Anmeldungen",
                                            (pattern, id, o) -> (pattern != null && pattern.matcher(o.toString()).find()) ||
                                                                o.getApplicationId() == id || o.getUserId() == id,
                                            SelectionList.FilterStyle.HiddenFilter, null);
        _receiverList.setElements(subscriptionInfo == null ? Collections.emptyList() : subscriptionInfo.getReceiverSubscriptions());
        _receiverList.setListRenderer(IconRenderer.getInstance());
        paneReceive.add(_receiverList, BorderLayout.CENTER);

        final JComponent panePotDav = new JPanel(new BorderLayout());
        _potDavList = new SelectionList<>("Potenzielle Zentraldatenverteiler", "Potenzielle Zentraldatenverteiler",
                                          (pattern, id, o) -> (pattern != null && pattern.matcher(o.toString()).find()) ||
                                                              o.getCentralDavId() == id || o.getConnectionDavId() == id || o.getUserId() == id,
                                          SelectionList.FilterStyle.HiddenFilter, null);
        _potDavList.setElements(subscriptionInfo == null ? Collections.emptyList() : subscriptionInfo.getPotentialCentralDavs());
        _potDavList.setListRenderer(IconRenderer.getInstance());
        panePotDav.add(_potDavList, BorderLayout.CENTER);

        _jSplitPaneH.setLeftComponent(paneSend);
        _jSplitPaneH.setRightComponent(paneReceive);
        _jSplitPaneH.setResizeWeight(0.5);
        _jSplitPaneV.setTopComponent(_jSplitPaneH);
        _jSplitPaneV.setBottomComponent(panePotDav);
        _jSplitPaneV.setResizeWeight(0.5);
        _senderList.addListSelectionListener(new MySelectionListener<>(_senderList));
        _receiverList.addListSelectionListener(new MySelectionListener<>(_receiverList));
        _potDavList.addListSelectionListener(new MySelectionListener<>(_potDavList));
        _senderList.setFocusable(false);
        _receiverList.setFocusable(false);
        this.add(_jSplitPaneV, BorderLayout.CENTER);
        _label = new JLabel();
        _label.setFont(_label.getFont().deriveFont(Font.PLAIN));
        _label.setBorder(labelBorder);
        _label.setVerticalAlignment(SwingConstants.TOP);
        final JScrollPane pane = new JScrollPane(_label);
        pane.setBorder(BorderFactory.createEmptyBorder());
        pane.setPreferredSize(new Dimension(400, 130));
        this.add(pane, BorderLayout.SOUTH);
    }

    public static String stateToString(final SubscriptionState state) {
	    return switch (state) {
		    case NoSendersAvailable -> "Kein Sender";
		    case NotAllowed -> "Keine Rechte";
		    case InvalidSubscription -> "Ungültige Anmeldung";
		    case ReceiversAvailable -> "Bereit zum Senden";
		    case NoReceiversAvailable -> "Kein Empfänger";
		    case SendersAvailable -> "Bereit zum Empfangen";
		    case Waiting -> "Warte auf andere Datenverteiler";
		    case NotResponsible -> "Nicht zuständig";
		    case MultiRemoteLock -> "Ungültige Anmeldung, Mehrere Zentraldatenverteiler";
	    };
    }

    public static String stateToString(final ClientConnectionState state) {
	    return switch (state) {
		    case FromLocalOk -> "Lokale Anmeldung";
		    case FromRemoteOk -> "Eingehende Anmeldung";
		    case ToRemoteWaiting -> "Warte auf Antwort";
		    case ToRemoteOk -> "Ausgehende Anmeldung: Erfolgreich";
		    case ToRemoteNotResponsible -> "Ausgehende Anmeldung: Nicht verantwortlich";
		    case ToRemoteNotAllowed -> "Ausgehende Anmeldung: Keine Rechte";
		    case ToRemoteMultiple -> "Ausgehende Anmeldung: Mehrere Zentraldatenverteiler";
	    };
    }

    private void showSubscriptionInfo(final ClientSubscriptionInfo.ClientReceivingSubscription clientReceivingSubscription) {
        _senderList.clearSelection();
        _potDavList.clearSelection();
        _label.setText("<html>" + "<b>Verbindung mit: </b>" + clientReceivingSubscription.getApplicationPidOrId() + "<br>" + "<b>Benutzer: </b>" +
                       clientReceivingSubscription.getUserPidOrId() + "<br>" + "<b>Typ: </b>" +
                       (clientReceivingSubscription.isDrain() ? "Senke" : "Empfänger") + "<br>" + "<b>Nachgelieferte Daten: </b>" +
                       (clientReceivingSubscription.isDelayed() ? "Ja" : "Nein") + "<br>" + "<b>Nur Änderungen: </b>" +
                       (clientReceivingSubscription.isDelta() ? "Ja" : "Nein") + "<br>" + "<b>Status: </b>" +
                       stateToString(clientReceivingSubscription.getState()) + "<br>" + "<b>Verbindung: </b>" +
                       stateToString(clientReceivingSubscription.getConnectionState()));
    }

    private void showSubscriptionInfo(final ClientSubscriptionInfo.ClientSendingSubscription clientSendingSubscription) {
        _receiverList.clearSelection();
        _potDavList.clearSelection();
        _label.setText("<html>" + "<b>Verbindung mit: </b>" + clientSendingSubscription.getApplicationPidOrId() + "<br>" + "<b>Benutzer: </b>" +
                       clientSendingSubscription.getUserPidOrId() + "<br>" + "<b>Typ: </b>" +
                       (clientSendingSubscription.isSource() ? "Quelle" : "Sender") + "<br>" + "<b>Unterstützt Sendesteuerung: </b>" +
                       (clientSendingSubscription.isRequestSupported() ? "Ja" : "Nein") + "<br>" + "<b>Status: </b>" +
                       stateToString(clientSendingSubscription.getState()) + "<br>" + "<b>Verbindung: </b>" +
                       stateToString(clientSendingSubscription.getConnectionState()));
    }

    private void showDavInfo(final ClientSubscriptionInfo.DavInformation davInfo) {
        _receiverList.clearSelection();
        _senderList.clearSelection();
        _label.setText("<html>" + "<b>Zentraldatenverteiler: </b>" + davInfo.getCentralDavPidOrId() + "<br>" + "<b>Verbindung über: </b>" +
                       davInfo.getConnectionDavPidOrId() + "<br>" + "<b>Als Benutzer: </b>" + davInfo.getUserPidOrId() + "<br>" +
                       "<b>Wichtung: </b>" + davInfo.getThroughputResistance());
    }

    private class MySelectionListener<T> implements ListSelectionListener {

        private final SelectionList<T> _list;

        public MySelectionListener(final SelectionList<T> list) {
            _list = list;
        }

        @Override
        public void valueChanged(final ListSelectionEvent e) {
            List<T> selectedValues = _list.getSelectedValues();
            if (selectedValues.isEmpty()) {
                return;
            }
            T element = selectedValues.get(0);
            if (element instanceof ClientSubscriptionInfo.ClientSendingSubscription) {
                showSubscriptionInfo((ClientSubscriptionInfo.ClientSendingSubscription) element);
            }
            if (element instanceof ClientSubscriptionInfo.ClientReceivingSubscription) {
                showSubscriptionInfo((ClientSubscriptionInfo.ClientReceivingSubscription) element);
            }
            if (element instanceof ClientSubscriptionInfo.DavInformation) {
                showDavInfo((ClientSubscriptionInfo.DavInformation) element);
            }
        }
    }

    class MyListCellRenderer extends DefaultListCellRenderer {
        public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected, boolean cellHasFocus) {
            Component c = super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
	        if (value instanceof ClientSubscriptionInfo.ClientSendingSubscription subscription) {
		        //				if(subscription.isSource()){
//					c.setForeground(new Color(95, 0, 66));
//				}
//				if(!subscription.isLocal()){
//					c.setForeground(new Color(0, 95, 66));
//				}
            }
	        if (value instanceof ClientSubscriptionInfo.ClientReceivingSubscription subscription) {
		        //				if(subscription.isDrain()){
//					c.setForeground(new Color(95, 0, 66));
//				}
//				if(!subscription.isLocal()){
//					c.setForeground(new Color(0, 95, 66));
//				}
            }
            return c;
        }
    }
}
