/*
 * Copyright 2004 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2006-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.pat.sysbed.preselection.lists;

import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.pat.sysbed.preselection.tree.PreselectionTree;
import de.bsvrz.pat.sysbed.preselection.tree.PreselectionTreeListener;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.selectionlist.ObjectListRenderer;
import de.bsvrz.sys.funclib.kappich.selectionlist.SelectionList;
import java.awt.BorderLayout;
import java.awt.Dimension;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Objects;
import javax.swing.BorderFactory;
import javax.swing.Box;
import javax.swing.BoxLayout;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JSpinner;
import javax.swing.JSplitPane;
import javax.swing.ListSelectionModel;
import javax.swing.SpinnerModel;
import javax.swing.SpinnerNumberModel;
import javax.swing.SwingUtilities;

/**
 * Die Klasse {@code PreselectionLists} ist ein Teil der Datenidentifikationsauswahl. Sie stellt die konkreten Auswahloptionen anhand von Listen zur
 * Verfügung. Folgende Listen helfen dem Anwender dabei: Objekttyp, Attributgruppe, Aspekt und Objekt. Außerdem kann die Simulationsvariante angegeben
 * werden.
 * <p>
 * Der Konstruktor {@code PreselectionLists} erstellt das Panel und mit der Methode {@code setObjects} werden die Listen gefüllt.
 *
 * @author Kappich Systemberatung
 * @see #PreselectionLists
 * @see #setObjects
 */
public class PreselectionLists extends JPanel implements PreselectionTreeListener {

    /** Der Debugg-Logger */
    private static final Debug _debug = Debug.getLogger();

    /** Der PreselectionListsHandler verwaltet die Daten und aktualisiert die Listen. */
    private final PreselectionListsHandler _preselectionListsHandler;
    /** speichert die linke Seite des Splitpane */
    private final Box _leftBox = Box.createVerticalBox();
    /** Diese Liste wird für den Listener benötigt. */
    private List<PreselectionListsListener> _listenerList = new LinkedList<>();
    /** speichert ein Objekt zum Filtern der anzuzeigenden Listen */
    private PreselectionListsFilter _listsFilter;
    /** speichert die Liste Objekttyp */
    private SystemObjectSelectionList<SystemObjectType> _objtypList;
    /** speichert die Liste Attributgruppe */
    private SystemObjectSelectionList<AttributeGroup> _atgList;
    /** speichert die Liste Aspekt */
    private SystemObjectSelectionList<Aspect> _aspList;
    /** speichert die Liste Objekt */
    private SystemObjectSelectionList<SystemObject> _objList;
    /** Gibt an, ob die Objekt-Typen angezeigt werden sollen. */
    private boolean _showObjectTypes = true;

    /** Gibt an, ob Attributgruppen angezeigt werden sollen. */
    private boolean _showAttributeGroups = true;

    /** Gibt an, ob die Aspekte angezeigt werden sollen. */
    private boolean _showAspects = true;

    /** speichert die Simulationsvariante */
    private int _simulationsVariant = -1;

    /** speichert den JSpinner zum Anzeigen und Ändern der Simulationsvariante */
    private JSpinner _simulationVariantSpinner;

    private JSplitPane _divideLists;

    /* ################### Methoden ###################### */

    /**
     * Konstruktor, der ein Objekt der Klasse {@code PreselectionLists} erstellt.
     *
     * @see #createAndShowGui()
     */
    public PreselectionLists() {
        _preselectionListsHandler = new PreselectionListsHandler(this);
        createAndShowGui();
    }

    /**
     * @return Liste der Objekte
     */
    public SystemObjectSelectionList<SystemObject> getObjList() {
        return _objList;
    }

    /**
     * Mit dieser Methode werden zur Initialisierung Objekte (z.B. vom {@link PreselectionTree} übergeben. Aus diesen Werten werden dann die
     * Objekttypen, Attributgruppen und Aspekte rekonstruiert und dargestellt.
     *
     * @param systemObjects die darzustellenden Systemobjekte
     */
    public void setObjects(Collection<SystemObject> systemObjects) {
        notifyListSelectionChanged();
        _preselectionListsHandler.setObjects(systemObjects);
    }

    /**
     * Diese Methode gibt an, ob die Objekt-Typen angezeigt werden sollen.
     *
     * @param flag gibt an, ob die Objekt-Typen angezeigt werden sollen
     *
     * @throws IllegalStateException Falls die Objekt-Typen ausgeblendet werden sollen und die Attributgruppen und Aspekte nicht angezeigt werden.
     */
    public void showObjectTypes(boolean flag) {
        if (!flag && !_showAttributeGroups && !_showAspects) {
            throw new IllegalStateException("Mindestens eine der drei Listen (Objekt-Typen, Attributgruppen oder Aspekte) muss angezeigt bleiben.");
        }
        _showObjectTypes = flag;
        if (flag) {
            // Objekt-Typen anzeigen
            _leftBox.add(_objtypList, 0);
        } else {
            _leftBox.remove(_objtypList);
        }
        revalidate(); // Falls das PreselectionLists-Panel bereits angezeigt wird, muss es neu gezeichnet werden.
    }

    /**
     * Diese Methode gibt an, ob die Attributgruppen angezeigt werden sollen.
     *
     * @param flag gibt an, ob die Attributgruppen angezeigt werden sollen
     *
     * @throws IllegalStateException Falls die Attributgruppen ausgeblendet werden sollen und die Objekt-Typen und Aspekte nicht angezeigt werden.
     */
    public void showAttributeGroups(boolean flag) {
        if (!flag && !_showObjectTypes && !_showAspects) {
            throw new IllegalStateException("Mindestens eine der drei Listen (Objekt-Typen, Attributgruppen oder Aspekte) muss angezeigt bleiben.");
        }
        _showAttributeGroups = flag;
        if (flag) {
            int position = 0;
            if (_showObjectTypes) {
                position = 1;
            }
            _leftBox.add(_atgList, position);
        } else {
            _leftBox.remove(_atgList);
        }
        revalidate(); // Falls das PreselectionLists-Panel bereits angezeigt wird, muss es neu gezeichnet werden.
    }

    /**
     * Diese Methode gibt an, ob die Aspekte angezeigt werden sollen.
     *
     * @param flag gibt an, ob die Aspekte angezeigt werden sollen
     *
     * @throws IllegalStateException Falls die Aspekte ausgeblendet werden sollen und die Objekt-Typen und Attributgruppen nicht angezeigt werden.
     */
    public void showAspects(boolean flag) {
        if (!flag && !_showAttributeGroups && !_showObjectTypes) {
            throw new IllegalStateException("Mindestens eine der drei Listen (Objekt-Typen, Attributgruppen oder Aspekte) muss angezeigt bleiben.");
        }
        _showAspects = flag;
        if (flag) {
            int position = 0;
            if (_showObjectTypes) {
                position++;
            }
            if (_showAttributeGroups) {
                position++;
            }
            _leftBox.add(_aspList, position);
        } else {
            _leftBox.remove(_aspList);
        }
        revalidate(); // Falls das PreselectionLists-Panel bereits angezeigt wird, muss es neu gezeichnet werden.
    }

    /**
     * Methode, um die Simulationsvariante anzuzeigen. Default-Wert ist "0". Soll ein anderer Wert voreingestellt sein, dann ist die Methode {@link
     * #setSimulationVariant} aufzurufen.
     */
    public void showSimulationVariant() {
        JPanel simPanel = new JPanel();
        simPanel.setLayout(new BoxLayout(simPanel, BoxLayout.X_AXIS));
        if (_simulationsVariant == -1) {
            _simulationsVariant = 0;
        }
        SpinnerModel spinnerModel = new SpinnerNumberModel(_simulationsVariant, 0, 999, 1);
        _simulationVariantSpinner = new JSpinner(spinnerModel);
        _simulationVariantSpinner.addChangeListener(e -> _simulationsVariant = (Integer) _simulationVariantSpinner.getValue());
        JLabel simLabel = new JLabel("Simulationsvariante: ");
        simLabel.setLabelFor(_simulationVariantSpinner);
        simPanel.add(simLabel);
        simPanel.add(_simulationVariantSpinner);
        simPanel.setBorder(BorderFactory.createEmptyBorder(3, 3, 3, 3));
        _leftBox.add(simPanel);
    }

    /**
     * Die Methode wird vom Konstruktor aufgerufen und stellt konkrete Auswahloptionen für die Datenidentifikationsauswahl in Form von Auswahllisten
     * für Objekttypen, Attributgruppen, Aspekte und Objekte zur Verfügung.
     */
    private void createAndShowGui() {
        setPreferredSize(new Dimension(500, 350));

        _objtypList = new SystemObjectSelectionList<>("Objekttyp", "Objekttypen", SelectionList.FilterStyle.HiddenFilter);
        _atgList = new SystemObjectSelectionList<>("Attributgruppe", "Attributgruppen", SelectionList.FilterStyle.HiddenFilter);
        _aspList = new SystemObjectSelectionList<>("Aspekt", "Aspekte", SelectionList.FilterStyle.HiddenFilter);

        _leftBox.add(_objtypList);
        _leftBox.add(_atgList);
        _leftBox.add(_aspList);
//		_leftBox.setBorder(BorderFactory.createEmptyBorder(3, 3, 3, 3));
        _leftBox.setMinimumSize(new Dimension(0, 0));

        // Rechte Seite des SplitPane wird gefüllt
        _objList = new SystemObjectSelectionList<>("Objekte", "Objekte", SelectionList.FilterStyle.HiddenFilter);
        _objtypList.addListSelectionListener(e -> {
            if (!e.getValueIsAdjusting()) {
                notifyListSelectionChanged();
                final List<SystemObjectType> objectTypes = new LinkedList<>();
                objectTypes.addAll(_objtypList.getSelectedValues());
                final List<AttributeGroup> atgs = new LinkedList<>();
                atgs.addAll(_atgList.getSelectedValues());
                final List<Aspect> asps = new LinkedList<>();
                asps.addAll(_aspList.getSelectedValues());

                _preselectionListsHandler.objectsDependOnObjectType(objectTypes, atgs, asps);
            }
        });
        _atgList.addListSelectionListener(e -> {
            if (!e.getValueIsAdjusting()) {
                notifyListSelectionChanged();
                final List<AttributeGroup> atgs = new ArrayList<>();
                atgs.addAll(_atgList.getSelectedValues());
                final List<SystemObjectType> objectTypes = new LinkedList<>();
                objectTypes.addAll(_objtypList.getSelectedValues());
                final List<Aspect> asps = new LinkedList<>();
                asps.addAll(_aspList.getSelectedValues());

                _preselectionListsHandler.objectsDependOnAtg(objectTypes, atgs, asps);
            }
        });
        _aspList.addListSelectionListener(e -> {
            if (!e.getValueIsAdjusting()) {
                notifyListSelectionChanged();
                final List<Aspect> asps = new LinkedList<>();
                asps.addAll(_aspList.getSelectedValues());
                final List<SystemObjectType> objectTypes = new LinkedList<>();
                objectTypes.addAll(_objtypList.getSelectedValues());
                final List<AttributeGroup> atgs = new LinkedList<>();
                atgs.addAll(_atgList.getSelectedValues());
                _preselectionListsHandler.objectsDependOnAsp(objectTypes, atgs, asps);
            }
        });
        _objList.addListSelectionListener(e -> {
            if (!e.getValueIsAdjusting()) {
                notifyListSelectionChanged();
            }
        });

        _divideLists = new JSplitPane(JSplitPane.HORIZONTAL_SPLIT);
        _divideLists.setContinuousLayout(true);
        _divideLists.setOneTouchExpandable(true);
        _divideLists.setResizeWeight(0.60);
        _divideLists.setLeftComponent(_leftBox);
        _divideLists.setRightComponent(_objList);

        // fügt den SplitPane zum JPanel PreselectionLists
        setLayout(new BorderLayout());
        add(_divideLists, BorderLayout.CENTER);
    }

    /**
     * Mit dieser Methode können Objekte angegeben werden, die beim Füllen der Listen vorselektiert sein sollen.
     *
     * @param preselectedObjectTypes Objekte, die vorselektiert sein sollen. Wird {@code null} übergeben, wird die Selektion gelöscht.
     */
    public void setPreselectedObjectTypes(final List<SystemObjectType> preselectedObjectTypes) {
        SwingUtilities.invokeLater(() -> {
            if (preselectedObjectTypes == null) {
                _objtypList.clearSelection();
            } else {
                // falls schon Elemente in der Liste sind -> versuchen die Objekte zu selektieren
                _objtypList.selectElements(preselectedObjectTypes);
            }
        });
    }

    /**
     * Mit dieser Methode können Attributgruppen angegeben werden, die beim Füllen der Listen vorselektiert sein sollen.
     *
     * @param preselectedAttributeGroups Attributgruppen, die vorselektiert sein sollen. Wird {@code null} übergeben, wird die Selektion gelöscht.
     */
    public void setPreselectedAttributeGroups(final List<AttributeGroup> preselectedAttributeGroups) {
        SwingUtilities.invokeLater(() -> {
            if (preselectedAttributeGroups == null) {
                _atgList.clearSelection();
            } else {
                // falls schon Elemente in der Liste sind -> versuchen die Objekte zu selektieren
                _atgList.selectElements(preselectedAttributeGroups);
            }
        });
    }

    /**
     * Mit dieser Methode können Aspekte angegeben werden, die beim Füllen der Listen vorselektiert sein sollen.
     *
     * @param preselectedAspects Aspekte, die vorselektiert sein sollen. Wird {@code null} übergeben, wird die Selektion gelöscht.
     */
    public void setPreselectedAspects(final List<Aspect> preselectedAspects) {
        SwingUtilities.invokeLater(() -> {
            if (preselectedAspects == null) {
                _aspList.clearSelection();
            } else {
                // falls schon Elemente in der Liste sind -> versuchen die Objekte zu selektieren
                _aspList.selectElements(preselectedAspects);
            }
        });
    }

    /**
     * Mit dieser Methode können Objekte angegeben werden, die beim Füllen der Listen vorselektiert sein sollen.
     *
     * @param preselectedObjects Objekte, die vorselektiert sein sollen. Wird {@code null} übergeben, wird die Selektion gelöscht.
     */
    public void setPreselectedObjects(final List<SystemObject> preselectedObjects) {
        SwingUtilities.invokeLater(() -> {
            if (preselectedObjects == null) {
                _objList.clearSelection();
            } else {
                // falls schon Elemente in der Liste sind -> versuchen die Objekte zu selektieren
                _objList.selectElements(preselectedObjects);
            }
        });
    }

    /**
     * Aktualisiert die Liste mit den Objekten und wendet ggf. einen Filter der Anwendung an.
     *
     * @param objectList Die Liste mit den Objekten.
     */
    void setObjectList(Collection<? extends SystemObject> objectList) {
        if (_listsFilter != null) {
            // Filter von der Anwendung anwenden
            objectList = applyFilter(PreselectionListsFilter.OBJECT_LIST, objectList);
            if (objectList != null) {
                objectList.removeIf(Objects::isNull);
            } else {
                _debug.error(
                    "Bei der Bearbeitung der darzustellenden Objekte durch ein Filter-Plugin wurde statt der Ergebnisliste eine Null-Referenz " +
                    "zurückgeliefert.");
                objectList = new LinkedList<>();
            }
        }
        _objList.setElements(objectList);
    }

    /**
     * Aktualisiert die Liste mit den Objekttypen und wendet ggf. einen Filter der Anwendung an.
     *
     * @param objecttypeList Die Liste mit den Objekttypen.
     */
    void setObjectTypeList(Collection<? extends SystemObjectType> objecttypeList) {
        if (_listsFilter != null) {
            // Filter von der Anwendung anwenden
            objecttypeList = applyFilter(PreselectionListsFilter.OBJECTTYPE_LIST, objecttypeList);
            if (objecttypeList != null) {
                objecttypeList.removeIf(Objects::isNull);
            } else {
                _debug.error(
                    "Bei der Bearbeitung der darzustellenden Objekttypen durch ein Filter-Plugin wurde statt der Ergebnisliste eine Null-Referenz " +
                    "zurückgeliefert.");
                objecttypeList = new LinkedList<>();
            }
        }
        _objtypList.setElements(objecttypeList);
    }

    /**
     * Aktualisiert die Liste mit den Attributgruppen und wendet ggf. einen Filter der Anwendung an.
     *
     * @param atgList Die Liste mit den Attributgruppen.
     */
    void setAtgList(Collection<? extends AttributeGroup> atgList) {
        if (_listsFilter != null) {
            // Filter von der Anwendung anwenden
            atgList = applyFilter(PreselectionListsFilter.ATTRIBUTEGROUP_LIST, atgList);
            if (atgList != null) {
                atgList.removeIf(Objects::isNull);
            } else {
                _debug.error("Bei der Bearbeitung der darzustellenden Attributgruppen durch ein Filter-Plugin wurde statt der Ergebnisliste eine " +
                             "Null-Referenz zurückgeliefert.");
                atgList = new LinkedList<>();
            }
        }
        _atgList.setElements(atgList);
    }

    /**
     * Aktualisiert die Liste mit den Aspekten und wendet ggf. einen Filter der Anwendung an.
     *
     * @param aspList Die Liste mit den Aspekten.
     */
    void setAspList(Collection<? extends Aspect> aspList) {
        if (_listsFilter != null) {
            // Filter der Anwendung anwenden
            aspList = applyFilter(PreselectionListsFilter.ASPECT_LIST, aspList);
            if (aspList != null) {
                aspList.removeIf(Objects::isNull);
            } else {
                aspList = new LinkedList<>();
            }
        }
        _aspList.setElements(aspList);
    }

    /**
     * Legt Einfach- oder Mehrfachauswahl für die Liste Objekttyp fest.
     *
     * @param selectionMode Als Argument kann eine der folgenden Konstanten übergeben werden:<br> {@link ListSelectionModel#SINGLE_SELECTION} {@link
     *                      ListSelectionModel#SINGLE_INTERVAL_SELECTION} {@link ListSelectionModel#MULTIPLE_INTERVAL_SELECTION}
     */
    public void setObjectTypeSelectionMode(int selectionMode) {
        _objtypList.setSelectionMode(selectionMode);
    }

    /**
     * Legt Einfach- oder Mehrfachauswahl für die Liste Attributgruppe fest.
     *
     * @param selectionMode Als Argument kann eine der folgenden Konstanten übergeben werden:<br> {@link ListSelectionModel#SINGLE_SELECTION} {@link
     *                      ListSelectionModel#SINGLE_INTERVAL_SELECTION} {@link ListSelectionModel#MULTIPLE_INTERVAL_SELECTION}
     */
    public void setAtgSelectionMode(int selectionMode) {
        _atgList.setSelectionMode(selectionMode);
    }

    /**
     * Legt Einfach- oder Mehrfachauswahl für die Liste Aspekt fest.
     *
     * @param selectionMode Als Argument kann eine der folgenden Konstanten übergeben werden:<br> {@link ListSelectionModel#SINGLE_SELECTION} {@link
     *                      ListSelectionModel#SINGLE_INTERVAL_SELECTION} {@link ListSelectionModel#MULTIPLE_INTERVAL_SELECTION}
     */
    public void setAspSelectionMode(int selectionMode) {
        _aspList.setSelectionMode(selectionMode);
    }

    /**
     * Legt Einfach- oder Mehrfachauswahl für die Liste Objekte fest.
     *
     * @param selectionMode Als Argument kann eine der folgenden Konstanten übergeben werden:<br> {@link ListSelectionModel#SINGLE_SELECTION} {@link
     *                      ListSelectionModel#SINGLE_INTERVAL_SELECTION} {@link ListSelectionModel#MULTIPLE_INTERVAL_SELECTION}
     */
    public void setObjectSelectionMode(int selectionMode) {
        _objList.setSelectionMode(selectionMode);
    }

    /**
     * Gibt die selektierten Objekte zurück.
     *
     * @return die selektierten Objekte
     */
    public List<SystemObject> getSelectedObjects() {
        return _objList.getSelectedValues();
    }

    /**
     * Gibt die selektierten Objekttypen zurück.
     *
     * @return die selektierten Objekttypen
     */
    public List<SystemObjectType> getSelectedObjectTypes() {
        final List<SystemObjectType> systemObjectTypes = new ArrayList<>();
        for (SystemObjectType object : _objtypList.getSelectedValues()) {
            if (object != null) {
                systemObjectTypes.add(object);
            }
        }
        return Collections.unmodifiableList(systemObjectTypes);
    }

    /**
     * Gibt die selektierten Attributgruppen zurück.
     *
     * @return die selektierten Attributgruppen
     */
    public List<AttributeGroup> getSelectedAttributeGroups() {
        final List<AttributeGroup> atgGroups = new ArrayList<>();
        for (AttributeGroup object : _atgList.getSelectedValues()) {
            if (object != null) {
                atgGroups.add(object);
            }
        }
        return Collections.unmodifiableList(atgGroups);
    }

    /**
     * Gibt die selektierten Aspekte zurück.
     *
     * @return die selektierten Aspekte
     */
    public List<Aspect> getSelectedAspects() {
        final List<Aspect> aspects = new ArrayList<>();
        for (Aspect object : _aspList.getSelectedValues()) {
            if (object != null) {
                aspects.add(object);
            }
        }
        return Collections.unmodifiableList(aspects);
    }

    /**
     * Gibt die eingestellte Simulationsvariante zurück.
     *
     * @return die eingestellte Simulationsvariante
     */
    public int getSimulationVariant() {
        return _simulationsVariant;
    }

    /**
     * Mit dieser Methode kann die Simulationsvariante gesetzt werden.
     *
     * @param value neuer Wert der Simulationsvariante
     */
    public void setSimulationVariant(int value) {
        _simulationsVariant = value;
        if (_simulationVariantSpinner != null) {
            _simulationVariantSpinner.setValue(value);
        }
    }

    /**
     * Fügt einen {@code PreselectionListsListener} hinzu.
     *
     * @param listener der hinzuzufügende PreselectionListsListener
     */
    public void addPreselectionListener(PreselectionListsListener listener) {
        _listenerList.add(listener);
    }

    /**
     * Entfernt einen {@code PreselectionListsListener}.
     *
     * @param listener der zu entfernende PreselectionListsListener
     */
    public void removePreselectionListener(PreselectionListsListener listener) {
        _listenerList.remove(listener);
    }

    /** Gibt dem Listener Bescheid, ob bei der Selektion einer der vier Listen des PreselectionLists-Panels eine Änderung eingetreten ist. */
    private void notifyListSelectionChanged() {
        for (PreselectionListsListener preselectionListsListener : _listenerList) {
            preselectionListsListener.listSelectionChanged(this);
        }
    }

    /**
     * Setzt einen Filter, welcher vor Anzeige der Listen diese bei Bedarf filtert. Ist der Übergabeparameter {@code null}, dann wird das gesetzte
     * Filter-Objekt gelöscht.
     *
     * @param listsFilter der Filter
     */
    public void setPreselectionListsFilter(PreselectionListsFilter listsFilter) {
        _listsFilter = listsFilter;
        // Filter anwenden, wenn Elemente in der Liste angezeigt werden
        if (!_objList.getElements().isEmpty()) {
            setObjectList(_objList.getElements());
        }
        if (!_objtypList.getElements().isEmpty()) {
            setObjectTypeList(_objtypList.getElements());
        }
        if (!_atgList.getElements().isEmpty()) {
            setAtgList(_atgList.getElements());
        }
        if (!_aspList.getElements().isEmpty()) {
            setAspList(_aspList.getElements());
        }
    }

    /**
     * Ruft die Methode {@code applyFilter} des {@link PreselectionListsFilter} Interfaces auf.
     *
     * @param whichList Konstante, die angibt, welche der vier Listen übergeben wird
     * @param list      die zu filternde Liste
     *
     * @return die gefilterte Liste
     */
    private <T extends SystemObject> Collection<? extends T> applyFilter(int whichList, Collection<? extends T> list) {
        //noinspection unchecked
        return (Collection<? extends T>) _listsFilter.applyFilter(whichList, list);
    }

    /**
     * Mit dieser Methode können die maximal anzuzeigenden Objekt-Typen der entsprechenden Liste eingeschränkt werden.
     *
     * @param objectTypes die maximal anzuzeigenden Objekt-Typen
     */
    public void setObjectTypeFilter(final Collection<SystemObjectType> objectTypes) {
        SwingUtilities.invokeLater(() -> _preselectionListsHandler.setObjectTypeFilter(objectTypes));
    }

    /**
     * Mit dieser Methode können die maximal anzuzeigenden Attributgruppen der entsprechenden Liste eingeschränkt werden.
     *
     * @param attributeGroups die maximal anzuzeigenden Attributgruppen
     */
    public void setAttributeGroupFilter(final Collection<AttributeGroup> attributeGroups) {
        SwingUtilities.invokeLater(() -> _preselectionListsHandler.setAttributeGroupFilter(attributeGroups));
    }

    /**
     * Mit dieser Methode können die maximal anzuzeigenden Aspekte der entsprechenden Liste eingeschränkt werden.
     *
     * @param aspects die maximal anzuzeigenden Aspekte
     */
    public void setAspectFilter(final Collection<Aspect> aspects) {
        SwingUtilities.invokeLater(() -> _preselectionListsHandler.setAspectFilter(aspects));
    }

    /**
     * Mit dieser Methode, wird nur die Liste Objekt angezeigt.
     *
     * @param schowOnlyObjList gibt an, ob nur die Liste Objekt angezeigt werden soll
     */
    public void setOnlyObjectListVisible(boolean schowOnlyObjList) {
        if (schowOnlyObjList) {
            _divideLists.setDividerLocation(0.0);
        } else {
            _divideLists.setDividerLocation(_divideLists.getWidth() / 2);
        }
    }

    public ObjectListRenderer<? super SystemObject> getObjectListRenderer() {
        return _objList.getListRenderer();
    }

    public void setObjectListRenderer(final ObjectListRenderer<? super SystemObject> objectListRenderer) {
        _objList.setListRenderer(objectListRenderer);
        _objList.repaint();
    }
}
