/*
 * Copyright 2004 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.pat.sysbed.preselection.tree;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientReceiverInterface;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.DataNotSubscribedException;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.ObjectTimeSpecification;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.pat.sysbed.preselection.treeFilter.standard.Filter;
import de.bsvrz.sys.funclib.debug.Debug;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import javax.swing.SwingUtilities;
import javax.swing.event.TreeModelListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

/**
 * Die Klasse {@code PreselectionTreeHandler} verarbeitet die Daten des Panels {@link PreselectionTree}.<p> Mit dem Konstruktor wird das Panel {@code
 * PreselectionTree}, ein {@code ClientDavInterface} und ein {@code Systemobjekt} übergeben.<br> Mittels des {@code ClientDavInterfaces} und des
 * Objektes werden Daten aus dem Datenverteiler geholt, die im {@code PreselectionTree} dargestellt werden sollen. Außerdem werden alle
 * Konfigurationsobjekte und alle dynamische Objekte zur späteren Bearbeitung geholt.
 * <p>
 * Bei Anwahl eines Knotens werden die geholten Objekte ggf. durch zum Knoten gehörende Filter eingeschränkt. Die Liste dieser Objekte wird dann an
 * das {@link de.bsvrz.pat.sysbed.preselection.lists.PreselectionLists}-Panel weitergereicht, wo weiter eingeschränkt werden kann.
 *
 * @author Kappich Systemberatung
 */
class PreselectionTreeHandler {

    /** Der Debug-Logger der Klasse */
    private static Debug _debug = Debug.getLogger();

    /** speichert ein Objekt der Klasse {@code PreselectionTree} */
    private PreselectionTree _preselectionTree;

    /** speichert das ClientDavInterface */
    private ClientDavInterface _connection;

    /** speichert das aktuelle DataModel */
    private DataModel _configuration;

    /** speichert ein Objekt der Klasse {@code Receiver} */
    private Receiver _receiver;

    /** speichert alle Systemobjekte in einer Liste */
    private Collection<SystemObject> _allObjects = new ArrayList<>();

    /** speichert die Parameter für den Vorauswahldialog (Baum) */
    private Collection<Object> _treeNodes;

    /** speichert ein empfangenes Menü-Data-Objekt */
    private List<TreeNodeObject> _tree = new ArrayList<>();

    /**
     * Ein Objekt dieser Klasse wird erstellt.
     *
     * @param preselectionTree das Panel, wo die Baum dargestellt werden soll
     * @param connection       Verbindung zum Datenverteiler
     */
    PreselectionTreeHandler(PreselectionTree preselectionTree, ClientDavInterface connection) {
        _preselectionTree = preselectionTree;
        _connection = connection;
        _configuration = _connection.getDataModel();
    }

    /**
     * Auf Grundlage eines ClientDavInterfaces wird ein {@link DataModel} erstellt und an das {@link ClientReceiverInterface} (implementiert durch die
     * Klasse {@link Receiver}) weitergegeben.
     *
     * @param objectList Objektliste, die beim Datenverteiler angemeldet werden soll
     */
    private void getData(List<SystemObject> objectList) {
        AttributeGroup atg = _configuration.getAttributeGroup("atg.datenAuswahlMenü");
        Aspect aspect = _configuration.getAspect("asp.parameterSoll");
        DataDescription dataDescription = new DataDescription(atg, aspect);
        _receiver = new Receiver();
        _connection.subscribeReceiver(_receiver, objectList, dataDescription, ReceiveOptions.normal(), ReceiverRole.receiver());
    }

    /**
     * Liefert alle Konfigurations- und dynamischen Objekte.
     *
     * @return alle Konfigurations- und dynamischen Objekte
     */
    Collection<SystemObject> getAllObjects() {
        return Collections.unmodifiableCollection(_allObjects);
    }

    Collection<SystemObject> getAllObjects(final ObjectTimeSpecification timeSpecification) {
        List<SystemObject> result = new ArrayList<>();
        result.addAll(_configuration.getObjects(null, null, timeSpecification));
        return result;
    }

    /**
     * Holt auf Grundlage des {@link de.bsvrz.dav.daf.main.config.DataModel DataModels} alle Konfigurations- und dynamischen Objekte und speichert sie
     * in einer Collection. Diese kann mittels {@link #getAllObjects} geholt werden.
     */
    void initDataLists() {
        List<SystemObject> confObjs = _configuration.getType("typ.konfigurationsObjekt").getObjects();
        List<SystemObject> dynObjs = _configuration.getType("typ.dynamischesObjekt").getObjects();

        final List<SystemObject> list = new ArrayList<>(confObjs.size() + dynObjs.size());
        list.addAll(confObjs);
        list.addAll(dynObjs);

        _allObjects.addAll(list);
    }

    /**
     * Gibt die Parameter für die Vorauswahl (Baum) zurück. Die Collection enthält Systemobjekte und {@link TreeNodeObject Knotenobjekte}. Anhand der
     * Objekte wird der Baum für die Vorauswahl erzeugt.
     *
     * @return die Sammlung von System- und Knotenobjekten
     */
    Collection<Object> getTreeNodes() {
        return _treeNodes;
    }

    /**
     * Die Systemobjekte werden beim Datenverteiler angemeldet und mitsamt den Baumobjekten im {@code PreselectionTree} angezeigt. Die benötigten
     * Parameter (Systemobjekte) werden an den DaV übergeben, um mit den empfangenen Daten die spezifizierte Vorauswahl (Bäume) zu erstellen.
     *
     * @param treeNodes enthält die darzustellenden Systemobjekte und die implementierten Baumobjekte
     */
    public void setTreeNodes(Collection<Object> treeNodes) {
        _treeNodes = treeNodes;
        // Systemobjekte rausfiltern
        List<SystemObject> objectList = new LinkedList<>();
        for (Object object : treeNodes) {
	        if (object instanceof SystemObject systemObject) {
                objectList.add(systemObject);
            }
        }
        if (!objectList.isEmpty()) {
            try {
                // Empfänger anmelden
                getData(objectList);
            } catch (DataNotSubscribedException e) {
                e.printStackTrace();
            }
        }
        // Schon mal ein leeres TreeModel mit dem Alles-Node anlegen, wird beim Empfang der Parameterdaten erweitert.
        _preselectionTree.setTreeData(makeTreeModel(new ResultData[0]));
    }

    /**
     * Erstellt aus einer Collection von Systemobjekten und {@code TreeNodeObject}s einen Baum. Für jedes Systemobjekt wird der dazugehörige Datensatz
     * vom Datenverteiler in einen Baum umgewandelt. Die so erstellten Bäume und die {@code TreeNodeObject}s werden an eine Pseudo-Wurzel gehangen und
     * in ein {@code TreeModel} umgewandelt.
     *
     * @param results Daten vom Datenverteiler zu den Systemobjekten
     *
     * @return den Baum für das {@code PreselectionTree}
     *
     * @see TreeNodeObject
     * @see PreselectionTree
     */
    private TreeModel makeTreeModel(ResultData[] results) {
        // Pseudo-Wurzel erstellen
        TreeNodeObject rootNode = new TreeNodeObject("Wurzel", "");
        int position = 0;
        for (Object object : _treeNodes) {
            TreeNodeObject treeNodeObject = null;
	        if (object instanceof SystemObject systemObject) {           // vom Typ SystemObject
                // resultListe nach passendem SystemObjekt durchsuchen
                for (ResultData result : results) {  // falls es eins gibt -> nehme dieses
                    if (systemObject == result.getObject()) {       // passendes gefunden
                        treeNodeObject = makeRootNode(result);
                    }
                }
            } else if (object instanceof TreeNodeObject) {  // vom Typ TreeNodeObject
                treeNodeObject = (TreeNodeObject) object;
            }
            if (position < _tree.size()) {
                if (treeNodeObject != null) {
                    _tree.set(position, treeNodeObject);
                }
            } else {
                if (treeNodeObject == null) {
                    treeNodeObject = new TreeNodeObject("");
                }
                _tree.add(treeNodeObject);
            }
            position++;
        }
        for (TreeNodeObject treeNodeObject : _tree) {   // DataTreeModel erzeugen
            rootNode.addChild(treeNodeObject);
        }
        return new DataTreeModel(rootNode);
    }

    /**
     * Erstellt zu einem einzelnen Datensatz vom Datenverteiler einen Baum und gibt den Wurzelknoten ({@link TreeNodeObject}) zurück.
     *
     * @param result Datensatz vom Datenverteiler
     *
     * @return den Wurzelknoten
     */
    private TreeNodeObject makeRootNode(ResultData result) {
        // erste Knoten ist Wurzelknoten
        String rootPid = "";
        // alle erzeugten TreeNodeObjekte werden in einer HashMap gespeichert
        Map<String, TreeNodeObject> hashMap = new HashMap<>();
        // erst werden alle TreeNodeObjekte ohne Untermenüeinträge erstellt
        if (!result.hasData()) {
            _debug.warning("Zum Systemobject " + result.getObject().getName() + " gibt es keine Daten!");
            return new TreeNodeObject("");
        }
        Data data = result.getData();
        Data menu = data.getItem("Menü");
        Data.Array menuArray = menu.asArray();
        for (int i = 0; i < menuArray.getLength(); i++) {
            Data menuData = menuArray.getItem(i);   // den i-ten Eintrag
            String pid = menuData.getTextValue("Pid").getText();
            String name = menuData.getTextValue("Name").getText();
            TreeNodeObject treeNode = new TreeNodeObject(name, pid);
            Data.Array filterArray = menuData.getArray("Filter");
            for (int j = 0; j < filterArray.getLength(); j++) {
                Data filter = filterArray.getItem(j);
                String criteria = filter.getTextValue("Kriterium").getText();
                String[] values = filter.getTextArray("Wert").getTextArray();
                treeNode.addFilter(new Filter(criteria, values, _connection));
            }
            // speichern der Knoten in einer HashMap
            hashMap.put(pid, treeNode);
            // Wurzelknoten merken
            if (i == 0) {
                rootPid = pid;
            }
        }
        // Untermenü-Verweise werden erstellt.
        for (int i = 0; i < menuArray.getLength(); i++) {
            Data menuData = menuArray.getItem(i);
            String[] subItems = menuData.getArray("UnterMenü").asTextArray().getTextArray();
            if (subItems.length > 0) {
                String pid = menuData.getTextValue("Pid").getText();
                TreeNodeObject treeNode = hashMap.get(pid);    // hierzu Untermenüs erstellen
                for (String subItem : subItems) {
                    // gibt es zu dieser Pid überhaupt einen Eintrag?
                    if (hashMap.containsKey(subItem)) {
                        treeNode.addChild(hashMap.get(subItem));
                    } else {
                        _debug.warning("Zur Untermenü-Pid " + subItem + " gibt es kein Data-Objekt!");
                    }
                }
            }
        }
        TreeNodeObject rootNode = hashMap.get(rootPid);
        if (rootNode != null) {
            return rootNode;
        } else {
            return new TreeNodeObject("");
        }
    }

    /** Die Klasse {@code DataTreeModel} repräsentiert ein {@link TreeModel}. */
    private static class DataTreeModel implements TreeModel {

        /** speichert den Wurzelknoten */
        private TreeNodeObject _rootObject;

        /**
         * Konstruktor
         *
         * @param rootObject der Wurzelknoten
         */
        public DataTreeModel(TreeNodeObject rootObject) {
            _rootObject = rootObject;
        }

        /**
         * Gibt des Wurzelknoten zurück.
         *
         * @return den Wurzelknoten
         */
        public Object getRoot() {
            return _rootObject;
        }

        /**
         * Gibt zurück, wieviele Nachfolger ein Knoten hat.
         *
         * @param parent ein Knoten des Baums
         *
         * @return Anzahl der Nachfolger des Knotens
         */
        public int getChildCount(Object parent) {
	        if (parent instanceof TreeNodeObject treeNodeObject) {
                return treeNodeObject.getChildCount();
            }
            return 0;
        }

        /**
         * Gibt zurück, ob ein Knoten ein Blatt ist.
         *
         * @param node ein Knoten des Baums
         *
         * @return ob ein Knoten ein Blatt ist (true/false)
         */
        public boolean isLeaf(Object node) {
	        if (node instanceof TreeNodeObject treeNodeObject) {
                if (treeNodeObject.getChildCount() == 0) {
                    return true;
                } else {
                    return false;
                }
            }
            return false;
        }

        public void addTreeModelListener(TreeModelListener l) {
            // wird hier nicht benötigt
        }

        public void removeTreeModelListener(TreeModelListener l) {
            // wird hier nicht benötigt
        }

        /**
         * Gibt zu einem Knoten im Baum einen bestimmten Nachfolger zurück.
         *
         * @param parent ein Knoten im Baum
         * @param index  der wievielte Nachfolger
         *
         * @return den gewünschten Nachfolger
         */
        public Object getChild(Object parent, int index) {
	        if (parent instanceof TreeNodeObject treeNodeObject) {
                return treeNodeObject.getChild(index);
            }
            return null;
        }

        /**
         * Gibt zu einem Nachfolger eines Knotens seine Position innerhalb alles Nachfolger dieses Knotens zurück.
         *
         * @param parent ein Knoten im Baum
         * @param child  ein Nachfolger des Knotens
         *
         * @return Position des Nachfolgers
         */
        public int getIndexOfChild(Object parent, Object child) {
	        if (parent instanceof TreeNodeObject treeNodeObject) {
                return treeNodeObject.indexOfChild((TreeNodeObject) child);
            }
            return 0;
        }

        public void valueForPathChanged(TreePath path, Object newValue) {
            // wird hier nicht benötigt
        }
    }

    /**
     * Die Klasse {@code Receiver} implementiert das Interface {@code ClientReceiverInterface} und dient somit als Schnittstelle, um Aktualisierungen
     * von Daten, die zum Empfang angemeldet sind, zu verarbeiten.
     *
     * @see Receiver#update(de.bsvrz.dav.daf.main.ResultData[])
     */
    private class Receiver implements ClientReceiverInterface {

        /**
         * Diese Methode erhält die Daten vom Datenverteiler. Die empfangenen Daten werden in einen Baum umgewandelt und dem Panel {@link
         * PreselectionTree} übermittelt.
         *
         * @param results die Daten vom Datenverteiler
         */
        public void update(ResultData[] results) {
            final TreeModel treeModel;
            treeModel = makeTreeModel(results);     // Baum wird erstellt
            SwingUtilities.invokeLater(new Runnable() {
                public void run() {
                    // Baum wird an PreselectionTree übermittelt
                    _preselectionTree.setTreeData(treeModel);
                }
            });
        }
    }
}
