/*
 * Copyright 2004 by Kappich+Kniß Systemberatung Aachen (K2S)
 * Copyright 2007-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.bsvrz.pat.sysbed.
 *
 * de.bsvrz.pat.sysbed is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.pat.sysbed is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.pat.sysbed.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.bsvrz.pat.sysbed.preselection.treeFilter.standard;

import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.ConfigurationArea;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.config.SystemObjectType;
import de.bsvrz.pat.sysbed.preselection.treeFilter.plugins.api.ExtendedFilter;
import de.bsvrz.sys.funclib.configObjectAcquisition.ConfigurationHelper;
import de.bsvrz.sys.funclib.debug.Debug;

import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

/**
 * Die Klasse {@code Filter} speichert ein Kriterium und die dazugehörigen Werte und bietet Methoden an, um Systemobjekte anhand des Kriteriums zu
 * filtern.
 *
 * @author Kappich Systemberatung
 */
public class Filter {

    /** filtert die aufgelisteten Objekte nach dem Konfigurationsbereich */
    public static final String CONFIGURATIONAREA = "Konfigurationsbereich";
    /** filtert die aufgelisteten Objekte nach dem Objekttyp */
    public static final String OBJECTTYPE = "Objekttyp";
    /** filtert die aufgelisteten Objekte nach der Attributgruppe */
    public static final String ATTRIBUTEGROUP = "Attributgruppe";
    /** filtert die aufgelisteten Objekte nach dem Aspekt */
    public static final String ASPECT = "Aspekt";
    /** filtert die aufgelisteten Objekte nach dem Objekt */
    public static final String OBJECT = "Objekt";
    /** filtert die aufgelisteten Objekte nach einem erweiterten Filter */
    public static final String EXTENDED = "Erweitert";
    /** Der Debug-Logger der Klasse */
    private static Debug _debug = Debug.getLogger();
    /** speichert das Kriterium des Filters */
    private String _criteria;
    /** speichert die Einträge zum Kriterium */
    private String[] _values;
    /** speichert die Verbindung zum Datenverteiler */
    private ClientDavInterface _connection;

    /**
     * Erzeugt ein neues Objekt der Klasse {@code Filter}.
     *
     * @param criteria   Filterkriterium
     * @param values     Filterattribute
     * @param connection Verbindung zum Datenverteiler
     */
    public Filter(String criteria, String[] values, ClientDavInterface connection) {
        _criteria = criteria;
        _values = values;
        _connection = connection;
    }

    /**
     * Die übergebenen Systemobjekte werden entsprechend des Kriteriums gefiltert und zurückgegeben.
     *
     * @param systemObjects die zu filternden Systemobjekte
     *
     * @return die gefilterten Systemobjekte
     */
    public Collection<SystemObject> filterObjects(Collection<SystemObject> systemObjects) {
        Collection<SystemObject> tempObjects;
        if (systemObjects == null) {
            tempObjects = new LinkedList<>();
        } else {
            tempObjects = applyFilter(systemObjects);
        }
        return Collections.unmodifiableCollection(tempObjects);
    }

    /**
     * Die Systemobjekte werden in Abhängigkeit vom Kriterium gefiltert.
     *
     * @param systemObjects die zu filternden System-Objekte
     *
     * @return die gefilterten System-Objekte
     */
    private Collection<SystemObject> applyFilter(Collection<SystemObject> systemObjects) {
        DataModel configuration = _connection.getDataModel();
        Collection<SystemObject> tempObjects = new HashSet<>();
	    switch (_criteria) {
		    case "Konfigurationsbereich" -> {
			    for (String value : _values) {
				    List<SystemObject> resultList;
				    try {
					    resultList = ConfigurationHelper.getObjects(value, configuration);
				    } catch (IllegalArgumentException e) {
					    _debug.warning("Zum Wert " + value + " kein passendes Objekt gefunden!");
					    resultList = new ArrayList<>();
				    }
				    if (!resultList.isEmpty()) {
					    if (resultList.size() > 1) {
						    _debug.finest("Anzahl der erstellten ObjekttypenObjekte: " + resultList.size());
					    }
					    for (final Object aResultList : resultList) {
						    ConfigurationArea configurationArea = (ConfigurationArea) aResultList;
						    for (SystemObject systemObject : systemObjects) {
							    ConfigurationArea confArea = systemObject.getConfigurationArea();
							    if (confArea == configurationArea) {
								    tempObjects.add(systemObject);
							    }
						    }
					    }
				    } else {
					    _debug.warning("Zum Wert " + value + " kein passendes Objekt gefunden!");
				    }
			    }
		    }
		    case "Objekttyp" -> {
			    for (String value : _values) {
				    List<SystemObject> resultList;
				    try {
					    resultList = ConfigurationHelper.getObjects(value, configuration);
				    } catch (IllegalArgumentException e) {
					    _debug.warning("Zum Wert " + value + " kein passendes Objekt gefunden!");
					    resultList = new ArrayList<>();
				    }
				    if (!resultList.isEmpty()) {
					    if (resultList.size() > 1) {
						    _debug.finest("Anzahl der erstellten ObjekttypenObjekte: " + resultList.size());
					    }
					    // Typen und Subtypen sammeln, isOfType is zu langsam für jedes einzelne Objekt
					    final Set<SystemObjectType> relevantTypes = new LinkedHashSet<>();
					    for (final Object aResultList : resultList) {
						    SystemObjectType systemObjectType = (SystemObjectType) aResultList;
						    addSubTypes(relevantTypes, systemObjectType);
					    }
					    for (SystemObject systemObject : systemObjects) {
						    if (relevantTypes.contains(systemObject.getType())) {
							    tempObjects.add(systemObject);
						    }
					    }
				    } else {
					    _debug.warning("Zum Wert " + value + " kein passendes Objekt gefunden!");
				    }
			    }
		    }
		    case "Attributgruppe" -> {
			    // AttributGruppen für alle Values holen
			    for (String value : _values) {
				    List<SystemObject> resultList;
				    try {
					    resultList = ConfigurationHelper.getObjects(value, configuration);
				    } catch (IllegalArgumentException e) {
					    _debug.warning("Zum Wert " + value + " kein passendes Objekt gefunden!");
					    resultList = new ArrayList<>();
				    }
				    if (!resultList.isEmpty()) {
					    if (resultList.size() > 1) {
						    _debug.finest("Anzahl der erstellten AttributgruppenObjekte: " + resultList.size());
					    }
					    for (final Object aResultList : resultList) {
						    AttributeGroup attributeGroup = (AttributeGroup) aResultList;
						    for (SystemObject systemObject : systemObjects) {
							    List<AttributeGroup> atgs = systemObject.getType().getAttributeGroups();
							    if (atgs.contains(attributeGroup)) {
								    tempObjects.add(systemObject);
							    }
						    }
					    }
				    } else {
					    _debug.warning("Zum Wert " + value + " kein passendes Objekt gefunden!");
				    }
			    }
		    }
		    case "Aspekt" -> {
			    for (String value : _values) {
				    List<SystemObject> resultList;
				    try {
					    resultList = ConfigurationHelper.getObjects(value, configuration);
				    } catch (IllegalArgumentException e) {
					    _debug.warning("Zum Wert " + value + " kein passendes Objekt gefunden!");
					    resultList = new ArrayList<>();
				    }
				    if (!resultList.isEmpty()) {
					    if (resultList.size() > 1) {
						    _debug.finest("Anzahl der erstellten AspektObjekte: " + resultList.size());
					    }
					    for (final Object aResultList : resultList) {
						    Aspect aspect = (Aspect) aResultList;
						    for (SystemObject systemObject : systemObjects) {
							    List<AttributeGroup> atgs = systemObject.getType().getAttributeGroups();
							    for (final AttributeGroup atg : atgs) {
								    Collection<Aspect> aspects = atg.getAspects();
								    if (aspects.contains(aspect)) {
									    tempObjects.add(systemObject);
								    }
							    }
						    }
					    }
				    } else {
					    _debug.warning("Zum Wert " + value + " kein passendes Objekt gefunden!");
				    }
			    }
		    }
		    case "Objekt" -> {
			    // Objekt holen zur Pid und dann alle durchlaufen und vergleichen
			    for (String value : _values) {
				    List<SystemObject> resultList;
				    try {
					    resultList = ConfigurationHelper.getObjects(value, configuration);
				    } catch (IllegalArgumentException e) {
					    _debug.warning("Zum Wert " + value + " kein passendes Objekt gefunden!");
					    resultList = new ArrayList<>();
				    }
				    if (!resultList.isEmpty()) {
					    if (resultList.size() > 1) {
						    _debug.finest("Anzahl der erstellten Objekte: " + resultList.size());
					    }
					    for (final SystemObject aResultList : resultList) {
						    if (systemObjects.contains(aResultList)) {
							    tempObjects.add(aResultList);
						    }
					    }
				    } else {
					    _debug.warning("Zum Wert " + value + " kein passendes Objekt gefunden!");
				    }
			    }
		    }
		    case "Erweitert" -> {
			    try {
				    // Klassenname mit Pfad angeben z.B. sys.funclib.preselection.test.Filtertest
				    String str = _values[0];
				    Class<?> c = Class.forName(str);
				    ExtendedFilter extendedFilter = (ExtendedFilter) c.getDeclaredConstructor().newInstance();
				    extendedFilter.setValues(_values);
				    extendedFilter.setConnection(_connection);
				    tempObjects = extendedFilter.applyFilter(systemObjects);
			    } catch (NoSuchMethodException | InstantiationException | IllegalAccessException |
			             ClassNotFoundException | InvocationTargetException e) {
				    e.printStackTrace();
			    }
		    }
		    default -> {
			    _debug.error("Dieses Kriterium '" + _criteria + "' wird nicht unterstützt. Verwenden Sie dafür einen Erweiterten Filter.");
			    tempObjects = systemObjects;
		    }
        }
        return tempObjects;
    }

    private void addSubTypes(final Set<SystemObjectType> relevantTypes, final SystemObjectType systemObjectType) {
        if (relevantTypes.contains(systemObjectType)) {
            return;
        }
        List<SystemObjectType> subTypes = systemObjectType.getSubTypes();
        for (SystemObjectType subType : subTypes) {
            addSubTypes(relevantTypes, subType);
        }
        relevantTypes.add(systemObjectType);
    }
}
