/*
 * Copyright 2006-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.testumg.
 *
 * de.kappich.pat.testumg is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.testumg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.testumg.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.testumg.util;

import de.bsvrz.dav.daf.accessControl.AccessControlMode;
import de.bsvrz.dav.daf.main.config.ConfigurationArea;
import de.bsvrz.dav.daf.main.config.ConfigurationChangeException;
import de.bsvrz.dav.daf.main.config.ConfigurationTaskException;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.management.ConfigAreaAndVersion;
import de.bsvrz.dav.daf.main.config.management.ConfigurationControl;
import de.bsvrz.dav.daf.main.config.management.consistenycheck.ConsistencyCheckResultInterface;
import de.bsvrz.puk.config.configFile.datamodel.ConfigDataModel;
import de.bsvrz.puk.config.main.managementfile.ConfigurationManagementFile;
import de.bsvrz.puk.config.main.managementfile.ManagementFile;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.LinkedList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Diese Klasse stellt Methoden zum Starten, Neustarten und Beenden des Datenmodells auf Konfigurationsseite zur Verfügung. Versorgungsdateien können
 * importiert, aktiviert und exportiert werden.
 * <p>
 * Zu importierende Versorgungsdateien können in ein dafür vorgesehenes {@link #getImportPath() Verzeichnis} kopiert werden. <br> Exportierte
 * Versorgungsdateien können aus einem dafür vorgesehenes {@link #getExportPath() Verzeichnis} kopiert werden.
 *
 * @author Kappich Systemberatung
 * @version $Revision:3804 $
 */
public class ConfigurationController {

    private static final Pattern AREA_AND_VERSION_PATTERN = Pattern.compile("^(.*):([0-9]+)$");

    private final Class _testClass;

    private final File _pathBase;

    private final File _adminFile;

    private File _importPath;

    private File _exportPath;

    private DataModel _dataModel;

    /** Wird gesetzt, sobald das Objekt zum ersten mal angefordert wird. */
    private DaVStarter _davStarter;

    public ConfigurationController(final File adminFile) {
        _testClass = null;
        _pathBase = null;
        _adminFile = adminFile;
    }

    /**
     * Konstruktor erstellt eine Verzeichnisstruktur für Tests mit der Konfiguration. Die {@link #getPathBase() Verzeichnis-Basis} besteht aus dem
     * Klassennamen. Darunter gibt es ein Verzeichnis für die Konfiguration, eines für den {@link #getImportPath() Import} und eines für den {@link
     * #getExportPath() Export}.
     * <p>
     * Der angegebene Klassenname sollte der Name der Testklasse sein. Im Verzeichnis der Testklasse befindliche Versorgungsdateien können dann in das
     * Import-Verzeichnis {@link #copyImportFile(String) kopiert} werden. Ein bestehendes Verzeichnis mit gleichem Namen wird gelöscht!
     *
     * @param className Klassenname inkl. Package der Testklasse (z.B. Testklasse.class.getName()).
     */
    public ConfigurationController(final String className) {
        this(className, null);
    }

    /**
     * Konstruktor erstellt eine Verzeichnisstruktur für Tests mit der Konfiguration. Die {@link #getPathBase() Verzeichnis-Basis} besteht aus dem
     * Klassennamen. Darunter gibt es ein Verzeichnis für die Konfiguration, eines für den {@link #getImportPath() Import} und eines für den {@link
     * #getExportPath() Export}.
     * <p>
     * Der angegebene Klassenname sollte der Name der Testklasse sein. Im Verzeichnis der Testklasse befindliche Versorgungsdateien können dann in das
     * Import-Verzeichnis {@link #copyImportFile(String) kopiert} werden. Ein bestehendes Verzeichnis mit gleichem Namen wird gelöscht!
     *
     * @param className     Klassenname inkl. Package der Testklasse (z.B. Testklasse.class.getName()).
     * @param baseDirectory Verzeichnis in dem die temporären Dateien angelegt werden sollen oder {@code null}, falls ein Verzeichnis entsprechend des
     *                      Klassennamens im aktuellen Arbeitsverzeichnis gewählt werden soll.
     */
    public ConfigurationController(final String className, File baseDirectory) {
        try {
            // angegebene Pfad muss eine Klasse sein
            _testClass = Class.forName(className);
            if (baseDirectory == null) {
                _pathBase = new File(TempDirectoryCreator.createTemporaryDirectory().toFile(), _testClass.getSimpleName()).getAbsoluteFile();
            } else {
                _pathBase = baseDirectory;
            }

            // bestehendes Testverzeichnis löschen
            if (_pathBase.exists()) {
                FileCopy.deleteDirectoryOrFile(_pathBase);
            }
            final File configurationPath = new File(_pathBase, "konfiguration").getCanonicalFile();

            // bestehende Konfiguration kopieren
            FileCopy.copyTestConfigurationAreaFiles(configurationPath);

            // Verwaltungsdaten-Datei
            _adminFile = new File(configurationPath, "verwaltungsdaten.xml");

            // Verzeichnis zum Import und Export festlegen
            setImportPath(new File(_pathBase, "import"));
            setExportPath(new File(_pathBase, "export"));
        } catch (Exception ex) {
            throw new IllegalStateException(ex);
        }
    }

    /**
     * Beendet die Konfiguration und löscht das Verzeichnis, welches die Konfiguration enthält und über den {@link #ConfigurationController(String)}
     * gesetzt wird.
     */
    public void closeAndDeleteConfiguration() {
        closeConfiguration();
        if (_pathBase != null) {
            System.out.println("Verzeichnis " + _pathBase.getName() + " wird gelöscht.");
            FileCopy.deleteDirectoryOrFile(_pathBase);
        } else {
            throw new IllegalStateException(
                "Das Stammverzeichnis der Konfiguration wurde nicht im Konstruktor gesetzt und kann deshalb nicht gelöscht werden.");
        }
    }

    /**
     * Startet die Konfiguration mit dem angegebenen Konfigurationsverantwortlichen.
     *
     * @param authorityPid die PId des Konfigurationsverantwortlichen
     *
     * @throws IOException Falls die Konfiguration nicht gestartet werden kann.
     */
    public void startConfiguration(final String authorityPid) throws IOException {
        // Konfigurationsverantwortlichen ändern
        final ConfigurationManagementFile managementFile = new ManagementFile(_adminFile);
        managementFile.setConfigurationAuthority(authorityPid);
        managementFile.close();

        System.out.println("Konfiguration wird mit " + authorityPid + " gestartet.");
        // Konfiguration starten
        loadConfiguration();
    }

    /**
     * Startet die Konfiguration mit dem Konfigurationsverantwortlichen {@code kv.testKonfiguration}.
     *
     * @throws IOException Falls die Konfiguration nicht gestartet werden kann.
     */
    public void startConfigurationWithTestAuthority() throws IOException {
        startConfiguration("kv.testKonfiguration");
    }

    public DataModel getDataModel() {
        return _dataModel;
    }

    /** Startet die Konfiguration. */
    private DataModel loadConfiguration() {
        System.out.println("Konfiguration wird gestartet");
        _dataModel = new ConfigDataModel(_adminFile);
        return _dataModel;
    }

    /** Startet die Konfiguration neu. */
    public DataModel reloadConfiguration() {
        // die Konfiguration wird beendet
        closeConfiguration();

        return loadConfiguration();
    }

    public void reloadConfiguration(final String authorityPid) throws IOException {
        closeConfiguration();
        startConfiguration(authorityPid);
    }

    /** Beendet die Konfiguration. */
    public void closeConfiguration() {
        System.out.println("Konfiguration wird beendet.");
        // die Lock-Dateien der Konfiguration müssen gelöscht werden
        if (_dataModel != null) {
            ((ConfigDataModel) _dataModel).close();

            // Objekte werden gelöscht
            _dataModel = null;
        }
    }

    /**
     * Aktiviert den angegebenen Konfigurationsbereich
     *
     * @param pid der zu aktivierende Konfigurationsbereich
     *
     * @return Ergebnis der Konsistenzprüfung
     *
     * @throws ConfigurationChangeException Falls Fehler beim Aktivieren der Konfigurationsbereiche auftreten.
     */
    public ConsistencyCheckResultInterface activateConfigurationArea(final String pid) throws ConfigurationChangeException {
        final Collection<String> pids = new LinkedList<>();
        pids.add(pid);
        return activateConfigurationAreas(pids);
    }

    /**
     * Aktiviert die angegebenen Konfigurationsbereiche.
     *
     * @param pids die Pids der Konfigurationsbereiche
     *
     * @return Ergebnis der Konsistenzprüfung
     *
     * @throws ConfigurationChangeException Falls Fehler beim Aktivieren der Konfigurationsbereiche auftreten.
     */
    public ConsistencyCheckResultInterface activateConfigurationAreas(final Collection<String> pids) throws ConfigurationChangeException {
        System.out.println("Bereiche werden aktiviert");
        // Bereich aktivieren
        final List<ConfigAreaAndVersion> configurationAreas = new ArrayList<>();
        for (String pid : pids) {
            final ConfigurationArea configArea = ((ConfigurationControl) getDataModel()).getAllConfigurationAreas().get(pid);
//			configurationAreas.add(new ConfigAreaAndVersion(configArea, (short)(configArea.getModifiableVersion())));
            // das Ermitteln der nächsten Version wird der activateConfigurationAreas-Methode überlassen
            configurationAreas.add(new ConfigAreaAndVersion(configArea, (short) 0));
        }
        return ((ConfigurationControl) getDataModel()).activateConfigurationAreas(configurationAreas);
    }

    /**
     * Aktiviert die angegebenen Konfigurationsbereiche.
     *
     * @param pids die Pids der Konfigurationsbereiche
     *
     * @return Ergebnis der Konsistenzprüfung
     *
     * @throws ConfigurationChangeException Falls Fehler beim Aktivieren der Konfigurationsbereiche auftreten.
     */
    public ConsistencyCheckResultInterface activate(final String... pids) throws ConfigurationChangeException {
        return activateConfigurationAreas(Arrays.asList(pids));
    }

    /**
     * Gibt den angegebenen Bereich für andere zur Aktivierung frei, ohne das dieser Bereich durch den KV des Bereichs lokal aktiviert wurde.
     *
     * @param pid Bereich, der zur Aktivierung freigegeben werden soll.
     *
     * @return Ergebnis der Konsistenzprüfung. Es darf zu Interferenzfehlern gekommen sein, bei lokalen Fehlern wurde der Bereich nicht für andere
     *     freigegeben.
     *
     * @throws ConfigurationChangeException Die Konfiguration weigert sich den Bereich frei zu geben.
     */
    public ConsistencyCheckResultInterface releaseConfigurationAreaForActivationWithoutCAActivation(final String pid)
        throws ConfigurationChangeException {
        System.out.println("Bereich wird aktiviert, ohne das dieser vorher durch den KV lokal aktiviert wurde");
        final List<ConfigAreaAndVersion> configurationAreas = new ArrayList<>();
        final ConfigurationArea area = getDataModel().getConfigurationArea(pid);

        configurationAreas.add(new ConfigAreaAndVersion(area));

        return ((ConfigurationControl) getDataModel()).releaseConfigurationAreasForActivationWithoutCAActivation(configurationAreas);
    }

    /**
     * Gibt den angegebenen Bereich für andere zur Aktivierung frei. Damit dies geschehen kann, muss der Bereich vorher durch den KV des Bereichs
     * lokal aktiviert worden sein.
     *
     * @param pid Bereich, der für andere zur Aktivierung freigegeben werden soll.
     *
     * @throws ConfigurationChangeException Die Konfiguration weigert sich den Bereich frei zu geben.
     */
    public void releaseConfigurationAreaForActivation(final String pid) throws ConfigurationChangeException {
        final List<String> pids = new ArrayList<>();
        pids.add(pid);
        releaseConfigurationAreasForActivation(pids);
    }

    public void releaseConfigurationAreasForActivation(final Collection<String> pids) throws ConfigurationChangeException {
        System.out.println("Bereich(e) für andere zur Aktivierung freigeben");
        final List<ConfigAreaAndVersion> configurationAreas = new ArrayList<>();

        for (String pid : pids) {
            final ConfigurationArea area = getDataModel().getConfigurationArea(pid);
            configurationAreas.add(new ConfigAreaAndVersion(area));
        }

        ((ConfigurationControl) getDataModel()).releaseConfigurationAreasForActivation(configurationAreas);
    }

    public void releaseConfigurationAreaForTransfer(final String pid) throws ConfigurationChangeException {
        final Collection<String> helper = new ArrayList<>();
        helper.add(pid);
        releaseConfigurationAreasForTransfer(helper);
    }

    public void releaseConfigurationAreasForTransfer(final Collection<String> pids) throws ConfigurationChangeException {

        final List<ConfigAreaAndVersion> configurationAreas = new ArrayList<>();
        for (String pid : pids) {
            final ConfigurationArea area = getDataModel().getConfigurationArea(pid);
            configurationAreas.add(new ConfigAreaAndVersion(area));
        }
        ((ConfigurationControl) getDataModel()).releaseConfigurationAreasForTransfer(configurationAreas);
    }

    /**
     * Kopiert die angegebenen Versorgungsdateien und importiert die Konfigurationsbereiche in die aktuelle Konfiguration.
     *
     * @param pids die Pids der Konfigurationsbereiche
     *
     * @throws ConfigurationChangeException Falls die Bereiche nicht importiert werden konnten.
     */
    public void importConfigurationAreas(final Collection<String> pids) throws ConfigurationChangeException {
        importConfigurationAreas(getImportPath(), pids);
    }

    /**
     * Importiert die Konfigurationsbereiche in die aktuelle Konfiguration.
     *
     * @param importPath Verzeichnis, aus dem importiert werden soll
     * @param pids       die Pids der Konfigurationsbereiche
     *
     * @throws ConfigurationChangeException Falls die Bereiche nicht importiert werden konnten.
     */
    public void importConfigurationAreas(final File importPath, final Collection<String> pids) throws ConfigurationChangeException {
        ((ConfigurationControl) getDataModel()).importConfigurationAreas(importPath, pids);
    }

    /**
     * Importiert einen Konfigurationsbereich in die aktuelle Konfiguration.
     *
     * @param areaPid die Pid des Konfigurationsbereichs
     *
     * @throws ConfigurationChangeException Falls der Bereich nicht importiert werden konnte.
     */
    public void importConfigurationArea(final String areaPid) throws ConfigurationChangeException {
        final Collection<String> pids = new ArrayList<>();
        pids.add(areaPid);
        importConfigurationAreas(getImportPath(), pids);
    }

    /**
     * Kopiert die angegebenen Versorgungsdateien und importiert die Konfigurationsbereiche in die aktuelle Konfiguration. Die Konfigurationsbereiche
     * werden dabei ähnlich zu {@link #importVariant(int, String...)} in der angegebenen Version importiert.
     *
     * @param pidsWithVersion die Pids der Konfigurationsbereiche, optional mit Version. Die String sollten (wenn sie eine Version enthalten) das
     *                        Format "kb.bereich:2" oder ähnlich haben, also die Version mit einem Doppelpunkt getrennt enthalten.
     *
     * @return Pids der gerade importierten Bereiche ohne Versionsnummern (für nachfolgende Aktivierungsschritte o.ä.)
     *
     * @throws ConfigurationChangeException Falls die Bereiche nicht importiert werden konnten.
     */
    @NotNull
    public List<String> importConfigurationAreasWithVersion(final Collection<String> pidsWithVersion) throws ConfigurationChangeException {
        final List<String> pids = new ArrayList<>();
        for (String pid : pidsWithVersion) {
            int version = -1;
            Matcher matcher = AREA_AND_VERSION_PATTERN.matcher(pid);
            if (matcher.matches()) {
                version = Integer.parseInt(matcher.group(2));
                pid = matcher.group(1);
            }
            copyImportFile(pid, version);
            pids.add(pid);
        }
        importConfigurationAreas(pids);
        return pids;
    }

    /**
     * Erzeugt ein Objekt, mit dem der Datenverteiler gestartet werden kann. Das Objekt benutzt die Konfigurationsdateien, die im Verzeichnis {@link
     * #getConfigurationPath()} gespeichert sind. Wurde die Konfiguration zuvor gestartet, wird sie erst beendet.
     * <p>
     * Der mehrfache Aufruf der Methode gibt immer das selbe Objekt zurück.
     *
     * @return Objekt zum Starten des Datenverteilers
     */
    public synchronized DaVStarter getDaVStarter() {
        closeConfiguration();
        if (_davStarter == null) {
            _davStarter = new DaVStarter(getConfigurationPath(), true);
        }
        return _davStarter;
    }

    /**
     * Erzeugt ein Objekt, mit dem der Datenverteiler gestartet werden kann. Das Objekt benutzt die Konfigurationsdateien, die im Verzeichnis {@link
     * #getConfigurationPath()} gespeichert sind. Wurde die Konfiguration zuvor gestartet, wird sie erst beendet.
     * <p>
     * Der mehrfache Aufruf der Methode gibt immer das selbe Objekt zurück.
     *
     * @return Objekt zum Starten des Datenverteilers
     */
    public synchronized DaVStarter getDaVStarter(String _configurationDebugLevel, String _transmitterDebugLevel, String _paramDebugLevel) {
        closeConfiguration();
        if (_davStarter == null) {
            _davStarter = new DaVStarter(getConfigurationPath(), true, _configurationDebugLevel, _transmitterDebugLevel, _paramDebugLevel);
        }
        return _davStarter;
    }

    /**
     * Erzeugt ein Objekt, mit dem der Datenverteiler gestartet werden kann. Das Objekt benutzt die Konfigurationsdateien, die im Verzeichnis {@link
     * #getConfigurationPath()} gespeichert sind. Wurde die Konfiguration zuvor gestartet, wird sie erst beendet.
     * <p>
     * Der mehrfache Aufruf der Methode gibt immer das selbe Objekt zurück.
     *
     * @return Objekt zum Starten des Datenverteilers
     */
    public synchronized DaVStarter getDaVStarter(final String configurationDebugLevel, final String transmitterDebugLevel,
                                                 final String paramDebugLevel, final int appPort, final int davPort, final long davId,
                                                 @Nullable final String remoteConf, final AccessControlMode accessControlType,
                                                 final String... accessControlPlugIns) {
        closeConfiguration();
        if (_davStarter == null) {
            _davStarter =
                new DaVStarter(getConfigurationPath(), new File(getConfigurationPath(), "benutzerverwaltung.xml"), true, appPort, davPort, davId,
                               remoteConf, configurationDebugLevel, transmitterDebugLevel, paramDebugLevel, accessControlType, accessControlPlugIns);
        }
        return _davStarter;
    }

    /**
     * Importiert einen Konfigurationsbereich in die aktuelle Konfiguration.
     *
     * @param importPath Verzeichnis, aus dem importiert werden soll
     * @param areaPid    die Pid des Konfigurationsbereichs
     *
     * @throws ConfigurationChangeException Falls der Bereich nicht importiert werden konnte.
     */
    public void importConfigurationArea(final File importPath, final String areaPid) throws ConfigurationChangeException {
        final Collection<String> pids = new ArrayList<>();
        pids.add(areaPid);
        importConfigurationAreas(importPath, pids);
    }

    /**
     * Exportiert die angegebenen Versorgungsdateien.
     *
     * @param pids die Pids der Konfigurationsbereiche
     *
     * @throws ConfigurationTaskException Falls es zu Fehlern beim Export kommt.
     */
    public void exportConfigurationAreas(final Collection<String> pids) throws ConfigurationTaskException {
        ((ConfigurationControl) getDataModel()).exportConfigurationAreas(getExportPath(), pids);
    }

    /**
     * Exportiert die angegebenen Versorgungsdateien.
     *
     * @param exportPath Verzeichnis, in das die Konfigurationsbereiche exportiert werden sollen
     * @param pids       die Pids der Konfigurationsbereiche
     *
     * @throws ConfigurationTaskException Falls es zu Fehlern beim Export kommt.
     */
    public void exportConfigurationAreas(final File exportPath, final Collection<String> pids) throws ConfigurationTaskException {
        ((ConfigurationControl) getDataModel()).exportConfigurationAreas(exportPath, pids);
    }

    /**
     * Exportiert zu einem angegebenen Bereich die Versorgungsdatei.
     *
     * @param areaPid Pid des Konfigurationsbereichs
     *
     * @throws ConfigurationTaskException Falls es zu einem Fehler beim Export kommt.
     */
    public void exportConfigurationArea(final String areaPid) throws ConfigurationTaskException {
        final Collection<String> pids = new ArrayList<>();
        pids.add(areaPid);
        exportConfigurationAreas(getExportPath(), pids);
    }

    /**
     * Exportiert zu einem angegebenen Bereich die Versorgungsdatei.
     *
     * @param exportPath Verzeichnis, in das die Konfigurationsbereiche exportiert werden sollen
     * @param areaPid    Pid des Konfigurationsbereichs
     *
     * @throws ConfigurationTaskException Falls es zu einem Fehler beim Export kommt.
     */
    public void exportConfigurationArea(final File exportPath, final String areaPid) throws ConfigurationTaskException {
        final Collection<String> pids = new ArrayList<>();
        pids.add(areaPid);
        exportConfigurationAreas(exportPath, pids);
    }

    /**
     * Die Konsistenzprüfung wird durchgeführt und das Ergebnis der Prüfung zurückgegeben. Die Bereiche werden in der Version geprüft, die sich in
     * Bearbeitung befindet.
     *
     * @param pids die Konfigurationsbereiche, die überprüft werden sollen
     *
     * @return Ergebnis der Konsistenzprüfung
     */
    public ConsistencyCheckResultInterface checkConsistency(final Collection<String> pids) {
        final ConfigurationControl control = (ConfigurationControl) getDataModel();
        final Collection<ConfigAreaAndVersion> areas = new ArrayList<>();
        for (String pid : pids) {
            final ConfigurationArea area = control.getAllConfigurationAreas().get(pid);
            short version = area.getModifiableVersion();
            areas.add(new ConfigAreaAndVersion(area, version));
        }
        return control.checkConsistency(areas);
    }

    /**
     * Importiert die Konfigurationsbereiche in die aktuelle Konfiguration.
     *
     * @param variant Zu importierende Variante der Versorgungsdateien.
     * @param pids    die Pids der Konfigurationsbereiche
     *
     * @throws ConfigurationChangeException Falls die Bereiche nicht importiert werden konnten.
     */
    public void importVariant(int variant, final String... pids) throws ConfigurationChangeException {
        copyVariant(variant, pids);
        importConfigurationAreas(getImportPath(), Arrays.asList(pids));
    }

    /**
     * Importiert und aktiviert die angegebenen Konfigurationsbereiche und lädt die Konfiguration neu.
     *
     * @param variant Zu importierende Variante der Versorgungsdateien.
     * @param pids    die Pids der Konfigurationsbereiche
     *
     * @return Objekt zum Zugriff auf die neu geladene Konfiguration.
     *
     * @throws ConfigurationChangeException Falls Fehler beim Aktivieren der Konfigurationsbereiche auftreten.
     */
    public DataModel importActivateAndReloadVariant(final int variant, final String... pids) throws ConfigurationChangeException {
        importVariant(variant, pids);
        return activateAndReload(pids);
    }

    /**
     * Aktiviert die angegebenen Konfigurationsbereiche und lädt die Konfiguration neu.
     *
     * @param pids die Pids der Konfigurationsbereiche
     *
     * @return Objekt zum Zugriff auf die neu geladene Konfiguration.
     *
     * @throws ConfigurationChangeException Falls Fehler beim Aktivieren der Konfigurationsbereiche auftreten.
     */
    public DataModel activateAndReload(final String... pids) throws ConfigurationChangeException {
        final ConsistencyCheckResultInterface consistencyCheckResult = activate(pids);
        if (!consistencyCheckResult.getLocalErrors().isEmpty()) {
            throw new ConfigurationChangeException("Lokale Fehler bei der Konsistenzprüfung");
        }
        if (!consistencyCheckResult.getInterferenceErrors().isEmpty()) {
            throw new ConfigurationChangeException("Interferenzfehler bei der Konsistenzprüfung");
        }
        reloadConfiguration();
        return getDataModel();
    }

    /**
     * Kopiert die Dateien, die importiert werden sollen. Dann werden die Dateien importiert und aktiviert. Danach wird das Datenmodell erneut geladen
     * und zurückgegeben.
     *
     * @param pids Pids der Bereiche, die importiert und aktiviert werden sollen.
     *
     * @return Datenmodel, in dem die spezifizierten Bereiche aktiviert wurden
     *
     * @see #copyImportFiles(Collection)
     * @see #importConfigurationAreas(Collection)
     * @see #activateConfigurationAreas(Collection)
     * @see #reloadConfiguration()
     */
    public DataModel importAndActivateAreas(final Collection<String> pids) throws ConfigurationChangeException {
        copyImportFiles(pids);
        importConfigurationAreas(pids);
        activateConfigurationAreas(pids);
        reloadConfiguration();
        return getDataModel();
    }

    /** @see #importAndActivateAreas(Collection) */
    public DataModel importAndActivateArea(final String pid) throws ConfigurationChangeException {
        final List<String> pidList = new ArrayList<>();
        pidList.add(pid);
        return importAndActivateAreas(pidList);
    }

    /**
     * Gibt das Stammverzeichnis der Konfiguration zurück. (z.B. ..\TestKlasse)
     *
     * @return das Stammverzeichnis der Konfiguration
     */
    public File getPathBase() {
        return _pathBase;
    }

    public File getAdminFile() {
        return _adminFile;
    }

    /**
     * Gibt das Verzeichnis der Konfigurationsdateien zurück.
     *
     * @return das Verzeichnis der Konfigurationsdateien
     */
    public File getConfigurationPath() {
        return _adminFile.getParentFile();
    }

    /**
     * Hierüber kann das Verzeichnis abgefragt werden, aus dem die Versorgungsdateien importiert werden.
     *
     * @return das Verzeichnis, aus dem die Versorgungsdateien importiert werden
     */
    public File getImportPath() {
        return _importPath;
    }

    /**
     * Setzt das Verzeichnis, aus dem die Versorgungsdateien importiert werden sollen.
     *
     * @param importPath das Verzeichnis, aus dem die Versorgungsdateien importiert werden
     */
    public void setImportPath(final File importPath) {
        try {
            _importPath = importPath.getCanonicalFile();
        } catch (IOException ex) {
            throw new IllegalArgumentException(ex);
        }
        if (!_importPath.exists()) {
            _importPath.mkdirs();
        }
        if (!_importPath.isDirectory()) {
            throw new IllegalArgumentException("Das angegebene Objekt ist kein Verzeichnis.");
        }
    }

    /**
     * Gibt das Verzeichnis zurück, in das exportiert wird.
     *
     * @return das Verzeichnis, in dem die exportierten Versorgungsdateien sind
     */
    public File getExportPath() {
        return _exportPath;
    }

    /**
     * Setzt das Verzeichnis, in das exportiert werden soll.
     *
     * @param exportPath das Verzeichnis, in dem die exportierten Versorgungsdateien sind
     */
    public void setExportPath(final File exportPath) {
        try {
            _exportPath = exportPath.getCanonicalFile();
        } catch (IOException ex) {
            throw new IllegalArgumentException(ex);
        }
        if (!_exportPath.exists()) {
            _exportPath.mkdirs();
        }
        if (!_exportPath.isDirectory()) {
            throw new IllegalArgumentException("Das angegebene Objekt ist kein Verzeichnis.");
        }
    }

    /**
     * Kopiert die Bereichsdatei, die in dem gleichen Verzeichnis stehen muss, wie die Testklasse (Name wurde im Konstruktor angegeben), in das
     * Import-Verzeichnis.
     *
     * @param pid Pid eines Konfigurationsbereichs
     */
    public void copyImportFile(final String pid) {
        final URL resource;
        try {
            final String sourceName = pid + ".xml";
            // Verzeichnis der Versorgungsdatei ermitteln, die kopiert werden soll.
            resource = _testClass.getResource(sourceName);
            if (resource != null && _importPath != null) {
                final File targetFile = new File(_importPath, sourceName);
                FileCopy.copyFile(resource.openStream(), targetFile, true);
            } else {
                throw new IllegalStateException(
                    "Kopieren der Datei " + sourceName + ": Import-Verzeichnis (" + _importPath + ") oder Versorgungsdatei (" + resource +
                    ") nicht vorhanden.");
            }
        } catch (Exception ex) {
            throw new IllegalStateException("Die Versorgungsdatei konnte nicht kopiert werden", ex);
        }
    }

    public void createConfigurationFile(final String pid, final String content) {
        createConfigurationFile(pid, new ByteArrayInputStream(content.getBytes(Charset.forName("ISO-8859-1"))));
    }

    public void createConfigurationFile(final String pid, final ByteArrayInputStream inputStream) {
        try {
            final String fileName = pid + ".xml";
            // Ziel-Verzeichnis der Versorgungsdatei ermitteln
            final File targetFile = new File(_importPath, fileName);
            FileCopy.copyFile(inputStream, targetFile, true);
        } catch (Exception ex) {
            throw new IllegalStateException("Die Versorgungsdatei konnte nicht erstellt werden", ex);
        }
    }

    /**
     * Diese Methode kopiert eine Datei, die im gleichen Verzeichnis steht, wie die Testklasse (Name wurde im Konstruktor angegeben), mit einem
     * anderen Dateinamen in das Import-Verzeichnis.
     * <p>
     * Der Name der zu kopiernden Datei besitzt folgenden Aufbau (+ wird als Stringkonkatenation verstanden): pid+version+.xml
     * <p>
     * Die Kopie besitzt folgenden Aufbau: pid+.xml
     * <p>
     * Beispiel: aus kb.test2.xml wird kb.test.xml
     * <p>
     * Damit ist es möglich, nacheinander Bereichsdateien zu importieren, die den gleichen Bereich betreffen.
     *
     * @param pid     Erste Teil des Dateinamens. Dieser wird benutzt um die zu kopierende Datei zu erkennen und der Kopie den richtigen Namen zu
     *                geben.
     * @param version Version des Bereichs. Bei Zahlen &gt;= 0 wird nach einer Datei pid+version+.xml gesucht. Bei Zahlen &lt; 0 wird nach pid+.xml
     *                gesucht.
     */
    public void copyImportFile(final String pid, final int version) {
        try {
            final String sourceName;
            if (version >= 0) {
                sourceName = pid + version + ".xml";
            } else {
                sourceName = pid + ".xml";
            }

            final URL resource = _testClass.getResource(sourceName);
            if (resource != null) {
                final File targetFile = new File(getImportPath(), pid + ".xml");
                FileCopy.copyFile(resource.openStream(), targetFile, true);
            }
        } catch (IOException e) {
            throw new IllegalStateException("Die Versorgungsdatei konnte nicht kopiert werden.", e);
        }
    }

    /**
     * Kopiert die Bereichsdateien, die in dem gleichen Verzeichnis stehen, wie diese Testklasse, in das Import-Verzeichnis. Die Versionsnummer gibt
     * die Dateierweiterung der Bereichsdatei an, die verwendet werden soll. Diese wird dann entfernt.
     * <p>
     * Beispiel: aus kb.test2.xml wird kb.test.xml
     * <p>
     * Damit ist es möglich, nacheinander Bereichsdateien zu importieren, die den gleichen Bereich betreffen.
     *
     * @param pids    Pids der Konfigurationsbereiche
     * @param version die Dateiversion
     */
    public void copyImportFiles(final Collection<String> pids, final int version) {
        for (String pid : pids) {
            copyImportFile(pid, version);
        }
    }

    /**
     * Kopiert die Bereichsdateien, die in dem gleichen Verzeichnis stehen, wie diese Testklasse, in das Import-Verzeichnis. Die Versionsnummer gibt
     * die Dateierweiterung der Bereichsdatei an, die verwendet werden soll. Diese wird dann entfernt.
     * <p>
     * Beispiel: aus kb.test2.xml wird kb.test.xml
     * <p>
     * Damit ist es möglich, nacheinander Bereichsdateien zu importieren, die den gleichen Bereich betreffen.
     *
     * @param pids    Pids der Konfigurationsbereiche
     * @param variant Zu importierende Variante der Versorgungsdateien.
     */
    public void copyVariant(final int variant, final String... pids) {
        for (String pid : pids) {
            copyImportFile(pid, variant);
        }
    }

    /**
     * Kopiert die Bereichsdateien, die in dem gleichen Verzeichnis stehen muss, wie die Testklasse (Name wurde im Konstruktor angegeben), in das
     * Import-Verzeichnis.
     *
     * @param pids Pids der zu kopierenden Konfigurationsbereiche
     */
    public void copyImportFiles(final Collection<String> pids) {
        for (String pid : pids) {
            copyImportFile(pid);
        }
    }
}
