/*
 * Copyright 2006-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.testumg.
 *
 * de.kappich.pat.testumg is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.testumg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.testumg.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.testumg.util;

import static de.kappich.pat.testumg.util.ReleaseVersion.Current;
import static de.kappich.pat.testumg.util.ReleaseVersion.V3_5_5;
import static de.kappich.pat.testumg.util.ReleaseVersion.V3_6_5;


import com.google.common.collect.ImmutableList;
import de.bsvrz.dav.daf.accessControl.AccessControlMode;
import de.bsvrz.dav.daf.communication.lowLevel.ServerConnectionInterface;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.NormalCloser;
import de.bsvrz.dav.daf.main.authentication.ClientCredentials;
import de.bsvrz.dav.dav.main.Transmitter;
import java.io.BufferedReader;
import java.io.EOFException;
import java.io.File;
import java.io.IOException;
import java.io.InputStreamReader;
import java.net.InetAddress;
import java.net.Socket;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Timer;
import java.util.TimerTask;
import java.util.stream.Collectors;

/**
 * Diese Klasse startet den Datenverteiler, die Konfiguration und die Parametrierung.
 * <p> Es werden typischerweise drei Prozesse gestartet, die über eine Methode beendet werden können.
 * <p> Hinweis: Wird die Methode {@link #stopDaV()} oder {@link #stopDavWithoutFileDeletion()} nicht aufgerufen, so laufen die Prozesse weiter, auch
 * wenn die VM, in der die Prozesse gestartet wurden beendet wird.
 * <p> Dies kann zum Beispiel bei einem fehlerhaften JUnit-Test geschehen, der nicht mehr in TearDown oder @after kommt. In diesem Fall sind 3 Java
 * Prozesse vorhanden, die mittels Task-Manager (kill) beendet werden müssen.
 * <p> Hinweis: Diese Klasse arbeitet auf einem relativ niedrigen Level und ist umständlich zu benutzen. Für gewöhnliche Tests sollte stattdessen
 * eine
 * der folgenden komfortableren Klassen verwendet werden:
 * <ul>
 *     <li>{@link SingleDavStarter} für Tests mit einem Datenverteiler</li>
 *     <li>{@link MultiDavTestEnvironment} für Tests mit beliebig vielen Datenverteilern</li>
 * </ul>
 * <p>Diese erledigen eine Konfigurationsarbeiten automatisch und sind einfacher zu benutzen.
 *
 * @author Achim Wullenkord (AW), Kappich Systemberatung
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class DaVStarter {

    public static final String DEFAULT_CONFIGURATION_DEBUG = "WARNING";
    public static final String DEFAULT_TRANSMITTER_DEBUG = "WARNING";
    public static final String DEFAULT_PARAM_DEBUG = "WARNING";
    private final File _userFile;
    private final int _davAppPort;
    private final int _davDavPort;
    private final long _davID;
    private final String _remoteConfiguration;
    private final String _configurationDebugLevel;
    private final String _transmitterDebugLevel;
    private final String _paramDebugLevel;
    private final AccessControlMode _accessControlType;
    private final String[] _accessControlPlugIns;
    private final File _passwd;
    private final File _configurationManagementFile;
    private final File _paramDirectory;
    /**
     * In diesem Verzeichnis wird eine Kopie von Dateien erstellt, die zum starten des Datenverteilers benötigt werden. Wurde keine Kopie erstellt,
     * ist diese Variabel {@code null}.
     */
    private final File _workingDirectory;
    private ReleaseVersion _releaseVersion = Current;
    private Process _transmitter;
    private Process _configuration;
    private Process _param;
    private Process _operatingMessageManagement;
    private String _name = "";
    /**
     * Verzögerungszeit, die innerhalb des Datenverteilers gewartet wird, bevor Verbindungen zu anderen Datenverteilern zugelassen bzw. aufgebaut
     * werden.
     */
    private int _davDavConnectDelay;
    /** Verzögerungszeit, die innerhalb des Datenverteilers gewartet wird, bevor versucht wird, abgebrochene Verbindungen neu aufzubauen. */
    private int _davDavReconnectDelay = 1000;
    /** Kommunikationsprotokoll-Klasse */
    private Class<? extends ServerConnectionInterface> _protocolClass;
    private String[] _additionalTransmitterArgs = new String[0];
    private Transmitter _transmitterObject;
    private String _debugName = "Datenverteiler";
    private volatile FakeParamApp _fakeParamApp = new FakeParamApp();
    private String[] _classPathOverride;
    private Object _protocolParameter;
    private String _userNameParam;
    private String _userNameConfiguration;
    private String _userNameTransmitter;
    private int _passivePort;
    private ImmutableList<Integer> _activePorts = ImmutableList.of();

    /**
     * Startet Datenverteiler, Konfiguration und die Parametrierung mit Default-Dateien und einer Benutzerdefinierten Datei, die alle Benutzer
     * enthält, die sich beim Datenverteiler anmelden können.
     *
     * @param workingDirectory Ein Verzeichnis in dem sich alle Dateien befinden, mit dem der Datenverteiler gestartet werden soll.
     * @param userFile         Datei, die alle Benutzer enthält, die sich beim Datenverteiler anmelden können. Die übergebene Datei kann durch den
     *                         Datenverteiler geändert werden und wird beim beenden des Datenverteilers nicht gelöscht.
     */
    public DaVStarter(final File workingDirectory, File userFile) {
        this(workingDirectory, userFile, false, 8083 + getDavPortNumberOffset(), -1, 10001, null);
    }

    /**
     * Startet den Datenverteiler, Konfiguration und die Parametrierung mit default-Werten.
     *
     * @param workingDirectory Verzeichnis, in das alle Dateien kopiert werden, die zum starten des Datenverteilers benötigt werden.
     */
    public DaVStarter(final File workingDirectory) {
        this(workingDirectory, false);
    }

    /**
     * Startet den Datenverteiler, Konfiguration und die Parametrierung mit default-Werten/Dateien oder mit übergebenen Dateien. <p> In beiden Fällen
     * wird das übergebene Verzeichnis vollständig beim herunterfahren des Datenverteilers gelöscht !
     *
     * @param workingDirectory Ein Verzeichnis in dem sich entweder alle Dateien befinden, mit dem der Datenverteiler gestartet werden soll, oder aber
     *                         es befinden sich keine Dateien in diesem Verzeichnis. Im zweiten Fall werden die benötigten Dateien in das Verzeichnis
     *                         kopiert. In beiden Fällen wird beim Herunterfahren des Datenverteilers das Verzeichnis mit allen Daten gelöscht.
     *                         Befanden sich vor dem Start des Datenverteilers Dateien in dem Verzeichnis, werden diesen ebenfalls gelöscht.
     * @param containsDaVFiles true = Das Verzeichnis enthält alle Dateien, die für den Start des Datenverteiles benötigt werden; false = Das
     *                         Verzeichnis enthält nicht die Dateien, die zum Start des Datenverteilers benötigt werden
     */
    public DaVStarter(final File workingDirectory, boolean containsDaVFiles) {
        this(workingDirectory, new File(workingDirectory, "benutzerverwaltung.xml"), containsDaVFiles, 8083 + getDavPortNumberOffset(), -1, 10001,
             null);
    }

    /**
     * Startet den Datenverteiler, Konfiguration und die Parametrierung mit default-Werten/Dateien oder mit übergebenen Dateien. <p> In beiden Fällen
     * wird das übergebene Verzeichnis vollständig beim herunterfahren des Datenverteilers gelöscht !
     *
     * @param workingDirectory        Ein Verzeichnis in dem sich entweder alle Dateien befinden, mit dem der Datenverteiler gestartet werden soll,
     *                                oder aber es befinden sich keine Dateien in diesem Verzeichnis. Im zweiten Fall werden die benötigten Dateien in
     *                                das Verzeichnis kopiert. In beiden Fällen wird beim Herunterfahren des Datenverteilers das Verzeichnis mit allen
     *                                Daten gelöscht. Befanden sich vor dem Start des Datenverteilers Dateien in dem Verzeichnis, werden diesen
     *                                ebenfalls gelöscht.
     * @param containsDaVFiles        true = Das Verzeichnis enthält alle Dateien, die für den Start des Datenverteiles benötigt werden; false = Das
     *                                Verzeichnis enthält nicht die Dateien, die zum Start des Datenverteilers benötigt werden
     * @param configurationDebugLevel Debug-Level, der von der Konfiguration verwendet werden soll
     * @param transmitterDebugLevel   Debug-Level, der vom Datenverteiler verwendet werden soll
     * @param paramDebugLevel         Debug-Level, der von der Parametrierung verwendet werden soll
     */
    public DaVStarter(final File workingDirectory, boolean containsDaVFiles, String configurationDebugLevel, String transmitterDebugLevel,
                      String paramDebugLevel) {
        this(workingDirectory, new File(workingDirectory, "benutzerverwaltung.xml"), containsDaVFiles, 8083 + getDavPortNumberOffset(), -1, 10001,
             null, configurationDebugLevel, transmitterDebugLevel, paramDebugLevel, AccessControlMode.Disabled, new String[0]);
    }

    /**
     * Startet den Datenverteiler, Konfiguration und die Parametrierung mit default-Werten/Dateien oder mit übergebenen Dateien. <p> In beiden Fällen
     * wird das übergebene Verzeichnis vollständig beim herunterfahren des Datenverteilers gelöscht !
     *
     * @param workingDirectory        Ein Verzeichnis in dem sich entweder alle Dateien befinden, mit dem der Datenverteiler gestartet werden soll,
     *                                oder aber es befinden sich keine Dateien in diesem Verzeichnis. Im zweiten Fall werden die benötigten Dateien in
     *                                das Verzeichnis kopiert. In beiden Fällen wird beim Herunterfahren des Datenverteilers das Verzeichnis mit allen
     *                                Daten gelöscht. Befanden sich vor dem Start des Datenverteilers Dateien in dem Verzeichnis, werden diesen
     *                                ebenfalls gelöscht.
     * @param containsDaVFiles        true = Das Verzeichnis enthält alle Dateien, die für den Start des Datenverteiles benötigt werden; false = Das
     *                                Verzeichnis enthält nicht die Dateien, die zum Start des Datenverteilers benötigt werden
     * @param configurationDebugLevel Debug-Level, der von der Konfiguration verwendet werden soll
     * @param transmitterDebugLevel   Debug-Level, der vom Datenverteiler verwendet werden soll
     * @param paramDebugLevel         Debug-Level, der von der Parametrierung verwendet werden soll
     * @param accessControlType       Ob die Rechteverwaltung aktiv sein soll
     * @param accessControlPlugIns    Rechteverwaltungsplugins, die geladen werden sollen
     */
    public DaVStarter(final File workingDirectory, boolean containsDaVFiles, String configurationDebugLevel, String transmitterDebugLevel,
                      String paramDebugLevel, final AccessControlMode accessControlType, final String[] accessControlPlugIns) {
        this(workingDirectory, new File(workingDirectory, "benutzerverwaltung.xml"), containsDaVFiles, 8083 + getDavPortNumberOffset(), -1, 10001,
             null, configurationDebugLevel, transmitterDebugLevel, paramDebugLevel, accessControlType, accessControlPlugIns);
    }

    /**
     * Startet den Datenverteiler. <p> Das übergebene Verzeichnis wird vollständig beim herunterfahren des Datenverteilers gelöscht !
     *
     * @param workingDirectory    Ein Verzeichnis in dem sich alle Dateien befinden, mit dem der Datenverteiler gestartet werden soll. Beim
     *                            Herunterfahren des Datenverteilers wird das Verzeichnis mit allen Daten gelöscht. Befanden sich vor dem Start des
     *                            Datenverteilers Dateien in dem Verzeichnis, werden diesen ebenfalls gelöscht.
     * @param davAppPort          Der Port, der vom Datenverteiler für den Verbindungsaufbau der Applikation zur Verfügung gestellt wird.
     * @param davId               Die ID des Datenverteilers.
     * @param remoteConfiguration Vom datenverteiler benutzte Remotekonfiguration.
     */
    public DaVStarter(final File workingDirectory, int davAppPort, long davId, String remoteConfiguration) {
        this(workingDirectory, new File(workingDirectory, "benutzerverwaltung.xml"), true, davAppPort, -1, davId, remoteConfiguration);
    }

    /**
     * Startet den Datenverteiler. <p> Das übergebene Verzeichnis wird vollständig beim herunterfahren des Datenverteilers gelöscht !
     *
     * @param workingDirectory    Ein Verzeichnis in dem sich alle Dateien befinden, mit dem der Datenverteiler gestartet werden soll. Beim
     *                            Herunterfahren des Datenverteilers wird das Verzeichnis mit allen Daten gelöscht. Befanden sich vor dem Start des
     *                            Datenverteilers Dateien in dem Verzeichnis, werden diesen ebenfalls gelöscht.
     * @param davAppPort          Der Port, der vom Datenverteiler für den Verbindungsaufbau der Applikation zur Verfügung gestellt wird.
     * @param davDavPort          Der Port, der vom Datenverteiler für den Verbindungsaufbau der Datenverteiler zur Verfügung gestellt wird.
     * @param davId               Die ID des Datenverteilers.
     * @param remoteConfiguration Vom datenverteiler benutzte Remotekonfiguration.
     */
    public DaVStarter(final File workingDirectory, int davAppPort, final int davDavPort, long davId, String remoteConfiguration) {
        this(workingDirectory, new File(workingDirectory, "benutzerverwaltung.xml"), true, davAppPort, davDavPort, davId, remoteConfiguration);
    }

    public DaVStarter(final File workingDirectory, final File userFile, boolean containsDaVFiles, int davAppPort, final int davDavPort, long davId,
                      String remoteConfiguration) {
        this(workingDirectory, userFile, containsDaVFiles, davAppPort, davDavPort, davId, remoteConfiguration, DEFAULT_CONFIGURATION_DEBUG,
             DEFAULT_TRANSMITTER_DEBUG, DEFAULT_PARAM_DEBUG, AccessControlMode.Disabled, new String[0]);
    }

    public DaVStarter(final File workingDirectory, final File userFile, boolean containsDaVFiles, int davAppPort, final int davDavPort, long davId,
                      String remoteConfiguration, String configurationDebugLevel, String transmitterDebugLevel, String paramDebugLevel,
                      final AccessControlMode accessControlType, final String[] accessControlPlugIns) {

        addRuntimeExitHandler();

        _davAppPort = davAppPort;
        _davDavPort = davDavPort;
        _davID = davId;
        _remoteConfiguration = remoteConfiguration;
        _configurationDebugLevel = configurationDebugLevel;
        _transmitterDebugLevel = transmitterDebugLevel;
        _paramDebugLevel = paramDebugLevel;
        _accessControlType = accessControlType;
        _accessControlPlugIns = accessControlPlugIns;

        try {
            //Prüfung, ob bereits DaV auf Port _davAppPort läuft.

            testPort();

            if (!workingDirectory.exists() && containsDaVFiles == false) {
                workingDirectory.mkdirs();
            }

            // Kopien der Dateien zur Verfügung stellen, auf denen gearbeitet wird.
            _workingDirectory = workingDirectory;

            if (containsDaVFiles == false) {
                // Die benötigten Daten müssen noch kopiert werden
                FileCopy.copyTestConfigurationAreaFiles(_workingDirectory);
            }

            _userNameTransmitter = "Tester";
            // Benutzername, mit der sich die Parametrierung anmeldet
            _userNameParam = "parameter";
            // enthält das Passwort für die Parametrierung und für die Konfiguration
            _passwd = new File(_workingDirectory, "passwd.properties");
            // Verzeichnis für die Parametrierung
            _paramDirectory = new File(_workingDirectory, "Parametrierung");
            _paramDirectory.mkdir();
            // Verwaltungsdatei für die Konfiguration
            _configurationManagementFile = new File(_workingDirectory, "verwaltungsdaten.xml");
            // Benutzername, mit dem sich die Konfiguration beim Datenverteiler anmeldet
            _userNameConfiguration = "configuration";

            _userFile = userFile;

        } catch (IOException ex) {
            throw new IllegalStateException("Der Datenverteiler konnte nicht gestartet werden", ex);
        }
    }

    public static int getDavPortNumberOffset() {
        final String defaultOffset;
        if (System.getProperty("agent.name") == null) {
            // Nicht via TeamCity gestartet
            defaultOffset = "10000";
        } else {
            // Via Teamcity gestartet
            defaultOffset = "20000";
        }
        return Integer.parseInt(System.getProperty("de.kappich.dav.testTcpPortNumberOffset", defaultOffset));
    }

    public static String join(String[] s, String delimiter) {
        if (s.length == 0) {
            return "";
        }
        StringBuilder buffer = new StringBuilder(s[0]);
        int i = 1;
        while (i < s.length) {
            buffer.append(delimiter).append(s[i]);
        }
        return buffer.toString();
    }

    /**
     * Erzeugt eine Liste von Aufrufparametern, die benutzt werden um einen Prozess mit Javac zu starten.
     *
     * @param className
     * @param xmxSize   Wieviel XMX Speicher steht dem Prozess zur Verfügung (in MB)
     *
     * @return Aufrufparameter
     */
    private static String[] createDefaultCommandArray(String className, int xmxSize) {
        return createDefaultCommandArray(className, xmxSize, null);
    }

    /**
     * Erzeugt eine Liste von Aufrufparametern, die benutzt werden um einen Prozess mit Javac zu starten.
     *
     * @param className
     * @param xmxSize           Wieviel XMX Speicher steht dem Prozess zur Verfügung (in MB)
     * @param classPathOverride
     *
     * @return Aufrufparameter
     */
    public static String[] createDefaultCommandArray(String className, int xmxSize, final String[] classPathOverride) {
        String fileSeparator = System.getProperty("file.separator");
        String javaHome = System.getProperty("java.home");
        String classPath;
        if (classPathOverride == null || classPathOverride.length == 0) {
            classPath = System.getProperty("java.class.path");
        } else {
            String delim = System.getProperty("path.separator");
            StringBuilder stringBuilder = new StringBuilder(classPathOverride[0]);
            for (int i = 1; i < classPathOverride.length; i++) {
                String s = classPathOverride[i];
                stringBuilder.append(delim);
                stringBuilder.append(s);
            }
            stringBuilder.append(delim).append(System.getProperty("java.class.path"));
            classPath = stringBuilder.toString();
        }

        List<String> commandList = new LinkedList<>();
        commandList.add(javaHome + fileSeparator + "bin" + fileSeparator + "java");
        commandList.add("-Dfile.encoding=ISO-8859-1");
        commandList.add("-Xmx" + xmxSize + "m");
        commandList.add("-cp");
        commandList.add(classPath);
        commandList.add(className);
        return commandList.toArray(new String[0]);
    }

    public static String[] mergeArrays(String[] array1, String[] array2) {
        List<String> mergedList = new ArrayList<>();
        Collections.addAll(mergedList, array1);
        Collections.addAll(mergedList, array2);
//		System.out.println("");
//		System.out.println("*********Liste start*************");
//		System.out.println("");
//
//		for(String s : mergedList) {
//			System.out.println(s);
//		}
//		System.out.println("");
//		System.out.println("**********************");
//		System.out.println("");

        //		System.out.println(mergedList);
        return mergedList.toArray(new String[0]);
    }

    private static void waitForProcessExit(final String name, final Process process, final int timeout) {
        long start = System.currentTimeMillis();
        // Da der Datenverteiler schon "abgeschossen" wurde, sollte sich diese Anwendung automatisch beenden.
        // Falls die Applikation trotzdem hängenbleibt (timeout), wird diese ebenfalls "abgeschossen" und ein Fehler erzeugt.
        try {
            // bei waitFor() kann man leider kein Timeout angeben, also wird mit einem separaten Timer-Thread und einem Interrupt nachgeholfen

            final Thread mainThread = Thread.currentThread();
            Timer timer = new Timer("Prozess " + name + " beenden - Timeout");
            timer.schedule(new TimerTask() {
                @Override
                public void run() {
                    mainThread.interrupt();
                    process.destroy();
                }
            }, timeout);

            process.waitFor();
            timer.cancel(); // process wurde erfolgreich beendet, timer abbrechen.
            long delay = System.currentTimeMillis() - start;
            System.out.println(name + " in " + delay + "ms beendet");
        } catch (InterruptedException e) {
            throw new RuntimeException(name + " hat sich nicht rechtzeitig in " + timeout + "ms beendet");
        }
    }

    public static Process createProcess(String[] commandArray, final String outputPrefix, final String errorOutputPrefix, File workingDirectory)
        throws IOException {
        ProcessBuilder processBuilder = new ProcessBuilder(commandArray);
        processBuilder.directory(workingDirectory);

        final Process process = processBuilder.start();
    
        final InputStreamReader inputStreamReaderError = new InputStreamReader(process.getErrorStream(), StandardCharsets.UTF_8);
        final InputStreamReader inputStreamReaderInput = new InputStreamReader(process.getInputStream(), StandardCharsets.UTF_8);

        final Thread threadError = new Thread(new OutputThread(inputStreamReaderError, errorOutputPrefix));
        final Thread threadInput = new Thread(new OutputThread(inputStreamReaderInput, outputPrefix));

        threadError.setDaemon(true);
        threadError.setName("ErrorOutput");
        threadInput.setDaemon(true);
        threadInput.setName("InputOutput");

        threadError.start();
        threadInput.start();

        return process;
    }
    /**
     * Setzt die Verzögerungszeit, die innerhalb des Datenverteilers gewartet wird, bevor Verbindungen zu anderen Datenverteilern zugelassen bzw.
     * aufgebaut werden.
     *
     * @param davDavConnectDelay Verzögerungszeit in Millisekunden
     */
    public void setDavDavConnectDelay(final int davDavConnectDelay) {
        _davDavConnectDelay = davDavConnectDelay;
    }

    /**
     * Setzt die Verzögerungszeit, die innerhalb des Datenverteilers gewartet wird, bevor versucht wird, abgebrochene Verbindungen neu aufzubauen.
     *
     * @param davDavReconnectDelay Verzögerungszeit in Millisekunden
     */
    public void setDavDavReconnectDelay(final int davDavReconnectDelay) {
        _davDavReconnectDelay = davDavReconnectDelay;
    }

    /**
     * Gibt den Prefix zurück, der in der Konsole vor der Anwendung gezeigt wird.
     *
     * @return Prefix
     */
    public String getName() {
        return _name;
    }

    /**
     * Setzt den Prefix, der in der Konsole vor der Anwendung gezeigt wird
     *
     * @param name Prefix
     */
    public void setName(final String name) {
        _name = name;
    }

    /**
     * Gibt den Namen mit dem die Debug-Klasse vom Datenverteiler initialisiert wird zurück
     *
     * @return den Namen mit dem die Debug-Klasse vom Datenverteiler initialisiert wird
     */
    public String getDebugName() {
        return _debugName;
    }

    /**
     * Setzt den Namen mit dem die Debug-Klasse vom Datenverteiler initialisiert wird
     *
     * @param debugName Name des Datenverteilers
     */
    public void setDebugName(final String debugName) {
        _debugName = debugName;
    }

    /**
     * Gibt eine Verbindung zum Datenverteiler zurück, in der der Benutzer "Tester" und das Passwort "geheim" benutzt wird. <p> Bei jedem Aufruf der
     * Methode wird eine neue Verbindung aufgebaut.
     *
     * @return Verbindung zum Datenverteiler
     */
    public ClientDavInterface getConnection() {
        return new CreateClientDavConnection("Tester", ClientCredentials.ofString("geheim"), null, null, "localhost", _davAppPort).getConnection();
    }

    public File getWorkingDirectory() {
        return _workingDirectory;
    }

    private void testPort() {
        boolean printed = false;
        for (int i = 0; i < 10; i++) {
            try {
                try (Socket ignored = new Socket(InetAddress.getLocalHost(), _davAppPort)) {
                    if (!printed) {
                        System.out.print("Prüfe freien Port");
                        printed = true;
                    }
                    System.out.print(".");
                    System.out.flush();
                }
            } catch (IOException ignore) {
                // Hier wird eine Exception erwartet wenn kein Datenverteiler läuft.
                return; // Hat funktioniert, als beenden
            }
        }
        throw new IllegalStateException(
            "Der Datenverteiler konnte nicht gestartet werden: Es läuft bereits eine Applikation auf Port " + _davAppPort);
    }

    /**
     * Diese Funktion sorgt dafür, dass beim Beenden der JVM auf jeden Fall gestartete Unter-Prozesse wie der Datenverteiler beendet werden
     */
    private void addRuntimeExitHandler() {
        Runtime.getRuntime().addShutdownHook(new Thread() {
            @Override
            public void run() {
                if (_transmitter != null) {
                    try {
                        // Prüfen ob Prozess beendet
                        _transmitter.exitValue();

                        // falls beendet, ist keine Aktion erforderlich.
                        return;
                    } catch (IllegalThreadStateException ignored) {
                        // Prozess noch nicht beendet
                    }
                    System.err.println("Datenverteiler wurde von den Tests nicht korrekt beendet.");
                    System.err.println("Der Prozess wird daher jetzt getötet.");
                    stopDavWithoutFileDeletion();
                }
            }
        });
    }

    public int getDavAppPort() {
        return _davAppPort;
    }

    /**
     * @param configurationManagementFile Verwaltungsdatei der Konfiguration
     * @param userFile                    Datei, die alle Benutzer enthält, die sich beim Datenverteiler anmelden können.
     * @param userNameConfiguration       Benutzername, mit dem sich die Konfiguration beim Datenverteiler authentifiziert
     * @param passwordFileConfiguration   Datei, die das Passwort enthält, mit dem sich die Konfiguration beim Datenverteiler authentifiziert
     * @param xmxSize                     XMX Speicher, dem der Java VM zur Verfügung gestellt wird
     *
     * @return CommandArray mit dem ein Prozess gestartet werden kann, der der Konfiguration entspricht
     */
    private String[] createConfigCommandArray(File configurationManagementFile, File userFile, String userNameConfiguration,
                                              File passwordFileConfiguration, int xmxSize) {
        final String[] defaultConfiguration = {"-datenverteiler=localhost:" + _davAppPort,
                                               "-verwaltung=" + configurationManagementFile.getAbsolutePath(),
                                               "-benutzerverwaltung=" + userFile.getAbsolutePath(),
                                               "-benutzer=" + userNameConfiguration,
                                               "-authentifizierung=" + passwordFileConfiguration.getAbsolutePath(),
                                               "-debugLevelStdErrText=" + _configurationDebugLevel,
                                               "-debugLevelFileText=INFO",
                                               "-debugFilePath=" + _workingDirectory};
        final String[] defaultJavaCommandArray = createDefaultCommandArray(_releaseVersion.getConfigurationClass(), xmxSize, _classPathOverride);
        return mergeArrays(defaultJavaCommandArray, defaultConfiguration);
    }

    /**
     * @param usernameParam  Benutzername, mit dem sich die Parametrierung anmeldet
     * @param passwordFile   Datei mit dem Passwort der Parametrierung
     * @param paramDirectory Verzeichnis, in dem die Parametrierung Daten abspeichert
     * @param xmxSize        XMX Speicher, den die Java VM bekommt
     *
     * @return CommandArray, mit dem ein Prozess gestartet werden kann. Dieser Prozess stellt die Parametrierung dar.
     */
    private String[] createParamCommandArray(String usernameParam, File passwordFile, File paramDirectory, int xmxSize) {
        final String[] defaultParam = {"-datenverteiler=localhost:" + _davAppPort,
                                       "-benutzer=" + usernameParam,
                                       "-authentifizierung=" + passwordFile.getAbsolutePath(),
                                       "-parameterVerzeichnis=" + paramDirectory.getAbsolutePath(),
                                       "-sleep=9000",
                                       "-debugLevelStdErrText=" + _paramDebugLevel,
                                       "-debugLevelFileText=INFO",
                                       "-debugFilePath=" + _workingDirectory};
        final String[] defaultJavaCommandArray = createDefaultCommandArray(_releaseVersion.getParamClass(), xmxSize, _classPathOverride);

        // Die beiden Arrays zusammenführen
        return mergeArrays(defaultJavaCommandArray, defaultParam);
    }

    /**
     * @param xmxSize         Größe des XMX Speichers, mit dem die VM gestartet wird
     * @param waitForParamApp Soll auf die Parametrierung gewartet werden?
     *
     * @return CommandArray, mit dem ein Prozess gestartet werden kann. Dieser Prozess stellt den Datenverteiler dar.
     */
    private String[] createTransmitterCommandArray(final int xmxSize, final boolean waitForParamApp) {
        // Default Parameter für den Datenverteiler
        final String[] defaultTransmitter = createTransmitterArgs(waitForParamApp);
        // Java Parameter erzeugen
        final String[] defaultJavaCommandArray = createDefaultCommandArray(_releaseVersion.getTransmitterClass(), xmxSize, _classPathOverride);

        // Die beiden Arrays zusammenführen
        return mergeArrays(defaultJavaCommandArray, defaultTransmitter);
    }

    /**
     * Erzeugt ein Array mit Aufrufargumenten für den Datenverteiler. Ein damit gestarteter Datenverteiler wartet nicht auf die
     * Applikationsfertigmeldung der Parametrierung.
     *
     * @return Array mit Aufrufargumenten für den Datenverteiler
     */
    private String[] createTransmitterArgs() {
        return createTransmitterArgs(false);
    }

    /**
     * Erzeugt ein Array mit Aufrufargumenten für den Datenverteiler.
     *
     * @param waitForParamApp Falls {@code true} wartet ein mit den Aufrufargumenten gestarteter Datenverteiler auf die Applikationsfertigmeldung der
     *                        Parametrierung.
     *
     * @return Array mit Aufrufargumenten für den Datenverteiler
     */
    private String[] createTransmitterArgs(boolean waitForParamApp) {

        final File passwd = new File(_workingDirectory, "passwd.properties");

        final List<String> list = new ArrayList<>(Arrays.asList("-benutzer=" + _userNameTransmitter,
                                                                "-konfigurationsBenutzer=" + _userNameConfiguration,
                                                                "-parametrierungsBenutzer=" + _userNameParam,
                                                                "-authentifizierung=" + passwd.getAbsolutePath(), "-davAppPort=" + _davAppPort,
                                                                "-datenverteilerId=" + _davID, "-debugLevelStdErrText=" + _transmitterDebugLevel,
                                                                "-debugLevelFileText=INFO", "-debugFilePath=" + _workingDirectory));
        if (_releaseVersion.min(V3_5_5)) {
            list.add("-warteAufParametrierung=" + (waitForParamApp ? "ja" : "nein"));
        }
        if (_releaseVersion.min(V3_6_5)) {
            list.add("-debugName=" + _debugName);
        }

        if (_remoteConfiguration != null) {
            list.add("-remoteKonfiguration=" + _remoteConfiguration);
        }
        switch (_accessControlType) {
            case Disabled:
                list.add("-rechtePruefung=nein");
                break;
            case OldDataModel:
                list.add("-rechtePruefung=alt");
                break;
            case NewDataModel:
                list.add("-rechtePruefung=neu");
                break;
        }
        if (_accessControlPlugIns.length > 0) {
            list.add("-zugriffsRechtePlugins=" + join(_accessControlPlugIns, ","));
        }
        if (_davDavPort != -1) {
            list.add("-davDavPort=" + _davDavPort);
        } else {
            list.add("-davDavPortOffset=" + getDavPortNumberOffset());
        }
        if (_passivePort > 0) {
            list.add("-passiv=" + _passivePort);
        }
        if (!_activePorts.isEmpty()) {
            list.add("-aktiv=" + _activePorts.stream().map(it -> "127.0.0.1:" + it).collect(Collectors.joining(",")));
        }
        if (_protocolClass != null) {
            if (_releaseVersion.min(V3_6_5)) {
                list.add("-tcpKommunikationsModul=" + _protocolClass.getName() + (_protocolParameter == null ? "" : (":" + _protocolParameter)));
            }
        }
        list.add("-verzoegerungFuerAndereDatenverteiler=" + _davDavConnectDelay + "ms");
        if (_releaseVersion.min(V3_6_5)) {
            list.add("-wiederverbindungsWartezeit=" + _davDavReconnectDelay + "ms");
        }
        list.addAll(Arrays.asList(_additionalTransmitterArgs));
        return list.toArray(new String[0]);
    }

    public Class<? extends ServerConnectionInterface> getProtocolClass() {
        return _protocolClass;
    }

    public void setProtocolClass(final Class<? extends ServerConnectionInterface> protocolClass) {
        _protocolClass = protocolClass;
    }

    public Object getProtocolParameter() {
        return _protocolParameter;
    }

    public void setProtocolParameter(final Object protocolParameter) {
        _protocolParameter = protocolParameter;
    }

    public Process getTransmitter() {
        return _transmitter;
    }

    public Process getConfiguration() {
        return _configuration;
    }

    public Process getOperatingMessageManagement() {
        return _operatingMessageManagement;
    }

    /**
     * Startet den Datenverteiler, die Konfiguration und die Parametrierung. Die Methode stoppt zuerst alle laufenden Prozesse ({@link #stopDaV()})
     * und startet sie dann erneut. Falls die Methode zum ersten mal aufgerufen wird oder der Datenverteiler wurde bereits mit ({@link #stopDaV()})
     * gestoppt, so werden alle Prozesse ganz normal gestartet.
     */
    public synchronized void startDaV(ParamAppType paramAppType) throws IOException {
        stopDavWithoutFileDeletion();
        startTransmitter();
        startConfiguration();
        startParam(paramAppType);
    }

    /**
     * Startet die Parametrierung des angegegeben Typs.
     *
     * @param paramAppType Typ der Parametrierung
     */
    public synchronized void startParam(final ParamAppType paramAppType) throws IOException {
        switch (paramAppType) {
            case NoParamApp:
                break;
            case DefaultParamApp:
                startDefaultParam();
                break;
            case FakeParamApp:
                try {
                    Thread.sleep(3000); // Warten, bis Datenverteiler Verbindungen akzeptiert
                    _fakeParamApp.connect(this);
                } catch (Exception e) {
                    throw new IOException(e);
                }
                break;
        }
    }

    /**
     * Startet den Datenverteiler, die Konfiguration und die Parametrierung. Die Methode stoppt zuerst alle laufenden Prozesse ({@link #stopDaV()})
     * und startet sie dann erneut. Falls die Methode zum ersten mal aufgerufen wird oder der Datenverteiler wurde bereits mit ({@link #stopDaV()})
     * gestoppt, so werden alle Prozesse ganz normal gestartet.
     */
    public synchronized void startDaV() throws IOException {
        startDaV(ParamAppType.DefaultParamApp);
    }

    /**
     * Startet den Datenverteiler, die Konfiguration und eine Dummy-Parametrierung, die nur eine entsprechende Fertigmeldung erzeugt. Die Methode
     * stoppt zuerst alle laufenden Prozesse ({@link #stopDaV()}) und startet sie dann erneut. Falls die Methode zum ersten mal aufgerufen wird oder
     * der Datenverteiler wurde bereits mit ({@link #stopDaV()}) gestoppt, so werden alle Prozesse ganz normal gestartet.
     */
    public synchronized void startDaVWithDummyParam() throws IOException {
        startDaV(ParamAppType.FakeParamApp);
    }

    /**
     * Startet den Datenverteiler, die Konfiguration. Die Methode stoppt zuerst alle laufenden Prozesse ({@link #stopDaV()}) und startet sie dann
     * erneut. Falls die Methode zum ersten mal aufgerufen wird oder der Datenverteiler wurde bereits mit ({@link #stopDaV()}) gestoppt, so werden
     * alle Prozesse ganz normal gestartet.
     */
    public synchronized void startDaVWithoutParam() throws IOException {
        startDaV(ParamAppType.NoParamApp);
    }

    /**
     * Startet einen Datenverteiler als Thread im gleichen Prozess.
     *
     * @param withConfiguration Falls {@code true}, dann wird auch eine Konfiguration (als eigener Prozess) gestartet.
     * @param paramAppType      Art der Parametrierung Applikationsfertigmeldung der Parametrierung.
     *
     * @throws Exception
     */
    public synchronized void startTransmitterInSameProcess(final boolean withConfiguration, final ParamAppType paramAppType) throws Exception {
        startTransmitterInSameProcess(withConfiguration, paramAppType, paramAppType != ParamAppType.NoParamApp);
    }

    /**
     * Startet einen Datenverteiler als Thread im gleichen Prozess.
     *
     * @param withConfiguration Falls {@code true}, dann wird auch eine Konfiguration (als eigener Prozess) gestartet.
     * @param paramAppType      Art der Parametrierung Applikationsfertigmeldung der Parametrierung.
     * @param waitForParamApp   Soll auf die Parametrierung gewartet werden?
     *
     * @throws Exception
     */
    public synchronized void startTransmitterInSameProcess(final boolean withConfiguration, final ParamAppType paramAppType,
                                                           final boolean waitForParamApp) throws Exception {
        stopDavWithoutFileDeletion();

        final TransmitterThread helper = new TransmitterThread(waitForParamApp);
        final Thread transmitterThread = new Thread(helper);
        transmitterThread.start();

        Thread.sleep(10000);

        if (withConfiguration) {
            startConfiguration();
        }

        Thread.sleep(10000);

        startParam(paramAppType);

        // Der NormalCloser verhindert das Beenden der JVM des Transmitters und somit der Test-Applikation
        _transmitterObject = helper.getTransmitter();
        _transmitterObject.setCloseHandler(new NormalCloser());
        System.out.println("Transmitter gestartet");
    }

    public Transmitter getTransmitterObject() {
        return _transmitterObject;
    }

    public void startOperatingMessageManagement() throws IOException {
        if (_operatingMessageManagement != null) {
            System.out.println("Betriebsmeldungsverwaltung wurde bereits gestartet");
            return;
        }
        final String[] defaultParam = {"-datenverteiler=localhost:" + _davAppPort,
                                       "-benutzer=Tester",
                                       "-authentifizierung=" + new File(_workingDirectory, "passwd.properties").getAbsolutePath(),
                                       "-debugLevelStdErrText=CONFIG",
                                       "-debugLevelFileText=INFO"};
        final String[] defaultJavaCommandArray = createDefaultCommandArray("de.kappich.vew.bmvew.main.SimpleMessageManager", 64, _classPathOverride);
        final String[] commandArrayOperatingMessageManagement = mergeArrays(defaultJavaCommandArray, defaultParam);
        _operatingMessageManagement = createProcess(commandArrayOperatingMessageManagement, "bmv> ", "BMV> ", null);
        System.out.println("Betriebsmeldungsverwaltung gestartet");
    }

    public void setAdditionalTransmitterArgs(final String... additionalTransmitterArgs) {
        _additionalTransmitterArgs = additionalTransmitterArgs;
    }

    public void setClassPath(final String... classPathOverride) {
        _classPathOverride = classPathOverride;
    }

    public ReleaseVersion getReleaseVersion() {
        return _releaseVersion;
    }

    public void setReleaseVersion(final ReleaseVersion releaseVersion) {
        _releaseVersion = releaseVersion;
    }

    public int getPassivePort() {
        return _passivePort;
    }

    public void setPassivePort(final int passivePort) {
        _passivePort = passivePort;
    }

    /** Beendet alle Prozesse, löscht aber nicht die notwendigen Dateien zum starten des DaV. */
    public synchronized void stopDavWithoutWaiting() {
        if (_transmitter != null) {
            _transmitter.destroy();
            _transmitter = null;
            System.out.println("Prozess - Transmitter - gestoppt");
        }
        try {
            Thread.sleep(2000);
        } catch (InterruptedException e) {
            //wird ignoriert
        }

        if (_operatingMessageManagement != null) {
            _operatingMessageManagement.destroy();
            _operatingMessageManagement = null;
            System.out.println("Prozess - Betriebsmeldungsverwaltung - gestoppt");
        }
        if (_configuration != null) {
            _configuration.destroy();
            _configuration = null;
            System.out.println("Prozess - Konfiguration - gestoppt");
        }
        if (_param != null) {
            _param.destroy();
            _param = null;
            System.out.println("Prozess - Parametrierung - gestoppt");
        }
    }

    /**
     * Stoppt den Datenverteiler,Konfiguration,Parametrierung und beendet alle Prozesse. Gibt es keine Prozesse, die beendet werden können, wird
     * nichts gemacht. Alle automatisch erzeugen Dateien werden gelöscht. <p> Der Aufruf dieser Methode blockiert solange, bis alle Prozesse beendet
     * sind.
     */
    public synchronized void stopDaV() {
        try {
            Thread.sleep(2000);
        } catch (InterruptedException ignored) {
        }
        stopDavWithoutFileDeletion();

        if (_workingDirectory != null) {
            // Es wurden Dateien kopiert, diese können gelöscht werden
            FileCopy.deleteDirectoryOrFile(_workingDirectory);
        }
    }

    /**
     * Stoppt den Datenverteiler,Konfiguration,Parametrierung und beendet alle Prozesse. Gibt es keine Prozesse, die beendet werden können, wird
     * nichts gemacht. Alle automatisch erzeugen Dateien werden gelöscht.
     */
    public synchronized void stopDavWithoutSleep(boolean withFileDeletion) {
        stopDavWithoutFileDeletion();

        if (withFileDeletion && _workingDirectory != null) {
            // Es wurden Dateien kopiert, diese können gelöscht werden
            FileCopy.deleteDirectoryOrFile(_workingDirectory);
        }
    }

    public synchronized void stopDavWithoutFileDeletion() {
        if (_transmitterObject != null) {
            System.out.println("Datenverteiler wird beendet.");
            if (true) {
                _transmitterObject.shutdown(false, "Ende des Tests");
            } else {
                _configuration.destroy(); // Datenverteiler indirekt beenden, indem Konfiguration getötet wird
                try {
                    // Warten, bis der Datenverteiler das mitgekriegt hat und sich auch beendet
                    Thread.sleep(10000);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
            }
            System.out.println("Datenverteiler beendet.");
            _transmitterObject = null;
        }
        if (_transmitter != null) {
            System.out.println("Datenverteiler wird beendet.");

            _transmitter.destroy();
            try {

                _transmitter.waitFor();
            } catch (InterruptedException ignored) {
            }
            System.out.println("Datenverteiler beendet.");
            _transmitter = null;
        }
        if (_operatingMessageManagement != null) {
            waitForProcessExit("Betriebsmeldungsverwaltung", _operatingMessageManagement, 10000);
            _operatingMessageManagement = null;
        }
        if (_configuration != null) {
            waitForProcessExit("Konfiguration", _configuration, 60000);
            _configuration = null;
        }
        if (_param != null) {
            waitForProcessExit("Parametrierung", _param, 60000);
            _param = null;
        }
        System.out.println("Alle Prozesse gestoppt");
    }

    /**
     * Startet die Configuration. Ist bereits ein Prozess vorhanden, so wird dieser mit destroy gestopped und anschließend ein neuer Prozess
     * gestartet.
     */
    public synchronized void startConfiguration() throws IOException {
        if (_configuration != null) {
            _configuration.destroy();
        }
        _configuration = createProcess(createConfigCommandArray(_configurationManagementFile, _userFile, _userNameConfiguration, _passwd, 2000),
                                       "kon" + getPrefixString() + "> ", "KON" + getPrefixString() + "> ", null);
        System.out.println("Konfiguration gestartet");
    }

    private String getPrefixString() {
        if (_name.isEmpty()) {
            return "";
        }
        return "." + _name;
    }

    /**
     * Startet die Parametrierung. Ist bereits ein Prozess vorhanden, so wird dieser mit destroy gestopped und anschließend ein neuer Prozess
     * gestartet.
     */
    public synchronized void startDefaultParam() throws IOException {
        if (_param != null) {
            _param.destroy();
        }
        _param = createProcess(createParamCommandArray(_userNameParam, _passwd, _paramDirectory, 1000), "par" + getPrefixString() + "> ",
                               "PAR" + getPrefixString() + "> ", null);
        System.out.println("Parametrierung gestartet");
    }

    public FakeParamApp getFakeParamApp() {
        return _fakeParamApp;
    }

    public void setFakeParamApp(final FakeParamApp fakeParamApp) {
        _fakeParamApp = fakeParamApp;
    }

    public String getUserNameParam() {
        return _userNameParam;
    }

    public void setUserNameParam(final String userNameParam) {
        _userNameParam = userNameParam;
    }

    public String getUserNameConfiguration() {
        return _userNameConfiguration;
    }

    public void setUserNameConfiguration(final String userNameConfiguration) {
        _userNameConfiguration = userNameConfiguration;
    }

    public String getUserNameTransmitter() {
        return _userNameTransmitter;
    }

    public void setUserNameTransmitter(final String userNameTransmitter) {
        _userNameTransmitter = userNameTransmitter;
    }

    /**
     * Startet den Datenverteiler. Ist bereits ein Prozess vorhanden, so wird dieser mit destroy gestoppt und anschließend ein neuer Prozess
     * gestartet.
     */
    public void startTransmitter() throws IOException {
        startTransmitter(false);
    }

    /**
     * Startet den Datenverteiler. Ist bereits ein Prozess vorhanden, so wird dieser mit destroy gestoppt und anschließend ein neuer Prozess
     * gestartet.
     *
     * @param waitForParamApp
     */
    public void startTransmitter(final boolean waitForParamApp) throws IOException {
        if (_transmitter != null) {
            _transmitter.destroy();
        }
        _transmitter =
            createProcess(createTransmitterCommandArray(256, waitForParamApp), "dav" + getPrefixString() + "> ", "DAV" + getPrefixString() + "> ",
                          null);
        System.out.print("Starte Datenverteiler");
        // Etwas warten, um sicher zu gehen, dass der Datenverteiler seinen TCP-Server-Socket angelegt hat, bevor nachfolgende Applikationen 
        // versuchen eine
        // Verbindung herzustellen
        try {
            Thread.sleep(5000);
        } catch (InterruptedException ignored) {
        }
    }

    public ImmutableList<Integer> getActivePorts() {
        return _activePorts;
    }

    public void setActivePorts(final Collection<Integer> activePorts) {
        _activePorts = ImmutableList.copyOf(activePorts);
    }

    private static class OutputThread implements Runnable {

        private final BufferedReader _reader;

        private final String _outputPrefix;

        public OutputThread(final InputStreamReader inputReader, final String outputPrefix) {
            _reader = new BufferedReader(inputReader);
            _outputPrefix = outputPrefix;
        }

        public void run() {
            try {
                while (Thread.interrupted() != true) {
                    String line = _reader.readLine();
                    if (line == null) {
                        break;
                    }
                    System.out.println(_outputPrefix + line);
                }
            } catch (EOFException ignored) {
                // Passiert, wenn der Prozess, der Daten auf dem Stream ausgibt, beendet wird.
            } catch (IOException e) {
                // e.printStackTrace();
                // Passiert, wenn der Prozess, der Daten auf dem Stream ausgibt, beendet wird.
            }
        }
    }

    private final class TransmitterThread implements Runnable {

        private final Object _lockObject = new Object();
        private final boolean _waitForParamApp;
        private Transmitter _transmitter;
        private boolean _failed;

        public TransmitterThread(final boolean waitForParamApp) {
            _waitForParamApp = waitForParamApp;
        }

        public void run() {
            try {
                synchronized (_lockObject) {
                    _transmitter = new Transmitter(createTransmitterArgs(_waitForParamApp));
                    _lockObject.notifyAll();
                }
            } catch (Exception e) {
                synchronized (_lockObject) {
                    _failed = true;
                    _lockObject.notifyAll();
                }
                e.printStackTrace();
                stopDaV();
                throw new IllegalStateException(e);
            }
        }

        public Transmitter getTransmitter() {
            synchronized (_lockObject) {
                long timeOut = System.currentTimeMillis() + 300000; // 5 Min
                while (_transmitter == null && !_failed) {
                    try {
                        long remaining = timeOut - System.currentTimeMillis();
                        if (remaining <= 0) {
                            throw new RuntimeException("Transmitter konnte nicht gestartet werden");
                        }
                        _lockObject.wait(remaining);
                    } catch (InterruptedException e) {
                        throw new IllegalStateException(e);
                    }
                }
                if (_transmitter == null) {
                    throw new RuntimeException("Transmitter konnte nicht gestartet werden");
                }
                return _transmitter;
            }
        }
    }
}
