/*
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.testumg.
 *
 * de.kappich.pat.testumg is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.testumg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.testumg.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.testumg.util;

import com.google.common.collect.ImmutableList;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.ClientApplication;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.dav.main.TerminateConnection;
import de.bsvrz.sys.funclib.application.StandardApplication;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.function.Function;

/**
 * Diese Klasse ermöglicht es, eine beliebige Datenverteiler-Applikation in einer Datenverteiler-Testumgebung zu instantiieren und zu testen.
 * <p>In der Regel werden Instanzen dieser Klasse mit {@link SingleDavStarter#createStandardApplication(StandardApplication, List)} oder (falls die
 * Applikation nicht {@linkplain StandardApplication} implementiert) mit {@link SingleDavStarter#createApplication(Function, List)} erzeugt.
 * <p>Wenn mehr Kontrolle über die Aufrufparameter etc. benötigt wird, kann von dieser Klasse abgeleitet werden, dann muss im Konstruktor eine
 * Referenz auf die Testumgebung übergeben werden.
 *
 * @author Kappich Systemberatung
 */
public class DafApplication<T> {

    private final DafApplicationEnvironment _dafApplicationEnvironment;

    private final Function<String[], T> _creator;

    private final List<String> _args;

    private final String _user;

    private final String _id;

    private ClientDavInterface _managementConnection;

    private T _instance;

    /**
     * Konstruktor, bei dem die benötigten Informationen übergeben werden. Die Applikation wird nicht automatisch gestartet.
     *
     * @param dafApplicationEnvironment Testumgebung
     * @param creator                   Lambda-Ausdruck, der eine Instanz der Applikationsklasse erzeugt und dabei die Aufrufargumente übergibt
     * @param args                      Zusätzliche Aufrufargumente (Testspezifisch)
     * @param user                      Benutzer, unter der die Applikation gestartet wird
     *
     * @see SingleDavStarter#createApplication(Function, List)
     * @see de.kappich.pat.testumg.util.MultiDavTestEnvironment.MultiDavStarter#createApplication(Function, List)
     * @see SingleDavStarter#createStandardApplication(StandardApplication, List)
     */
    protected DafApplication(final DafApplicationEnvironment dafApplicationEnvironment, final Function<String[], T> creator, final List<String> args,
                             final String user) {
        _dafApplicationEnvironment = dafApplicationEnvironment;
        _creator = creator;
        _args = new ArrayList<>(args);
        _user = user;
        _id = UUID.randomUUID().toString();
    }

    /**
     * Konstruktor, bei dem die benötigten Informationen übergeben werden. Die Applikation wird nicht automatisch gestartet.
     *
     * @param dafApplicationEnvironment Testumgebung
     * @param creator                   Lambda-Ausdruck, der eine Instanz der Applikationsklasse erzeugt und dabei die Aufrufargumente übergibt
     * @param args                      Zusätzliche Aufrufargumente (Testspezifisch)
     * @param user                      Benutzer, unter der die Applikation gestartet wird
     * @param debugLevel                Wird ignoriert
     *
     * @see SingleDavStarter#createApplication(Function, List)
     * @see de.kappich.pat.testumg.util.MultiDavTestEnvironment.MultiDavStarter#createApplication(Function, List)
     * @see SingleDavStarter#createStandardApplication(StandardApplication, List)
     * @deprecated debugLevel-Parameter ist überflüssig
     */
    @Deprecated
    protected DafApplication(final DafApplicationEnvironment dafApplicationEnvironment, final Function<String[], T> creator, final List<String> args,
                             final String debugLevel, final String user) {
        _dafApplicationEnvironment = dafApplicationEnvironment;
        _creator = creator;
        _args = new ArrayList<>(args);
        _user = user;
        _id = UUID.randomUUID().toString();
    }

    /**
     * Startet die Applikation
     *
     * @return Instanz der Applikationsklasse (z. B. Main-Klasse der Applikation)
     */
    public T start() {
        _managementConnection = _dafApplicationEnvironment.connect();
        _instance = _creator.apply(getArguments());
        return _instance;
    }

    /**
     * Gibt das Objekt zurück, das Datenverteilerseitig die gestartete Applikation repräsentiert. Sollte die Applikation nicht sofort ermittelbar
     * sein, wartet diese Methode, bis Sie gefunden wurde.
     * <p>
     * Die Applikation wird über einen eindeutigen Inkarnationsnamen identifiziert
     *
     * @return Applikationsobjekt
     */
    public ClientApplication getClientApplication() {
        try {
            List<SystemObject> applications = _managementConnection.getLocalApplicationObject().getType().getObjects();
            DataModel dataModel = _managementConnection.getDataModel();
            AttributeGroup atg = dataModel.getAttributeGroup("atg.applikationsFertigmeldung");
            Aspect asp = dataModel.getAspect("asp.standard");
            DataDescription dataDescription = new DataDescription(atg, asp);
            while (true) {
                ResultData[] data = _managementConnection.getData(applications.toArray(new SystemObject[0]), dataDescription, 10000);
                for (ResultData datum : data) {
                    if (datum.hasData()) {
                        String id = datum.getData().getTextValue("Inkarnationsname").getText();
                        if (id.equals(_id)) {
                            return (ClientApplication) datum.getObject();
                        }
                    }
                }
                Thread.sleep(100);
            }
        } catch (InterruptedException e) {
            throw new AssertionError(e);
        }
    }

    /**
     * Wartet, bis die Applikation die Fertigmeldung verschickt hat.
     *
     * @return Applikationsobjekt
     */
    public ClientApplication waitUntilReady() {
        try {
            DataModel dataModel = _managementConnection.getDataModel();
            AttributeGroup atg = dataModel.getAttributeGroup("atg.applikationsFertigmeldung");
            Aspect asp = dataModel.getAspect("asp.standard");
            DataDescription dataDescription = new DataDescription(atg, asp);
            ClientApplication clientApplication = getClientApplication();
            while (true) {
                ResultData datum = _managementConnection.getData(clientApplication, dataDescription, 10000);
                if (datum.hasData()) {
                    String ready = datum.getData().getTextValue("InitialisierungFertig").getText();
                    if (ready.equals("Ja")) {
                        return (ClientApplication) datum.getObject();
                    }
                }
                Thread.sleep(100);
            }
        } catch (InterruptedException e) {
            throw new AssertionError(e);
        }
    }

    /**
     * Startet die Applikation und wartet, bis sie am Datenverteiler angemeldet ist.
     */
    public void startAndWait() {
        start();
        getClientApplication();
    }

    /**
     * Startet die Applikation und wartet, bis sie die Fertigmeldung verschickt hat.
     */
    public void startAndWaitUntilReady() {
        start();
        waitUntilReady();
    }

    /**
     * Gibt die Instanz der Applikationsklasse zurück
     *
     * @return Instanz
     */
    public T getInstance() {
        return _instance;
    }

    /**
     * Terminiert die Applikation
     */
    public final void terminate() {
        sendTerminationSignal();
        _instance = null;
    }

    /**
     * Sorgt dafür, dass die Applikation terminiert. In der Standard-Implementierung wird der Datenverteiler aufgefordert, die Verbindung zu
     * terminieren. Falls ein anderes Verhalten gewünscht ist, kann die Methode überschrieben werden.
     */
    protected void sendTerminationSignal() {
        try {
            TerminateConnection.sendTerminationData(_managementConnection, Collections.singletonList(getClientApplication()));
        } catch (InterruptedException e) {
            throw new AssertionError(e);
        }
    }

    /**
     * Gibt das Wurzelverzeichnis der Testumgebung zurück. Hier können allgemeien daten abgelegt werden
     *
     * @return Wurzelverzeichnis
     */
    public final Path getRootDir() {
        return _dafApplicationEnvironment.getRootDir();
    }

    /**
     * Fügt zusätzliche Aufrufargumente hinzu, muss vor dem Starten der Applikation aufgerufen werden.
     *
     * @param arguments
     */
    public void addArguments(String... arguments) {
        _args.addAll(Arrays.asList(arguments));
    }

    private String[] getArguments() {
        ImmutableList.Builder<String> builder = ImmutableList.builder();
        addConnectionArgs(builder);
        addCustomArgs(builder);
        builder.addAll(_args);
        return builder.build().toArray(new String[0]);
    }

    /**
     * Kann überschrieben werden, un Aufrufargumente hinzuzufügen, die die Applikation immer benötigt
     *
     * @param builder List-Builder, an den zusätzliche Argumente angehängt werden können
     */
    protected void addCustomArgs(final ImmutableList.Builder<String> builder) {

    }

    private void addConnectionArgs(final ImmutableList.Builder<String> builder) {
        builder.add("-benutzer=" + _user);
        builder.add("-inkarnationsName=" + _id);
        builder.add("-authentifizierung=" + getPasswordFilePath());
        builder.add("-datenverteiler=localhost:" + _dafApplicationEnvironment.getAppPort());
    }

    @NotNull
    private Path getPasswordFilePath() {
        return _dafApplicationEnvironment.getWorkingDirectory().resolve("passwd.properties");
    }

    /**
     * Gibt die {@link FakeParamApp} der Testumgebung zurück
     *
     * @return FakeParamApp
     */
    public FakeParamApp getFakeParamApp() {
        return _dafApplicationEnvironment.getFakeParamApp();
    }

    public ClientDavInterface getManagementConnection() {
        return _managementConnection;
    }

    public List<String> getParameters() {
        return Arrays.asList(getArguments());
    }

    @Override
    public String toString() {
        return String.valueOf(_instance);
    }
}
