/*
 * Copyright 2015-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.testumg.
 *
 * de.kappich.pat.testumg is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.testumg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.testumg.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.testumg.util;

import de.bsvrz.dav.daf.main.Data;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.util.Iterator;

/**
 * Hilfsklasse zur Serialisierung und Deserialiserung von Data-Objekten aus Strings. Wird u.a. für das JSOn-Ähnliche Format der {@link FakeParamApp}
 * verwendet.
 *
 * @author Kappich Systemberatung
 */
public final class DataUtil {
    private DataUtil() {
    }

    public static String serializeData(final Data dataObject) {
        StringWriter writer = new StringWriter();
        serializeData(writer, dataObject);
        return writer.toString();
    }

    public static void serializeData(StringWriter data, final Data dataObject) {
        if (dataObject.isPlain()) {
            String valueText;
            valueText = dataObject.asTextValue().getValueText();
            if (dataObject instanceof Data.NumberValue) {
                valueText = valueText.replace('.', ',');
            }
            if (valueText.contains("\'")) {
                data.write("\"");
                data.write(valueText);
                data.write("\"");
            } else {
                data.write("'");
                data.write(valueText);
                data.write("'");
            }
        } else if (dataObject.isList()) {
            data.write("{");
            for (Iterator<Data> iterator = dataObject.iterator(); iterator.hasNext(); ) {
                final Data subData = iterator.next();
                data.write(subData.getName());
                data.write(":");
                serializeData(data, subData);
                if (iterator.hasNext()) {
                    data.write(",");
                }
            }
            data.write("}");
        } else if (dataObject.isArray()) {
            Data.Array array = dataObject.asArray();
            if (array.getLength() == 1) {
                serializeData(data, array.getItem(0));
            } else {
                data.write("[");
                for (Iterator<Data> iterator = dataObject.iterator(); iterator.hasNext(); ) {
                    final Data subData = iterator.next();
                    serializeData(data, subData);
                    if (iterator.hasNext()) {
                        data.write(",");
                    }
                }
                data.write("]");
            }
        }
    }

    public static void deserializeData(final String s, final Data dataObject) throws IOException {
        deserializeData(new StringReader(s), dataObject);
    }

    public static void deserializeData(final StringReader data, final Data dataObject) throws IOException {
        if (dataObject.isPlain()) {
            char read = read(data, '"', '\'');
            String value = readUntil(data, read);
            read(data, read);
            dataObject.asTextValue().setText(value);
        } else if (dataObject.isList()) {
            read(data, '{');
            if (peek(data) == '}') {
                data.skip(1);
                return;
            }
            boolean cont;
            do {
                String att = readUntil(data, ':');
                read(data, ':');
                deserializeData(data, dataObject.getItem(att));
                char c = read(data, ',', '}');
                cont = c == ',';

            }
            while (cont);
        } else if (dataObject.isArray()) {
            Data.Array array = dataObject.asArray();
            if (peek(data) == '[') {
                read(data, '[');
                if (peek(data) == ']') {
                    array.setLength(0);
                    data.skip(1);
                    return;
                }
                boolean cont;
                int i = 0;
                do {
                    array.setLength(i + 1);
                    deserializeData(data, array.getItem(i));
                    i++;
                    char c = read(data, ',', ']');
                    cont = c == ',';

                }
                while (cont);
            } else {
                array.setLength(1);
                deserializeData(data, array.getItem(0));
            }
        }
    }

    private static char read(final StringReader data, final char... expected) throws IOException {
        String s = readUntil(data, expected);
        if (!s.trim().isEmpty()) {
            throw new IllegalArgumentException("Erwartet: " + new String(expected) + " Gefunden: " + s);
        }
        return (char) data.read();
    }

    private static char peek(final StringReader data) throws IOException {
        int c = data.read();
        data.skip(-1);
        return (char) c;
    }

    private static String readUntil(final StringReader data, final char... expected) throws IOException {
        StringBuilder result = new StringBuilder();
        while (true) {
            int c = data.read();
            for (char exp : expected) {
                if (exp == c) {
                    data.skip(-1);
                    return result.toString();
                }
            }
            if (c == -1) {
                throw new IllegalArgumentException("Erwartet: " + new String(expected) + " Gefunden: " + result);
            }
            result.append((char) c);
        }
    }
}
