/*
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.kappich.pat.testumg.
 *
 * de.kappich.pat.testumg is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.testumg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.testumg.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.kappich.pat.testumg.util;

import de.bsvrz.dav.daf.accessControl.AccessControlMode;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientDavParameters;
import de.bsvrz.dav.daf.main.DavConnectionListener;
import de.bsvrz.dav.daf.main.authentication.ClientCredentials;
import de.bsvrz.dav.daf.main.config.management.consistenycheck.ConsistencyCheckResultInterface;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.operatingMessage.MessageSender;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.JarURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.nio.file.FileSystemNotFoundException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.jar.JarFile;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import org.junit.Assert;

/**
 * Klasse zur einfachen Realisierung von JUnit-Tests, die eine Datenverteilerumgebung benötigen.
 * <p>Alternativen zu dieser Klasse:
 * <ul>
 *     <li>Die Klasse {@link SingleDavStarter} ist generell etwas komfortabler zu benutzen und bieten mehr Funktionen.</li>
 *     <li>Die Klasse {@link MultiDavTestEnvironment} erlaubt das Testen von mehreren Datenverteilern gleichzeitig.</li>
 * </ul>
 *
 * @author Kappich Systemberatung
 */
public class DavTestEnvironment implements DafApplicationEnvironment {

    private final String _applicationName;

    private final Debug _debug;
    private final Map<ClientDavInterface, ConnectionListener> _connectionsWithListeners = new HashMap<>();
    private final FakeParamApp _fakeParamApp = new FakeParamApp();
    private String _applicationClassName;
    private DaVStarter _davStarter;
    private String _davHostname;
    private int _davTcpPort;
    private File _temporaryDirectory;
    private String _configurationDebugLevel = DaVStarter.DEFAULT_CONFIGURATION_DEBUG;
    private String _transmitterDebugLevel = DaVStarter.DEFAULT_TRANSMITTER_DEBUG;
    private String _paramDebugLevel = DaVStarter.DEFAULT_PARAM_DEBUG;
    private ConfigurationController _configurationController;
    private String[] _additionalTransmitterArgs = new String[0];
    private boolean _startDavInSameProcess;
    private ParamAppType _paramAppType = ParamAppType.DefaultParamApp;

    /**
     * Erzeugt eine neue Testumgebung und initialisiert die Debug-Bibliothek.
     *
     * @param debugLevel Zu verwendender Debuglevel für die Standard-Error-Ausgabe. Mögliche Werte sind "ERROR", "WARNING", "CONFIG", "INFO", "FINE",
     *                   "FINER", "FINEST" und "ALL".
     */
    public DavTestEnvironment(String debugLevel) {
        this(debugLevel, null);
    }

    /**
     * Erzeugt eine neue Testumgebung und initialisiert die Debug-Bibliothek.
     *
     * @param debugLevelStdErr Zu verwendender Debuglevel für die Standard-Error-Ausgabe. Mögliche Werte sind "ERROR", "WARNING", "CONFIG", "INFO",
     *                         "FINE", "FINER", "FINEST" und "ALL".
     * @param debugLevelFile   Zu verwendender Debuglevel für die Datei-Ausgabe. Mögliche Werte sind "ERROR", "WARNING", "CONFIG", "INFO", "FINE",
     *                         "FINER", "FINEST" und "ALL".
     */
    public DavTestEnvironment(String debugLevelStdErr, String debugLevelFile) {
        try {
            final StackTraceElement[] stackTraceElements = Thread.currentThread().getStackTrace();
            _applicationClassName = DavTestEnvironment.class.getName();
            for (int i = stackTraceElements.length - 1; i >= 0; i--) {
                StackTraceElement stackTraceElement = stackTraceElements[i];
                final String stackTraceElementClassName = stackTraceElements[i].getClassName();
                if (stackTraceElementClassName.equals(DavTestEnvironment.class.getName())) {
                    break;
                }
                _applicationClassName = stackTraceElementClassName;
            }
            final String[] strings = _applicationClassName.split("\\.");
            _applicationName = strings[strings.length - 1];
            final String[] argumentStrings;
            if (debugLevelFile == null || debugLevelFile.isEmpty()) {
                argumentStrings = new String[] {"-debugLevelStdErrText=" + debugLevelStdErr};
            } else {
                argumentStrings = new String[] {"-debugLevelStdErrText=" + debugLevelStdErr, "-debugLevelFileText=" + debugLevelFile};
            }
            Debug.init(_applicationName, new ArgumentList(argumentStrings));
            _debug = Debug.getLogger();
            System.out.println(_applicationName + ": Testumgebung mit Debug-Level " + debugLevelStdErr + " erzeugt");
        } catch (Exception e) {
            final String message = "Fehler beim Erzeugen der Datenverteiler-Testumgebung";
            System.out.println(message + ": " + e);
            e.printStackTrace();
            throw new RuntimeException(message, e);
        }
    }

    /**
     * Ergänzt rekursiv die Resource-Namen aller Dateien im angegeben Verzeichnis in eine Liste.
     *
     * @param resourceNames      Liste mit Resource-Namen der im Verzeichnis oder in Unterverzeichnissen enthaltenen Dateien
     * @param directory          Zu untersuchendes Verzeichnis.
     * @param resourceNamePrefix REsource-Namen-Präfix des angegebenen Verzeichnisses
     */
    private static void addResourceNames(final ArrayList<String> resourceNames, final Path directory, final String resourceNamePrefix) {
        Stream<Path> files;
        try {
            files = Files.list(directory);
        } catch (IOException e) {
            throw new IllegalArgumentException(e);
        }
        files.forEach(subFile -> {
            final String resourceName = resourceNamePrefix + '/' + subFile.getFileName();
            if (Files.isDirectory(subFile)) {
                addResourceNames(resourceNames, subFile, resourceName);
            } else {
                resourceNames.add(resourceName);
            }
        });
    }

    /**
     * Liefert ein temporäres Verzeichnis, dass vom Testfall zur Ablage von Dateien verwendet werden kann zurück. Beim ersten Aufruf der Methode wird
     * das temporäre Verzeichnis im aktuellen Arbeitsverzeichnis erzeugt. Bei weiteren Aufrufen wird das im ersten Aufruf erzeugte Verzeichnis
     * unverändert zurückgegeben. Der Name ergibt sich aus dem Klassennamen der Klasse, die den Konstruktor der Testumgebung aufgerufen hat, mit dem
     * Präfix "TmpFiles-". Falls das Verzeichnis im ersten Aufruf der Methode bereits existiert wird es gelöscht und dann neu angelegt.
     *
     * @return File-Objekt des neu erzeugten temporären Verzeichnisses.
     */
    public File getTemporaryDirectory() {
        if (_temporaryDirectory == null) {
            _temporaryDirectory = TempDirectoryCreator.createTemporaryDirectory().toFile();
        }
        return _temporaryDirectory;
    }

    /**
     * Löscht das temporäre Verzeichnis, das mit der Methode getTemporaryDirectory() erzeugt wurde. Wenn die Methode getTemporaryDirectory noch nicht
     * aufgerufen wurde, dann tut diese Methode nichts.
     */
    public void deleteTemporaryDirectory() {
        if (_temporaryDirectory != null) {
            try {
                _debug.info("Temporäre Dateien werden gelöscht");
                FileCopy.deleteDirectoryOrFile(_temporaryDirectory);
                _temporaryDirectory = null;
            } catch (Exception e) {
                final String message = "Fehler beim Löschen von temporären Dateien";
                _debug.error(message, e);
                e.printStackTrace();
            }
        }
    }

    public void useDav(final String davHostname, final int davTcpPort) {
        _davHostname = davHostname;
        _davTcpPort = davTcpPort;
    }

    public void setConfigurationDebugLevel(final String configurationDebugLevel) {
        _configurationDebugLevel = configurationDebugLevel;
    }

    public void setTransmitterDebugLevel(final String transmitterDebugLevel) {
        _transmitterDebugLevel = transmitterDebugLevel;
    }

    public void setParamDebugLevel(final String paramDebugLevel) {
        _paramDebugLevel = paramDebugLevel;
    }

    /**
     * Startet die Datenverteiler Umgebung.
     *
     * @param configurationAreaPids Enhält die Pids von Konfigurationsbereichen, die vor dem Start importiert und aktiviert werden sollen. Wenn ein
     *                              Element den Wert {@code null} hat, dann werden alle bis dahin im Array enthaltenen Pids importiert und aktiviert,
     *                              bevor die nachfolgenden Pids bearbeitet werden. Mehrere Varianten des gleichen Bereichs können durch Angabe eine
     *                              Versionsnummer unterschieden werden. Dabei wird die Versionsnummer mit Doppelpunkt separiert an die Pid
     *                              angehangen. Es wird dann eine Versorgungsdatei gesucht, deren Name aus der Pid, der Versionsnummer und der Endung
     *                              ".xml" gebildet wird. Im Dateinamen darf dabei kein Doppelpunkt enthalten sein.
     */
    public void startDav(String... configurationAreaPids) {
        try {
            if (_davStarter != null) {
                throw new RuntimeException("Datenverteiler soll gestartet werden, obwohl er bereits gestartet wurde.");
            }
            if (_configurationController != null) {
                throw new RuntimeException("Datenverteiler soll gestartet werden, obwohl die Konfiguration bereits geladen wurde.");
            }
            final DaVStarter davStarter;

            if (configurationAreaPids == null || configurationAreaPids.length == 0) {
                davStarter =
                    new DaVStarter(new File(getTemporaryDirectory(), "konfiguration"), false, _configurationDebugLevel, _transmitterDebugLevel,
                                   _paramDebugLevel);
            } else {
                ConfigurationController configurationController = new ConfigurationController(_applicationClassName, getTemporaryDirectory());
                //final List<String> pids = Arrays.asList(configurationAreaPids);
                final List<String> pidsWithVersion = new ArrayList<>();
                for (int i = 0; i < configurationAreaPids.length; i++) {
                    String configurationAreaPid = configurationAreaPids[i];
                    if (configurationAreaPid != null) {
                        pidsWithVersion.add(configurationAreaPid);
                    }
                    if (configurationAreaPid == null || i == configurationAreaPids.length - 1 && !pidsWithVersion.isEmpty()) {
                        try {
                            System.out.println("Konfigurationsbereiche werden importiert und aktiviert: " + pidsWithVersion);
                            configurationController.startConfigurationWithTestAuthority();
                            final List<String> pids = new ArrayList<>();
                            for (String pid : pidsWithVersion) {
                                int version = -1;
                                if (pid.matches(".*:[0-9]*")) {
                                    version = Integer.parseInt(pid.replaceFirst(".*:", ""));
                                    System.out.println("version = " + version);
                                    pid = pid.replaceFirst(":[0-9]*$", "");
                                    System.out.println("pid = " + pid);

                                }
                                configurationController.copyImportFile(pid, version);
                                pids.add(pid);
                            }
                            configurationController.importConfigurationAreas(pids);
                            ConsistencyCheckResultInterface result = configurationController.activateConfigurationAreas(pids);
                            if (result.interferenceErrors() || result.localError()) {
                                Assert.fail("Fehler beim aktivieren der Konfigurationsbereiche:\n" + result.toString());
                            }
                        } finally {
                            configurationController.closeConfiguration();
                        }
                    }
                    if (configurationAreaPid == null) {
                        pidsWithVersion.clear();
                    }
                }
                davStarter = configurationController.getDaVStarter(_configurationDebugLevel, _transmitterDebugLevel, _paramDebugLevel);
            }
            davStarter.setDebugName(_applicationName);
            davStarter.setAdditionalTransmitterArgs(_additionalTransmitterArgs);
            _debug.info("Datenverteiler wird gestartet");
            _davStarter = davStarter;
            _davTcpPort = davStarter.getDavAppPort();
            _davHostname = "127.0.0.1";
            if (_fakeParamApp != null) {
                davStarter.setFakeParamApp(_fakeParamApp);
            }
            if (_startDavInSameProcess) {
                davStarter.startTransmitterInSameProcess(true, _paramAppType);
            } else {
                davStarter.startDaV(_paramAppType);
            }
        } catch (Exception e) {
            final String message = "Fehler beim Start des Datenverteilers";
            _debug.error(message, e);
            e.printStackTrace();
            MultiDavTestEnvironment.dumpThreads(System.err);
            throw new RuntimeException(message, e);
        }
    }

    /** Startet den Datenverteiler mit aktivierter Rechteprüfung */
    public void startDavWithAccessControl(final AccessControlMode accessControlType, final boolean containsDaVFiles, String... accessControlPlugins) {
        try {
            if (_davStarter != null) {
                throw new RuntimeException("Datenverteiler soll gestartet werden, obwohl er bereits gestartet wurde.");
            }
            if (_configurationController != null) {
                throw new RuntimeException("Datenverteiler soll gestartet werden, obwohl die Konfiguration bereits geladen wurde.");
            }
            final DaVStarter davStarter;

            // Falls die Konfiguration im vorhergehenden Lauf Versorgungsdateien importiert hat, dann liegen die Konfigurationsdateien etc. im 
            // Unterverzeichnis "konfiguration"
            File workingDirectory = getTemporaryDirectory();
            File[] temporaryFiles = workingDirectory.listFiles();
            if (temporaryFiles != null) {
	            for (File temporaryFile : temporaryFiles) {
                    if (temporaryFile.getName().equals("konfiguration") && temporaryFile.isDirectory()) {
                        workingDirectory = temporaryFile;
                    }
                }
            }

            davStarter = new DaVStarter(workingDirectory, containsDaVFiles, _configurationDebugLevel, _transmitterDebugLevel, _paramDebugLevel,
                                        accessControlType, accessControlPlugins);
            if (_fakeParamApp != null) {
                davStarter.setFakeParamApp(_fakeParamApp);
            }
            _debug.info("Datenverteiler wird gestartet");
            _davStarter = davStarter;
            _davTcpPort = davStarter.getDavAppPort();
            _davHostname = "127.0.0.1";
            if (_startDavInSameProcess) {
                davStarter.startTransmitterInSameProcess(true, _paramAppType);
            } else {
                davStarter.startDaV();
            }
            Thread.sleep(10000);
        } catch (Exception e) {
            final String message = "Fehler beim Start des Datenverteilers";
            _debug.error(message, e);
            e.printStackTrace();
            throw new RuntimeException(message, e);
        }
    }

    /**
     * Öffnet die Konfiguration im Offline-Modus.
     *
     * @param configurationAreaPids Enthält die Pids von Konfigurationsbereichen, die vor dem Start importiert und aktiviert werden sollen. Wenn ein
     *                              Element den Wert {@code null} hat, dann werden alle bis dahin im Array enthaltenen Pids importiert und aktiviert,
     *                              bevor die
     */
    public ConfigurationController startConfig(String... configurationAreaPids) {
        try {
            if (_davStarter != null) {
                throw new RuntimeException("Konfiguration soll geladen werden, obwohl der Datenverteiler bereits gestartet wurde.");
            }
            if (_configurationController != null) {
                throw new RuntimeException("Konfiguration soll geladen werden, obwohl sie bereits geladen wurde.");
            }
            ConfigurationController configurationController = new ConfigurationController(_applicationClassName, getTemporaryDirectory());
            if (configurationAreaPids != null && configurationAreaPids.length != 0) {
                final List<String> pids = new ArrayList<>();
                for (int i = 0; i < configurationAreaPids.length; i++) {
                    String configurationAreaPid = configurationAreaPids[i];
                    if (configurationAreaPid != null) {
                        pids.add(configurationAreaPid);
                    }
                    if ((configurationAreaPid == null) || ((i == (configurationAreaPids.length - 1)) && (!pids.isEmpty()))) {
                        try {
                            System.out.println("Konfigurationsbereiche werden importiert und aktiviert: " + pids);
                            configurationController.startConfigurationWithTestAuthority();
                            configurationController.copyImportFiles(pids);
                            configurationController.importConfigurationAreas(pids);
                            configurationController.activateConfigurationAreas(pids);
                        } finally {
                            configurationController.closeConfiguration();
                        }
                    }
                    if (configurationAreaPid == null) {
                        pids.clear();
                    }
                }
            }
            configurationController.startConfigurationWithTestAuthority();
            _debug.info("Konfiguration geladen");
            _configurationController = configurationController;
            return _configurationController;
        } catch (Exception e) {
            final String message = "Fehler beim Laden der Konfiguration";
            _debug.error(message, e);
            e.printStackTrace();
            throw new RuntimeException(message, e);
        }
    }

    /**
     * Schließt die im Offline-Modus geöffnete Konfiguration wieder.
     *
     * @param withFileDeletion Falls {@code true}, dann wird das Verzeichnis mit den temporären Dateien gelöscht.
     */
    public void stopConfig(final boolean withFileDeletion) {
        try {
            if (_configurationController == null) {
                _debug.warning("Konfiguration soll geschlossen werden, obwohl sie nicht geladen wurde.");
            } else {
                final ConfigurationController configurationController = _configurationController;
                _configurationController = null;
                configurationController.closeConfiguration();
            }
        } catch (Exception e) {
            final String message = "Fehler beim Schließen der Konfiguration";
            _debug.error(message, e);
            e.printStackTrace();
            throw new RuntimeException(message, e);
        }
        if (withFileDeletion) {
            deleteTemporaryDirectory();
        }
    }

    public void startOperatingMessageManagement() {
        if (_davStarter == null) {
            _debug.warning("Betriebsmeldungsverwaltung soll gestartet werden, obwohl Datenverteiler noch nicht gestartet wurde.");
        } else {
            try {
                _davStarter.startOperatingMessageManagement();
            } catch (IOException e) {
                _debug.warning("Fehler beim Start der Betriebsmeldungsverwaltung", e);
                throw new RuntimeException(e);
            }
        }
    }

    public void stopDav(boolean withFileDeletion) {
        try {
            if (_davStarter == null) {
                _debug.warning("Datenverteiler soll gestoppt werden, obwohl er nicht gestartet wurde.");
            } else {
                stopAllConnections();
                _debug.info("Datenverteiler wird gestoppt");
                final DaVStarter davStarter = _davStarter;
                _davStarter = null;
                davStarter.stopDavWithoutFileDeletion();
            }
        } catch (Exception e) {
            final String message = "Fehler beim Beenden des Datenverteilers";
            _debug.error(message, e);
            e.printStackTrace();
        }
        if (withFileDeletion) {
            deleteTemporaryDirectory();
        }
        try {
            Thread.sleep(1000);
        } catch (InterruptedException ignored) {
        }
    }

    public void stopAll(boolean withFileDeletion) {
        _debug.info("Alles wird gestoppt");
        if (_davStarter != null) {
            stopDav(withFileDeletion);
        }
        if (_configurationController != null) {
            stopConfig(withFileDeletion);
        }
        if (withFileDeletion) {
            deleteTemporaryDirectory();
        }
    }

    public ClientDavInterface startDavConnection() {
        return startDavConnection(null);
    }

    public ClientDavInterface startDavConnection(final String user, final ClientCredentials clientCredentials) {
        return startDavConnection(null, user, clientCredentials);
    }

    public ClientDavInterface startDavConnection(ClientDavParameters clientDavParameters) {
        return startDavConnection(clientDavParameters, "Tester", ClientCredentials.ofString("geheim"));
    }

    public ClientDavInterface startDavConnection(ClientDavParameters clientDavParameters, final String user,
                                                 final ClientCredentials clientCredentials) {
        _debug.info("Datenverteiler-Verbindung wird gestartet");
        try {
            final CreateClientDavConnection createClientDavConnection =
                new CreateClientDavConnection(user, clientCredentials, _applicationName, clientDavParameters, _davHostname, _davTcpPort);
            final ClientDavInterface connection = createClientDavConnection.getConnection();
            MessageSender.getInstance().init(connection, _applicationName, _applicationName);
            final ConnectionListener connectionListener = new ConnectionListener();
            connection.addConnectionListener(connectionListener);
            _connectionsWithListeners.put(connection, connectionListener);
            return connection;
        } catch (Exception e) {
            final String message = "Fehler beim Erzeugen einer Datenverteiler-Verbindung";
            _debug.error(message, e);
            e.printStackTrace();
            throw new RuntimeException(message, e);
        }
    }

    public void stopDavConnection(final ClientDavInterface connection) {
        try {
            if (connection == null) {
                _debug.warning("Datenverteiler-Verbindung zum Datenverteiler sollte gestoppt werden, ist aber null");
                return;
            }

            final ConnectionListener connectionListener = _connectionsWithListeners.remove(connection);
            if (connectionListener == null) {
                _debug.warning("Datenverteiler-Verbindung sollte gestoppt werden, ist aber bereits gestoppt worden oder wurde anders erzeugt");
            } else if (connectionListener.isClosed()) {
                _debug.warning("Datenverteiler-Verbindung sollte gestoppt werden, ist aber bereits vom Datenverteiler terminiert worden");
            } else {
                _debug.info("Datenverteiler-Verbindung wird gestoppt");
                connection.disconnect(false, "");
                if (connectionListener.waitForClosed(5000)) {
                    _debug.info("Datenverteiler-Verbindung wurde erfolgreich gestoppt");
                } else {
                    _debug.info("Datenverteiler-Verbindung konnte nicht gestoppt werden");
                }
                connection.removeConnectionListener(connectionListener);
                Thread.sleep(500);
            }
        } catch (Exception e) {
            final String message = "Fehler beim Stoppen einer Datenverteiler-Verbindung";
            _debug.error(message, e);
            e.printStackTrace();
        }
    }

    public void stopAllConnections() {
        _debug.info("Alle Datenverteiler-Verbindungen werden gestoppt");
        final ClientDavInterface[] connections = _connectionsWithListeners.keySet().toArray(new ClientDavInterface[0]);
        for (ClientDavInterface connection : connections) {
            stopDavConnection(connection);
        }
    }

    public Process startJavaProcessAsDavClient(final String outputPrefix, final String errorOutputPrefix, String className, String debugLevel,
                                               int maxHeapMegaBytes, String... arguments) {
        List<String> javaArguments = new LinkedList<>();
        javaArguments.add("-datenverteiler=localhost:" + (8083 + DaVStarter.getDavPortNumberOffset()));
        javaArguments.add("-benutzer=Tester");
        javaArguments.add("-authentifizierung=" + _davStarter.getWorkingDirectory().getAbsolutePath() + File.separator + "passwd.properties");
        javaArguments.add("-debugLevelStdErrText=" + debugLevel);
        Collections.addAll(javaArguments, arguments);
        final Process process = startJavaProcess(outputPrefix, errorOutputPrefix, className, maxHeapMegaBytes, javaArguments.toArray(new String[0]));
        //process.exitValue()
        System.out.println(className + " gestartet");
        System.out.println("javaArguments = " + javaArguments);
        return process;

    }

    public Process startJavaProcess(final String outputPrefix, final String errorOutputPrefix, String className, int maxHeapMegaBytes,
                                    String... arguments) {
        String fileSeparator = System.getProperty("file.separator");
        String javaHome = System.getProperty("java.home");
        String classPath = System.getProperty("java.class.path");
        List<String> javaArguments = new LinkedList<>();
        javaArguments.add(javaHome + fileSeparator + "bin" + fileSeparator + "java");
        javaArguments.add("-Xmx" + maxHeapMegaBytes + "m");
        javaArguments.add("-cp");
        javaArguments.add(classPath);
        javaArguments.add(className);
        Collections.addAll(javaArguments, arguments);
        return startProcess(outputPrefix, errorOutputPrefix, javaArguments.<String>toArray(new String[0]));
    }

    private Process startProcess(final String outputPrefix, final String errorOutputPrefix, String... arguments) {
        try {
            return DaVStarter.createProcess(arguments, outputPrefix, errorOutputPrefix, getTemporaryDirectory());
        } catch (IOException e) {
            final String message = "Fehler beim Starten eines Prozesses, Argumente: " + Arrays.toString(arguments);
            _debug.error(message, e);
            e.printStackTrace();
            throw new RuntimeException(message, e);
        }
    }

    /**
     * Setzt, ob der Datenverteiler im selben Prozess gestartet werden soll (Hilfreich zum Debuggen des Datenverteilers)
     *
     * @param startDavInSameProcess Datenverteiler im selben Prozess starten?
     */
    public void setStartDavInSameProcess(final boolean startDavInSameProcess) {
        _startDavInSameProcess = startDavInSameProcess;
    }

    /**
     * Sorgt dafür, dass statt der normalen Parametrierung eine Minimalimplementierung der Parametrierung verwendet wird, die einfacher fernzusteuern
     * ist und die unabhängig von Benutzerrechten arbeitet.
     *
     * @return Klasse um Fake-Parametrierung fernzusteuern
     */
    public FakeParamApp createFakeParamApp() {
        setParamAppType(ParamAppType.FakeParamApp);
        return _fakeParamApp;
    }

    public void setAdditionalTransmitterArgs(final String... additionalTransmitterArgs) {
        _additionalTransmitterArgs = additionalTransmitterArgs;
    }

    public String getDavHostname() {
        return _davHostname;
    }

    public int getDavTcpPort() {
        return _davTcpPort;
    }

    @Override
    public ClientDavInterface connect() {
        return startDavConnection();
    }

    @Override
    public Path getRootDir() {
        return getTemporaryDirectory().toPath();
    }

    @Override
    public Path getWorkingDirectory() {
        return getTemporaryDirectory().toPath().resolve("konfiguration");
    }

    @Override
    public int getAppPort() {
        return _davStarter.getDavAppPort();
    }

    @Override
    public FakeParamApp getFakeParamApp() {
        return _fakeParamApp;
    }

    @Override
    public String getConfigurationAuthority() {
        return "kv.testKonfiguration";
    }

    @Override
    public void setParamAppType(final ParamAppType paramAppType) {
        _paramAppType = Objects.requireNonNull(paramAppType);
    }

    /**
     * Kopiert Resourcedateien mit einem angegebenen Präfix in ein neues Verzeichnis. Die Resourcedateien müssen im Classpath auffindbar sein. Die
     * Methode kann sowohl mit wirklichen Dateien, als auch mit Resourcen in JAR-Dateien umgehen. Dabei geht die Methode davon aus, dass alle
     * Resource-Dateien in der gleichen JAR-Datei enthalten sind.
     *
     * @param resourceNamePrefix       Prefix der zu kopierenden Resourcen. Resourcenamen entsprechen Packagenamen, bei denen die Punkte durch Slashes
     *                                 (/) ersetzt wurden. Beispielsweise: 'de/kappich/pat/testumg/util'.
     * @param destinationDirectoryName Name des zu erzeugenden Verzeichnis, das innerhalb eines temporären Verzeichnisses (siehe @{link
     *                                 #getTemporaryDirectory}) angelegt wird.
     *
     * @return Erzeugtes Verzeichnis mit den kopierten Resource-Dateien
     */
    public File copyResources(final String resourceNamePrefix, String destinationDirectoryName) {
        try {
            final File destinationDirectory = new File(getTemporaryDirectory(), destinationDirectoryName);
            final ArrayList<String> resourceNames = new ArrayList<>();
            final URL startResource = DavTestEnvironment.class.getClassLoader().getResource(resourceNamePrefix);
            Path path = null;
            try {
                path = Paths.get(startResource.toURI());
            } catch (FileSystemNotFoundException ignored) {
            }
            if (path != null && Files.exists(path)) {
                if (Files.isDirectory(path)) {
                    addResourceNames(resourceNames, path, resourceNamePrefix);
                } else {
                    throw new IllegalArgumentException(
                        "Kopieren von Resourcen nicht möglich, da die angegebenen Resource '" + resourceNamePrefix + "' kein Verzeichnis ist.");
                }
            } else {
                URLConnection connection = startResource.openConnection();
	            if (connection instanceof JarURLConnection jarURLConnection) {
                    JarFile jarFile = jarURLConnection.getJarFile();
                    Enumeration<java.util.jar.JarEntry> e = jarFile.entries();
                    while (e.hasMoreElements()) {
                        ZipEntry entry = e.nextElement();
                        String entryname = entry.getName();
                        if (!entry.isDirectory() && entryname.startsWith(resourceNamePrefix)) {
                            resourceNames.add(entryname);
                        }
                    }
                } else {
                    throw new IllegalArgumentException("Kopieren von Resourcen nicht möglich, da '" + connection + "' keine JarURLConnection ist.");
                }
            }
            final int resourceNamePrefixLength = resourceNamePrefix.length();
            for (String resourceName : resourceNames) {
                final URL resource = DavTestEnvironment.class.getClassLoader().getResource(resourceName);
	            try (InputStream sourceInput = resource.openStream()) {
                    final String localName = resourceName.substring(resourceNamePrefixLength, resourceName.length());
                    final File destinationFile = new File((destinationDirectory.getAbsolutePath() + localName).replace('/', File.separatorChar));
                    FileCopy.copyFile(sourceInput, destinationFile, true);
                }
            }
            return destinationDirectory;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    private static class ConnectionListener implements DavConnectionListener {

        private boolean _closed;

        public void connectionClosed(ClientDavInterface connection) {
            synchronized (this) {
                _closed = true;
                notifyAll();
            }
        }

        public boolean isClosed() {
            synchronized (this) {
                return _closed;
            }
        }

        public boolean waitForClosed(long timeout) {
            final long timeoutTime = System.currentTimeMillis() + timeout;
            synchronized (this) {
                try {
                    while (!_closed) {
                        long delta = timeoutTime - System.currentTimeMillis();
                        if (delta <= 0) {
                            break;
                        }
                        wait(delta);
                    }
                } catch (InterruptedException e) {
                    // Abbruch der Warteschleife
                }
                return _closed;
            }
        }
    }
}
