/*
 * Copyright 2013-2020 by Kappich Systemberatung, Aachen
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.kappich.pat.testumg.
 *
 * de.kappich.pat.testumg is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.testumg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.testumg.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.kappich.pat.testumg.util;

import static org.junit.Assert.assertTrue;


import de.bsvrz.dav.daf.main.ApplicationSubscriptionInfo;
import de.bsvrz.dav.daf.main.ClientDavConnection;
import de.bsvrz.dav.daf.main.ClientReceiverInterface;
import de.bsvrz.dav.daf.main.ClientSenderInterface;
import de.bsvrz.dav.daf.main.ClientSubscriptionInfo;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.OneSubscriptionPerSendData;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.SendSubscriptionNotConfirmed;
import de.bsvrz.dav.daf.main.SenderRole;
import de.bsvrz.dav.daf.main.SubscriptionState;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.AttributeGroupUsage;
import de.bsvrz.dav.daf.main.config.ClientApplication;
import de.bsvrz.dav.daf.main.config.DavApplication;
import de.bsvrz.dav.daf.main.config.MutableSet;
import de.bsvrz.dav.daf.main.config.Pid;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.dav.daf.main.impl.config.DafDataModel;
import de.bsvrz.dav.dav.subscriptions.ReceivingSubscription;
import de.bsvrz.dav.dav.subscriptions.SendingSubscription;
import de.bsvrz.dav.dav.subscriptions.Subscription;
import de.bsvrz.dav.dav.subscriptions.SubscriptionInfo;
import java.io.IOException;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import org.junit.Assert;

/**
 * Hilfklasse zum Testen des Datenverteilers von Clientseite, enthält Methoden, den Anmeldezustand von Daten abzufragen und ähnliches
 *
 * @author Kappich Systemberatung
 * @version $Revision: 0000 $
 */
public final class DavTestUtil {

    public static final long DEFAULT_TIMEOUT = 5000L;
    private static final Map<DataId, LinkedBlockingQueue<ResultData>> _dataQueues = new ConcurrentHashMap<>();
    private static final ClientReceiverInterface _receiver = new ClientReceiverInterface() {
        @Override
        public void update(final ResultData[] results) {
            for (ResultData result : results) {
                DataId dataId = new DataId(result.getObject(), result.getDataDescription());
                LinkedBlockingQueue<ResultData> queue = getQueue(dataId);
                queue.add(result);
            }
        }
    };

    private static LinkedBlockingQueue<ResultData> getQueue(final DataId description) {
        if (!_dataQueues.containsKey(description)) {
            _dataQueues.putIfAbsent(description, new LinkedBlockingQueue<>());
            getConnection(description._object)
                .subscribeReceiver(_receiver, description._object, description._dataDescription, ReceiveOptions.normal(), ReceiverRole.receiver());
        }
        return _dataQueues.get(description);
    }

    /**
     * Sendet einen DAF-Datensatz als Sender. Diese Methode übernimmt das An- und Abmelden des Senders.
     *
     * @param dataset datensatz
     *
     * @throws OneSubscriptionPerSendData
     */
    public static void sendData(final ResultData dataset) throws OneSubscriptionPerSendData {
        sendData(dataset, SenderRole.sender());
    }

    /**
     * Sendet einen DAF-Datensatz als Sender. Diese Methode übernimmt das an- und Abmelden des Senders.
     *
     * @param dataset    datensatz
     * @param senderRole
     *
     * @throws OneSubscriptionPerSendData
     */
    public static void sendData(final ResultData dataset, final SenderRole senderRole) throws OneSubscriptionPerSendData {
        getConnection(dataset.getObject()).subscribeSender(new DummySender(dataset), dataset.getObject(), dataset.getDataDescription(), senderRole);
    }

    /**
     * Empfängt den nächsten Datensatz an das angegebene DE mit der angegebenen Datenidentifikation. Jeder Datensatz wird nur einmal zurückgegeben.
     * Wird kein nächster Datensatz empfangen liefert die Methode einen {@link AssertionError}.
     *
     * @param systemObject    Objekt
     * @param dataDescription Datenidentifikation
     *
     * @return Empfangener Datensatz
     *
     * @throws InterruptedException
     */
    public static ResultData readData(final SystemObject systemObject, final DataDescription dataDescription) throws InterruptedException {
        return readData(systemObject, dataDescription, DEFAULT_TIMEOUT);
    }

    /**
     * Empfängt den nächsten Datensatz an das angegebene DE mit der angegebenen Datenidentifikation. Jeder Datensatz wird nur einmal zurückgegeben.
     * Wird kein nächster Datensatz empfangen liefert die Methode einen {@link AssertionError}.
     *
     * @param systemObject    Objekt
     * @param dataDescription Datenidentifikation
     * @param timeout         Timeout in Millisekunden
     *
     * @return Empfangener Datensatz
     *
     * @throws InterruptedException
     */
    public static ResultData readData(final SystemObject systemObject, final DataDescription dataDescription, final long timeout)
        throws InterruptedException {
        DataId description = new DataId(systemObject, dataDescription);
        LinkedBlockingQueue<ResultData> linkedBlockingQueue = getQueue(description);
        ResultData poll = linkedBlockingQueue.poll(timeout, TimeUnit.MILLISECONDS);
        if (poll == null) {
            throw new AssertionError("Timeout beim Warten auf " + description);
        }
        return poll;
    }

    /**
     * Meldet sich als Empfänger auf eine Datenidentifikation an, sodass folgende Aufrufe von readData() Werte ab dem aktuellen Zeitpunkt
     * zurückgeben.
     *
     * @param systemObject    Objekt
     * @param dataDescription Datenidentifikation
     *
     * @throws InterruptedException
     */
    public static void startRead(final SystemObject systemObject, final DataDescription dataDescription) throws InterruptedException {
        DataId description = new DataId(systemObject, dataDescription);
        getQueue(description);
    }

    /**
     * Gibt die angemeldeten Daten einer Applikation aus Datenverteilersicht zurück
     *
     * @param application Applikation
     *
     * @return Anmeldungsliste
     *
     * @throws IOException Falls nicht ermittelt werden konnte
     */
    public static ApplicationSubscriptionInfo getSubscriptionState(ClientApplication application) throws IOException {
        return getConnection(application).getSubscriptionInfo(findDav(application), application);
    }

    private static DavApplication findDav(final ClientApplication application) {
        List<SystemObject> davs = application.getDataModel().getType(Pid.Type.DAV_APPLICATION).getObjects();
        for (SystemObject dav : davs) {
	        if (dav instanceof DavApplication davApplication) {
                MutableSet clientApplicationSet = davApplication.getClientApplicationSet();
                if (clientApplicationSet.getElements().contains(application)) {
                    return davApplication;
                }
            }
        }
        return getConnection(application).getLocalDav();
    }

    /**
     * Gibt die Anmeldungen an einer Datenidentifikation aus Datenverteilersicht zurück
     *
     * @param object Objekt
     * @param atg    Attributgruppe
     * @param asp    Aspekt
     * @param dav    Datenverteiler
     *
     * @return Anmeldungsliste
     *
     * @throws IOException Falls nicht ermittelt werden konnte
     */
    public static ClientSubscriptionInfo getSubscriptionInfo(SystemObject object, AttributeGroup atg, Aspect asp, final DavApplication dav)
        throws IOException {
        return getSubscriptionInfo(object, atg, asp, (short) 0, dav);
    }

    /**
     * Gibt die Anmeldungen an einer Datenidentifikation aus Datenverteilersicht zurück
     *
     * @param object Objekt
     * @param atg    Attributgruppe
     * @param asp    Aspekt
     * @param simVar Simulationsvariante
     * @param dav    Datenverteiler
     *
     * @return Anmeldungsliste
     *
     * @throws IOException Falls nicht ermittelt werden konnte
     */
    public static ClientSubscriptionInfo getSubscriptionInfo(SystemObject object, AttributeGroup atg, Aspect asp, short simVar,
                                                             final DavApplication dav) throws IOException {
        AttributeGroupUsage attributeGroupUsage = atg.getAttributeGroupUsage(asp);
        if (attributeGroupUsage == null) {
            throw new IllegalArgumentException("attributeGroupUsage ist nicht vorhanden");
        }
        return getConnection(object).getSubscriptionInfo(dav, object, attributeGroupUsage, simVar);
    }

    /**
     * Gibt die Anmeldungen an einer Datenidentifikation aus Datenverteilersicht zurück
     *
     * @param object          Objekt
     * @param dataDescription Attributgruppe, Aspekt und Simulationsvariante
     * @param dav             Datenverteiler
     *
     * @return Anmeldungsliste
     *
     * @throws IOException Falls ein Fehler auftrat
     */
    public static ClientSubscriptionInfo getSubscriptionInfo(SystemObject object, DataDescription dataDescription, final DavApplication dav)
        throws IOException {
        return getSubscriptionInfo(object, dataDescription.getAttributeGroup(), dataDescription.getAspect(), dataDescription.getSimulationVariant(),
                                   dav);
    }

    /**
     * Gibt die Details zu einer Empfangsanmeldung zurück
     *
     * @param application     Applikation
     * @param object          Objekt
     * @param dataDescription Attributgruppe, Aspekt und Simulationsvariante
     *
     * @return Anmeldungsinfo zu Empfangsanmeldung oder null falls eine solche Anmeldung nicht ermittelt werden konnte
     *
     * @throws IOException Falls ein Fehler auftrat
     */
    public static ClientSubscriptionInfo.ClientReceivingSubscription getReceivingInfo(ClientApplication application, SystemObject object,
                                                                                      DataDescription dataDescription) throws IOException {
        ClientSubscriptionInfo subscriptionState = getSubscriptionInfo(object, dataDescription, findDav(application));
        for (ClientSubscriptionInfo.ClientReceivingSubscription receivingSubscription : subscriptionState.getReceiverSubscriptions()) {
            if (receivingSubscription.getApplicationId() == application.getId()) {
                return receivingSubscription;
            }
        }
        return null;
    }

    /**
     * Gibt die Details zu einer Sendenden Anmeldung zurück
     *
     * @param application     Applikation
     * @param object          Objekt
     * @param dataDescription Attributgruppe, Aspekt und Simulationsvariante
     *
     * @return Anmeldungsinfo zu Anmeldung oder null falls eine solche Anmeldung nicht ermittelt werden konnte
     *
     * @throws IOException Falls ein Fehler auftrat
     */
    public static ClientSubscriptionInfo.ClientSendingSubscription getSendingInfo(ClientApplication application, SystemObject object,
                                                                                  DataDescription dataDescription) throws IOException {
        ClientSubscriptionInfo subscriptionState = getSubscriptionInfo(object, dataDescription, findDav(application));
        for (ClientSubscriptionInfo.ClientSendingSubscription sendingSubscription : subscriptionState.getSenderSubscriptions()) {
            if (sendingSubscription.getApplicationId() == application.getId()) {
                return sendingSubscription;
            }
        }
        return null;
    }

    /**
     * Gibt den Status zu einer Empfangsanmeldung zurück
     *
     * @param application     Applikation
     * @param object          Objekt
     * @param dataDescription Attributgruppe, Aspekt und Simulationsvariante
     *
     * @return Anmeldungsinfo zu Empfangsanmeldung oder null falls eine solche Anmeldung nicht ermittelt werden konnte
     *
     * @throws IOException Falls ein Fehler auftrat
     */
    public static SubscriptionState getReceivingState(ClientApplication application, SystemObject object, DataDescription dataDescription)
        throws IOException {
        ClientSubscriptionInfo subscriptionState = getSubscriptionInfo(object, dataDescription, findDav(application));
        for (ClientSubscriptionInfo.ClientReceivingSubscription receivingSubscription : subscriptionState.getReceiverSubscriptions()) {
            if (receivingSubscription.getApplicationId() == application.getId()) {
                return receivingSubscription.getState();
            }
        }
        return null;
    }

    /**
     * Gibt den Status zu einer Sendenden Anmeldung zurück
     *
     * @param application     Applikation
     * @param object          Objekt
     * @param dataDescription Attributgruppe, Aspekt und Simulationsvariante
     *
     * @return Anmeldungsinfo zu Anmeldung oder null falls eine solche Anmeldung nicht ermittelt werden konnte
     *
     * @throws IOException Falls ein Fehler auftrat
     */
    public static SubscriptionState getSendingState(ClientApplication application, SystemObject object, DataDescription dataDescription)
        throws IOException {
        ClientSubscriptionInfo subscriptionState = getSubscriptionInfo(object, dataDescription, findDav(application));
        for (ClientSubscriptionInfo.ClientSendingSubscription sendingSubscription : subscriptionState.getSenderSubscriptions()) {
            if (sendingSubscription.getApplicationId() == application.getId()) {
                return sendingSubscription.getState();
            }
        }
        return null;
    }

    private static ClientDavConnection getConnection(final SystemObject systemObject) {
        return (ClientDavConnection) ((DafDataModel) systemObject.getDataModel()).getConnection();
    }

    /**
     * Wandelt die Anmeldungen einer Anmeldung in einen Debug-String, der dann z.B. als Text verglichen werden kann
     *
     * @param subscription Anmeldung
     *
     * @return Textuelle darstellung
     */
    public static String getDebugString(Subscription subscription) {
        StringBuilder result = new StringBuilder();
        switch (subscription.getConnectionState()) {
            case FROM_LOCAL_OK:
                result.append("Lokal_");
                break;
            case FROM_REMOTE_OK:
                result.append("Ein_");
                break;
            case TO_REMOTE_WAITING:
                result.append("Wartend_");
                break;
            case TO_REMOTE_OK:
                result.append("Positiv_");
                break;
            case TO_REMOTE_NOT_RESPONSIBLE:
                result.append("Negativ_");
                break;
            case TO_REMOTE_NOT_ALLOWED:
                result.append("KeineRechte_");
                break;
            case TO_REMOTE_MULTIPLE:
                result.append("Mehrere_");
                break;
        }
	    if (subscription instanceof SendingSubscription sendingSubscription) {
            switch (sendingSubscription.getState()) {
                case RECEIVERS_AVAILABLE:
                    result.append("Bereit_");
                    break;
                case NO_RECEIVERS:
                    result.append("KeineEmpfänger_");
                    break;
                case WAITING:
                    result.append("Wartend_");
                    break;
                case NOT_ALLOWED:
                    result.append("KeineRechte_");
                    break;
                case INVALID_SUBSCRIPTION:
                    result.append("Ungültig_");
                    break;
                case NO_REMOTE_SOURCE:
                    result.append("NichtZuständig_");
                    break;
                case MULTIPLE_REMOTE_LOCK:
                    result.append("Gesperrt_");
                    break;
            }
            if (sendingSubscription.isSource()) {
                result.append("Quelle_");
            }
            if (sendingSubscription.isRequestSupported()) {
                result.append("SendeSteuerung_");
            }
	    } else if (subscription instanceof ReceivingSubscription receivingSubscription) {
            switch (receivingSubscription.getState()) {
                case SENDERS_AVAILABLE:
                    result.append("Bereit_");
                    break;
                case NO_SENDERS:
                    result.append("KeineSender_");
                    break;
                case WAITING:
                    result.append("Wartend_");
                    break;
                case NOT_ALLOWED:
                    result.append("KeineRechte_");
                    break;
                case INVALID_SUBSCRIPTION:
                    result.append("Ungültig_");
                    break;
                case NO_REMOTE_DRAIN:
                    result.append("NichtZuständig_");
                    break;
                case MULTIPLE_REMOTE_LOCK:
                    result.append("Gesperrt_");
                    break;
            }
            if (receivingSubscription.isDrain()) {
                result.append("Senke_");
            }
            if (receivingSubscription.getReceiveOptions().withDelayed()) {
                result.append("Nachgeliefert_");
            }
            if (receivingSubscription.getReceiveOptions().withDelta()) {
                result.append("Delta_");
            }
        }
        result.append(subscription.getCommunication());
        return result.toString();
    }

    public static void compareSenders(SubscriptionInfo subscriptionInfo, String... expected) {
        if (subscriptionInfo == null) {
            assertTrue(expected.length == 0);
            return;
        }
        TreeSet<String> expectedSet = new TreeSet<>(Arrays.asList(expected));
        TreeSet<String> actual = new TreeSet<>();
        for (SendingSubscription sendingSubscription : subscriptionInfo.getSendingSubscriptions()) {
            actual.add(getDebugString(sendingSubscription));
        }
        Assert.assertEquals("Sender stimmen nicht überein", expectedSet, actual);
    }

    public static void compareReceivers(SubscriptionInfo subscriptionInfo, String... expected) {
        if (subscriptionInfo == null) {
            assertTrue(expected.length == 0);
            return;
        }
        TreeSet<String> expectedSet = new TreeSet<>(Arrays.asList(expected));
        TreeSet<String> actual = new TreeSet<>();
        for (final ReceivingSubscription receivingSubscription : subscriptionInfo.getReceivingSubscriptions()) {
            actual.add(getDebugString(receivingSubscription));
        }
        Assert.assertEquals("Empfänger stimmen nicht überein", expectedSet, actual);
    }

    private static class DataId {

        final SystemObject _object;
        final DataDescription _dataDescription;

        public DataId(final SystemObject object, final DataDescription dataDescription) {
            _object = object;
            _dataDescription = dataDescription;
        }

        @Override
        public String toString() {
            return _object.getPidOrId() + ":" + _dataDescription.getAttributeGroup().getPidOrId() + ":" + _dataDescription.getAspect().getPidOrId();
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
	        if (!(o instanceof DataId that)) {
                return false;
            }

            if (!(_object == that._object)) {
                return false;
            }
            return _dataDescription.equals(that._dataDescription);

        }

        @Override
        public int hashCode() {
            int result = _object.hashCode();
            result = 31 * result + _dataDescription.hashCode();
            return result;
        }
    }

    private static class DummySender implements ClientSenderInterface {
        private final ResultData _dataset;

        public DummySender(final ResultData dataset) {
            _dataset = dataset;
        }

        @Override
        public void dataRequest(final SystemObject object, final DataDescription dataDescription, final byte state) {
            if (state == START_SENDING) {
                try {
                    getConnection(object).sendData(_dataset);
                    getConnection(object).unsubscribeSender(this, object, dataDescription);
                } catch (SendSubscriptionNotConfirmed sendSubscriptionNotConfirmed) {
                    sendSubscriptionNotConfirmed.printStackTrace();
                }
            }
        }

        @Override
        public boolean isRequestSupported(final SystemObject object, final DataDescription dataDescription) {
            return true;
        }
    }
}
