/*
 * Copyright 2013-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.testumg.
 *
 * de.kappich.pat.testumg is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.testumg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.testumg.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.testumg.util;

import de.bsvrz.dav.daf.main.ClientDavConnection;
import de.bsvrz.dav.daf.main.ClientDavParameters;
import de.bsvrz.dav.daf.main.ClientReceiverInterface;
import de.bsvrz.dav.daf.main.ClientSenderInterface;
import de.bsvrz.dav.daf.main.CommunicationError;
import de.bsvrz.dav.daf.main.ConnectionException;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.InconsistentLoginException;
import de.bsvrz.dav.daf.main.MissingParameterException;
import de.bsvrz.dav.daf.main.NormalCloser;
import de.bsvrz.dav.daf.main.OneSubscriptionPerSendData;
import de.bsvrz.dav.daf.main.ReceiveOptions;
import de.bsvrz.dav.daf.main.ReceiverRole;
import de.bsvrz.dav.daf.main.ResultData;
import de.bsvrz.dav.daf.main.SendSubscriptionNotConfirmed;
import de.bsvrz.dav.daf.main.config.Aspect;
import de.bsvrz.dav.daf.main.config.AttributeGroup;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.dataIdentificationSettings.DataIdentification;
import de.bsvrz.sys.funclib.debug.Debug;
import java.io.IOException;
import java.io.StringReader;
import java.util.HashSet;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * Applikation die eine Parametrierung simuliert, und der man einfach mit {@link #publishParam(String, String, String)} die zu versendenden Parameter
 * übergeben kann
 *
 * @author Kappich Systemberatung
 */
public class FakeParamApp implements ClientSenderInterface {
    private static final Debug _debug = Debug.getLogger();
    private final Map<StringDataIdent, String> _savedParameters = new ConcurrentHashMap<>();
    private final HashSet<DataIdentification> _dataIdentifications = new HashSet<>();
    private ClientDavConnection _connection;
    private Aspect _paramAspect;

    /**
     * Mit einem DavStarter verbinden. Vom DavStarter werden die Verbindungsparameter übernommen.
     *
     * @param davStarter DavStarter
     *
     * @throws MissingParameterException
     * @throws CommunicationError
     * @throws ConnectionException
     * @throws InconsistentLoginException
     */
    public void connect(final DaVStarter davStarter)
        throws MissingParameterException, CommunicationError, ConnectionException, InconsistentLoginException {
        ClientDavParameters clientDavParameters = new ClientDavParameters();
        clientDavParameters.setApplicationTypePid("typ.parametrierungsApplikation");
        clientDavParameters.setApplicationName("FakeParameterApplikation");
        clientDavParameters.setDavCommunicationSubAddress(davStarter.getDavAppPort());

        _connection = new ClientDavConnection(clientDavParameters);
        // Applikationsfertigmeldung soll erst nach dem Publizieren aller Parameter gesendet werden.
        _connection.enableExplicitApplicationReadyMessage();
        _connection.setCloseHandler(new NormalCloser());
        _connection.connect();
        _connection.login("parameter", "parameter");
        _paramAspect = _connection.getDataModel().getAspect("asp.parameterSoll");
        if (_paramAspect == null) {
            throw new IllegalArgumentException("_paramAspect ist null");
        }
        for (Map.Entry<StringDataIdent, String> entry : _savedParameters.entrySet()) {
            publishParam(entry.getKey().objectPid, entry.getKey().atgPid, entry.getValue());
        }
        _connection.sendApplicationReadyMessage();
    }

    /**
     * Entfernt alle Datenanmeldungen, vergisst alle bereits gesetzten Parameter
     */
    public void clear() {
        _savedParameters.clear();
        for (DataIdentification dataIdentification : _dataIdentifications) {
            _connection.unsubscribeSender(this, dataIdentification.getObject(), dataIdentification.getDataDescription());
        }
    }

    public void publishParam(final SystemObject object, final AttributeGroup attributeGroup, final Data data) {
        publishParam(object.getPid(), attributeGroup.getPid(), DataUtil.serializeData(data));
    }

    /**
     * Veröffentlicht einen Parameter. Das Format des Datenparameters entspricht etwa JSON oder der toString()-Ausgabe von Daten. Die Attributgruppe
     * darf hier aber nicht noch einmal angegeben werden. Folgende Regeln gelten:
     * <ul>
     *     <li>
     *         Listen werden durch geschweifte Klammern dargestellt. Einzelne Elemente werden durch Kommas getrennt.
     *         Elemente werden dargestellt durch <i>Name</i>:<i>Wert</i>. Beispiel:
     *         {@code {attribut1:"Test",attribut2:{bla:"2",foo:"bar"}}}.
     *     </li>
     *     <li>
     *         Arrays werden durch eckige Klammern dargestellt. Einzelne Werte werden durch Kommas getrennt.
     *         Beispiel:
     *         {@code ["abc","def","ghi"]}.
     *         Soll nur ein Wert gespeichert werden, können die eckigen Klammern weggelassen werden.
     *     </li>
     *     <li>
     *         Primitive Werte <b>müssen</b> in Anführungszeichen (einfach oder doppelt) gesetzt werden.
     *     </li>
     * </ul>
     * Überflüssige Leerzeichen können Probleme verursachen und sind zu vermeiden. Attribute, die auf dem Standardwert
     * bleiben sollen, brauchen nicht aufgeführt zu werden.
     *
     * @param objPid Pid des Objektes
     * @param atgPid Pid der Parameter-Attributgruppe
     * @param data   Daten als String.
     *
     * @throws IllegalArgumentException bei ungültigen Parametern
     */
    public void publishParam(String objPid, String atgPid, String data) {
        if (data != null) {
            _savedParameters.put(new StringDataIdent(objPid, atgPid), data);
        } else {
            _savedParameters.remove(new StringDataIdent(objPid, atgPid));
        }

        if (_paramAspect == null) {
            return;
        }

        AttributeGroup attributeGroup = _connection.getDataModel().getAttributeGroup(atgPid);
        SystemObject object = _connection.getDataModel().getObject(objPid);
        if (attributeGroup == null) {
            _debug.error("Unbekannte Attributgruppe: " + atgPid);
            return;
        }
        if (object == null) {
            _debug.error("Unbekanntes Objekt: " + objPid);
            return;
        }

        Data dataObject = null;
        if (data != null) {
            dataObject = _connection.createData(attributeGroup);
            try {
                DataUtil.deserializeData(new StringReader(data), dataObject);
            } catch (IOException e) {
                throw new IllegalArgumentException(e);
            }
            dataObject = dataObject.createUnmodifiableCopy();
        }
        DataDescription dataDescription = new DataDescription(attributeGroup, _paramAspect);
        final ResultData resultData = new ResultData(object, dataDescription, System.currentTimeMillis(), dataObject);
        _dataIdentifications.add(new DataIdentification(resultData.getObject(), resultData.getDataDescription()));
        try {
            _connection.subscribeSource(this, resultData);
        } catch (OneSubscriptionPerSendData ignored) {
            try {
                _connection.sendData(resultData);
            } catch (SendSubscriptionNotConfirmed sendSubscriptionNotConfirmed) {
                throw new IllegalStateException(sendSubscriptionNotConfirmed);
            }
        }

        // Warten, bis versandt
        final AtomicBoolean hasData = new AtomicBoolean();
        _connection.subscribeReceiver(new ClientReceiverInterface() {
            @Override
            public void update(final ResultData[] results) {
                for (ResultData result : results) {
                    if (result.getDataTime() == resultData.getDataTime() &&
                        String.valueOf(result.getData()).equals(String.valueOf(resultData.getData()))) {
                        synchronized (FakeParamApp.this) {
                            hasData.set(true);
                            FakeParamApp.this.notifyAll();
                        }
                    }
                }
            }
        }, object, dataDescription, ReceiveOptions.normal(), ReceiverRole.receiver());
        synchronized (this) {
            try {
                while (!hasData.get() && _connection.isConnected()) {
                    wait(1000);
                }
                wait(100);
            } catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
    }

    public ClientDavConnection getConnection() {
        return _connection;
    }

    @Override
    public void dataRequest(final SystemObject object, final DataDescription dataDescription, final byte state) {
    }

    @Override
    public boolean isRequestSupported(final SystemObject object, final DataDescription dataDescription) {
        return false;
    }

    static class StringDataIdent {
        final String objectPid;
        final String atgPid;

        public StringDataIdent(final String objPid, final String atgPid) {
            this.objectPid = objPid;
            this.atgPid = atgPid;
        }

        @Override
        public boolean equals(final Object o) {
            if (this == o) {
                return true;
            }
	        if (!(o instanceof StringDataIdent that)) {
                return false;
            }

            if (!objectPid.equals(that.objectPid)) {
                return false;
            }
            return atgPid.equals(that.atgPid);

        }

        @Override
        public int hashCode() {
            int result = objectPid.hashCode();
            result = 31 * result + atgPid.hashCode();
            return result;
        }
    }
}
