/*
 * Copyright 2006-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.testumg.
 *
 * de.kappich.pat.testumg is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.testumg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.testumg.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.testumg.util;

import de.kappich.pat.testumg.essentialconfigareafiles.Helper;
import java.io.BufferedInputStream;
import java.io.BufferedReader;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;

/**
 * Diese Klasse stellt statische Methoden zur Verfügung, mit denen Dateien und Verzeichnisse kopiert und gelöscht werden können.
 *
 * @author Achim Wullenkord (AW), Kappich Systemberatung
 * @version $Revision$ / $Date$ / ($Author$)
 */
public final class FileCopy {

    /**
     * Erstellt eine Kopie einer Datei. Es werden alle benötigten Verzeichnisse angelegt.
     *
     * @param sourceFile                  Datei, die kopiert werden soll
     * @param destinationFile             Datei, die erstellt werden soll
     * @param overwriteDestinationAllowed Falls bereits eine Datei(destinationFile) existiert, soll diese überschrieben werden (true = ja)
     *
     * @throws java.io.IOException Fehler beim kopieren der Datei
     */
    public static void copyFile(File sourceFile, File destinationFile, boolean overwriteDestinationAllowed) throws IOException {
        final InputStream in = new FileInputStream(sourceFile);
        copyFile(in, destinationFile, overwriteDestinationAllowed);
    }

    /**
     * Erstellt eine Kopie einer Datei. Es werden alle benötigten Verzeichnisse angelegt.
     *
     * @param sourceFile                  Datei, die kopiert werden soll
     * @param destinationFile             Datei, die erstellt werden soll
     * @param overwriteDestinationAllowed Falls bereits eine Datei(destinationFile) existiert, soll diese überschrieben werden (true = ja)
     *
     * @throws java.io.IOException Fehler beim kopieren der Datei
     */
    public static void copyFile(InputStream sourceFile, File destinationFile, boolean overwriteDestinationAllowed) throws IOException {
        if (overwriteDestinationAllowed || !destinationFile.exists()) {
            if (destinationFile.exists() && destinationFile.isFile()) {
                destinationFile.delete();
            }

            File parent = new File(destinationFile.getParent());
            if (!parent.exists()) {
                parent.mkdirs();
            }

	        try (BufferedInputStream in = new BufferedInputStream(sourceFile); FileOutputStream out = new FileOutputStream(destinationFile)) {
                byte[] buffer = new byte[8 * 1024];
                int count = 0;
                do {
                    out.write(buffer, 0, count);
                    count = in.read(buffer, 0, buffer.length);
                }
                while (count != -1);
            }
        }
    }

    /**
     * Kopiert den Inhalt des übergebenen Verzeichnisses {@code sourceDirectory} in das angegebene Verzeichnis {@code destinationDirectory}. Es werden
     * alle Dateien und Unterverzeichnisse kopiert. Die identische Struktur ist danach im angegebenen Verzeichnis {@code destinationDirectory}
     * enthalten. <p> Enthält das Verzeichnis {@code destinationDirectory} Dateien und/oder Untervzeichnisse wird eine Exception geworfen.
     *
     * @param sourceDirectory      Verzeichnis, dessen Struktur kopiert werden soll
     * @param destinationDirectory Leeres Verzeichnis, in dem die Verzeichnisstruktor von {@code sourceDirectory} entstehen soll. Ist das Verzeichnis
     *                             nicht vorhanden, wird es angelegt.
     *
     * @throws IllegalArgumentException Das übergebene Verzeichnis {@code destinationDirectory} war nicht leer, sondern enthielt Dateien und/oder
     *                                  Verzeichnisse
     * @throws IllegalStateException    Das zu kopierende Verzeichnis ist nicht vorhanden oder ist eine Datei
     */
    public static void copyDirectory(File sourceDirectory, File destinationDirectory) throws IllegalArgumentException, IOException {

        System.out.println("zu kopierendes Verzeichnis: " + sourceDirectory.getAbsolutePath());

        if (!sourceDirectory.exists()) {
            // Das zu kopierende Verzeichnis existiert nicht
            throw new IllegalStateException("Das zu kopierende Verzeichnis ist nicht vorhanden: " + sourceDirectory.getAbsolutePath());
        }

        if (!sourceDirectory.isDirectory()) {
            // Es wurde eine Datei angegeben
            throw new IllegalStateException(
                "Das zu kopierende Verzeichnis ist kein Verzeichnis sondern eine Datei: " + sourceDirectory.getAbsolutePath());
        }

        // Prüfen ob die Grundbediengungen vorhanden sind
        if (!destinationDirectory.exists()) {
            // Verzeichnis anlegen
            destinationDirectory.mkdirs();
        } else {
            if (!destinationDirectory.isDirectory()) {
                throw new IllegalStateException("Es wurde kein Verzeichnis übergeben: " + destinationDirectory.getAbsolutePath());
            } else {
                // Das Verzeichnis muss leer sein
                final File[] emptyDirectory = destinationDirectory.listFiles();
                if (emptyDirectory.length > 0) {
                    throw new IllegalArgumentException(
                        "Das Verzeichnis " + destinationDirectory.getAbsolutePath() + " enthält Dateien/Verzeichnisse");
                }
            }
        }

        // kopieren starten
        copyAll(sourceDirectory, destinationDirectory);
    }

    /**
     * Erstellt eine Kopie eines Verzeichnisses. Befindet sich die Kopie innerhalb des Verzeichnisses, das kopiert werden soll, so wird die Kopie
     * nicht doppelt erstellt.
     *
     * @param sourceDirectory      Verzeichnis, das kopiert werden soll
     * @param destinationDirectory Verzeichnis, in das kopiert werden soll
     *
     * @throws IOException
     */
    private static void copyAll(File sourceDirectory, File destinationDirectory) throws IOException {
        if (sourceDirectory.isFile()) {
            // Rekursionsende, Datei kopieren

            // An diesem Punkt wurden 2 Dateien übergeben. Das original und die Datei, in die kopiert werden soll
            copyFile(sourceDirectory, destinationDirectory, true);
            return;
        } else {
            // Alle Kinder betrachten und Rekursiv tiefer absteigen.

            // Ein Verzeichnis, alle "Kinder" anfordern (Das können Verzeichnisse aber auch Dateien sein)
            final File[] filesAndDirectories = sourceDirectory.listFiles();

            for (File file : filesAndDirectories) {

                // Wenn das Verzeichnis, in das kopiert werden soll, innerhalb des Verzeichnisses liegt, das kopiert wird würden
                // ohne diese Abfrage 2 Kopien entstehen.
                // Die "zweite Kopie" liegt dann innerhalb des Zielverzeichnisses (richtiger Weise, aber dieser Effekt(Kopie einer Kopie) ist nicht
                // gewünscht).

                if (!file.equals(destinationDirectory)) {
                    // Da im alten Verzeichnis weiter rekursiv abgestiegen wird, muss dieses Stück am neuen Verzeichnis angefügt werden
                    final String fileOrDirectoryName = file.getName();

                    final File expandedDestinationDirectory = new File(destinationDirectory, fileOrDirectoryName);
                    copyAll(file, expandedDestinationDirectory);
                }
            }
        }
    }

    /**
     * Löscht eine Datei oder ein gesamtes Verzeichnis mit allen Unterstrukturen.
     *
     * @param directoryOrFile Verzeichnis oder Datei, das/die gelöscht werden soll
     */
    public static void deleteDirectoryOrFile(File directoryOrFile) {
        // Der Algorithmus durchläuft rekursiv die Struktur.

        // Um ein Verzeichnis zu löschen, muss es leer sein. Befinden sich aber noch Dateien/Verzeichnisse in dem zu löschenden Verzeichnis,
        // müssen diese zuerst gelöscht werden.

        if (directoryOrFile.isDirectory()) {
            // Das Verzeichnis muss leer sein, damit es gelöscht werden kann.
            final File[] files = directoryOrFile.listFiles();
            if (files.length > 0) {
                // Alle Kinder des Verzeichnisses löschen
                for (File file : files) {
                    deleteDirectoryOrFile(file);
                }
                // An dieser Stelle ist das Verzeichnis leer und kann gelöscht werden
            }
        }

        // Datei oder leeres Verzeichnis löschen (Es kann zu einem Fehler kommen, wenn das File nicht existiert)
        directoryOrFile.delete();
    }

    /**
     * Kopiert eine Verwaltungsdatei und die dazugehörigen Konfigurationsbereiche in das angegebene Verzeichnis. Eine Konfiguration kann mit der
     * kopierten Verwaltunsdatei gestartet werden. <p>
     *
     * @param destinationDirectory Verzeichnis, in das die Verwaltungsdatei und die Konfigurationsbereiche kopiert werden sollen. Ist das Verzeichnis
     *                             nicht vorhanden, wird es angelegt. Das Verzeichnis muss leer sein (keine Dateien oder Verzeichnisse enthalten).
     *
     * @throws IOException Fehler beim kopieren der Daten
     */
    public static void copyTestConfigurationAreaFiles(final File destinationDirectory) throws IOException {

        if (!destinationDirectory.exists()) {
            // Verzeichnis anlegen
            destinationDirectory.mkdirs();
        } else {
            // Das Verzeichnis muss leer sein
            final File[] emptyDirectory = destinationDirectory.listFiles();
            if (emptyDirectory.length > 0) {
                throw new IllegalArgumentException("Das Verzeichnis " + destinationDirectory.getAbsolutePath() + " enthält Dateien/Verzeichnisse");
            }
        }

        // Es müssen alle Dateien, die zum starten der Konfiguration gebraucht werden, kopiert werden.
        // Da sich die Dateien in einer JAR Datei befinden können, müssen die Dateinamen, die kopiert werden sollen, bekannt sein.
        // Alle zu kopierenden Dateien befinden sich in einer Textdatei mit dem Namen "allfiles.txt".

        // Alle Dateien, die kopiert werden müssen
        final List<String> allFiles = getAllFileNames();

        for (String fileName : allFiles) {
            final URL resource = Helper.class.getResource(fileName);
            if (resource == null) {
                throw new IllegalStateException("Resource mit dem Namen " + fileName + " konnte nicht gefunden werden");
            }
            final InputStream original = resource.openStream();
            final File copyOfFile = new File(destinationDirectory, fileName);
            copyFile(original, copyOfFile, true);
        }
    }

    /**
     * Kopiert eine Verwaltungsdatei und die dazugehörigen Konfigurationsbereiche in das angegebene Verzeichnis. Eine Konfiguration kann mit der
     * kopierten Verwaltunsdatei gestartet werden. <p>
     *
     * @param destinationDirectory Verzeichnis, in das die Verwaltungsdatei und die Konfigurationsbereiche kopiert werden sollen. Ist das Verzeichnis
     *                             nicht vorhanden, wird es angelegt. Das Verzeichnis muss leer sein (keine Dateien oder Verzeichnisse enthalten).
     *
     * @throws IOException Fehler beim kopieren der Daten
     */
    public static void copyTestConfigurationAreaFilesWithDeleting(final File destinationDirectory) throws IOException {

        if (destinationDirectory.exists()) {
            // Verzeichnis löschen
            deleteDirectoryOrFile(destinationDirectory);
        }
        //bisherige Methode aufrufen
        copyTestConfigurationAreaFiles(destinationDirectory);
    }

    /**
     * Liest aus der Datei {@code allfiles.properties} die Dateinamen aus.
     *
     * @return Dateinamen
     *
     * @throws IOException Fehler beim lesen des Streams
     */
    private static final List<String> getAllFileNames() throws IOException {
        List<String> allFiles = new ArrayList<>();
        final URL filePathAsURL = Helper.class.getResource("allfiles.properties");

	    try (InputStream inputStreamAllFiles = filePathAsURL.openStream()) {

            final BufferedReader buffReader = new BufferedReader(new InputStreamReader(inputStreamAllFiles));
            String fileName = buffReader.readLine();

            while (fileName != null) {
                allFiles.add(fileName);
                fileName = buffReader.readLine();
            }
        }
        return allFiles;
    }

    /**
     * Diese Methode entfernt aus einem Verzeichnis alle Dateien deren Endung mit dem Parameter {@code fileEnding} definiert sind. Enthält das
     * Verzeichnis weitere Unterverzeichnisse, werden diese nicht berücksichtigt.
     *
     * @param directory  Verzeichnis, aus dem bestimmte Dateien entfernt werden sollen.
     * @param fileEnding Endung der Dateien, die entfernt werden sollen. Der Punkt muss mit angegeben werden. Beispiele(ohne ""): ".exe", ".lock",
     *                   ".bat", usw.. Die Eingabe wird in {@code lowerCase} umgewandelt (".exe" == ".EXE" == ".Exe").
     */
    public static void removeFiles(final File directory, final String fileEnding) {
        if (directory.isDirectory()) {

            // Alle Dateien, die gelöscht werden sollen und deren Endung stimmt
            final File[] delteFiles = directory.listFiles(new FilenameFilter() {
                public boolean accept(File dir, String name) {
                    return name.toLowerCase().endsWith(fileEnding.toLowerCase());
                }
            });

            for (File delteFile : delteFiles) {
                if (delteFile.isFile()) {
                    delteFile.delete();
                }
            }
        } else {
            throw new IllegalArgumentException("Es wurde kein Verzeichnis übergeben: " + directory.getAbsoluteFile());
        }
    }
}
