/*
 * Copyright 2011-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.testumg.
 *
 * de.kappich.pat.testumg is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.testumg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.testumg.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.testumg.util;

import de.kappich.pat.testumg.util.connections.DavInformation;
import de.kappich.pat.testumg.util.connections.LocalDavDavConnection;
import de.kappich.pat.testumg.util.connections.LocalDavDavConnectionServer;
import de.kappich.pat.testumg.util.connections.MultiDavProtocolParameter;

import de.bsvrz.dav.daf.accessControl.AccessControlMode;
import de.bsvrz.dav.daf.main.*;
import de.bsvrz.dav.daf.main.authentication.ClientCredentials;
import de.bsvrz.dav.daf.main.config.ConfigurationChangeException;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.management.consistenycheck.ConsistencyCheckResultInterface;
import de.bsvrz.puk.config.main.managementfile.ManagementFile;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

import com.google.common.collect.ImmutableList;

import java.io.*;
import java.nio.file.*;
import java.text.DateFormat;
import java.util.*;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;

/**
 * Klasse, die das Testen von mehreren verbundenen Datenverteilern erlaubt.
 * <p>
 * Ein Beispiel für einen einfachen Testfall ist de.kappich.pat.testumg.util.TestMultiDavTestEnvironment.
 *
 * @author Kappich Systemberatung
 */
public class MultiDavTestEnvironment {

    private final Path workingDir;

    private final Class<?> _testClass;

    private final String _configurationDebugLevel;

    private final String _transmitterDebugLevel;

    private final String _paramDebugLevel;

    private final HashMap<String, MultiDavStarter> _starterMap = new HashMap<>();
    private final Collection<DavDavConnection> _davConnections = new ArrayList<>();
    private final List<ClientDavInterface> _connections = new CopyOnWriteArrayList<>();
    private int _appPortCounter = 13081;
    private volatile Semaphore _semaphore;
    private ConfigurationController _globalConfigurationController;
    /**
     * Verzögerungszeit, die innerhalb des Datenverteilers gewartet wird, bevor Verbindungen zu anderen Datenverteilern zugelassen bzw. aufgebaut
     * werden.
     */
    private int _davDavConnectDelay = 500;

    /** Verzögerungszeit, die innerhalb des Datenverteilers gewartet wird, bevor versucht wird, abgebrochene Verbindungen erneut aufzubauen. */
    private int _davDavReconnectDelay = 1000;

    /** Ob schon der initiale Konfigurationsbereich erstellt wurde. Danach sind Änderungen an den dav und den Verbindungen nicht mehr möglich */
    private boolean _hasActivated;

    /** Standard-Konstruktor */
    public MultiDavTestEnvironment() {
        this("WARNING", "ERROR", "ERROR");
    }

    /**
     * Konstruktor mit vorgegeben Debug-Leveln
     *
     * @param transmitterDebugLevel   Debug-Level für Datenverteiler. Mögliche Werte sind "ERROR", "WARNING", "CONFIG", "INFO", "FINE", "FINER",
     *                                "FINEST" und "ALL".
     * @param configurationDebugLevel Debug-Level für Konfigurationen. Mögliche Werte sind "ERROR", "WARNING", "CONFIG", "INFO", "FINE", "FINER",
     *                                "FINEST" und "ALL".
     * @param paramDebugLevel         Debug-Level für Parametrierungen. Mögliche Werte sind "ERROR", "WARNING", "CONFIG", "INFO", "FINE", "FINER",
     *                                "FINEST" und "ALL".
     */
    public MultiDavTestEnvironment(final String transmitterDebugLevel, final String configurationDebugLevel, final String paramDebugLevel) {
        Debug.init("DAV", new ArgumentList(new String[] {"-debugLevelStdErrText=" + transmitterDebugLevel}));
//		Debug.getLogger().setLoggerLevel(Level.parse(transmitterDebugLevel));
        _configurationDebugLevel = configurationDebugLevel;
        _transmitterDebugLevel = transmitterDebugLevel;
        _paramDebugLevel = paramDebugLevel;
        _testClass = SingleDavStarter.getTestClass(MultiDavTestEnvironment.class);
        workingDir = TempDirectoryCreator.createTemporaryDirectory().resolve(_testClass.getSimpleName());

        // Zuvor deaktivierte Verbindungen in jedem Fall wieder aktivieren, das ist jetzt ja ein neuer Test.
        LocalDavDavConnection.enableAll();
    }

    public static void dumpThreads(PrintStream out) {
        out.println();
        out.println(DateFormat.getInstance().format(System.currentTimeMillis()));
        out.println("Full thread dump");
        Map<Thread, StackTraceElement[]> allStackTraces = Thread.getAllStackTraces();
        for (Map.Entry<Thread, StackTraceElement[]> entry : allStackTraces.entrySet()) {
            out.println();
            Thread thread = entry.getKey();
            out.printf("\"%s@%d\"%s prio=%d tid=0x%x%n", thread.getName(), thread.getId(), thread.isDaemon() ? " daemon" : "", thread.getPriority(),
                       thread.getId());
            StackTraceElement[] trace = entry.getValue();
            for (StackTraceElement traceElement : trace) {
                out.println("\t  at " + traceElement);
            }
        }
    }

    /**
     * Erstellt einen Datenverteiler mit eigener Konfiguration. Der Datenverteiler wird nicht automatisch gestartet, dies geschieht erst bei
     * startAll() oder startDav()
     *
     * @param name Name des Datenverteilers
     *
     * @return Objekt, dass den Datenverteiler kapselt (Benutzung ist nicht nötig)
     *
     * @throws Exception -
     */
    public MultiDavStarter createDavWithLocalConfiguration(final String name) throws Exception {
        return createDavWithLocalConfiguration(name, AccessControlMode.Disabled);
    }

    /**
     * Erstellt einen Datenverteiler mit eigener Konfiguration. Der Datenverteiler wird nicht automatisch gestartet, dies geschieht erst bei
     * startAll() oder startDav()
     *
     * @param name                 Name des Datenverteilers
     * @param accessControlType    Soll die Zugriffssteuerung aktiviert werden? (Standardmäßig nein)
     * @param accessControlPlugIns Zugriffssteuerungs-Plugins
     *
     * @return Objekt, dass den Datenverteiler kapselt (Benutzung ist nicht nötig)
     *
     * @throws Exception -
     */
    public MultiDavStarter createDavWithLocalConfiguration(final String name, final AccessControlMode accessControlType,
                                                           final String... accessControlPlugIns) throws Exception {
        if (_hasActivated) {
            throw new IllegalStateException("Konfiguration bereits aktiviert");
        }
        final int nextAppPort = nextFreePort();
        final MultiDavStarter starter = new MultiDavStarter(name, accessControlType, accessControlPlugIns, nextAppPort, nextAppPort - 1);
        _starterMap.put(name, starter);
        return starter;
    }

    /**
     * Erstellt einen Datenverteiler mit entfernter Konfiguration. Der Datenverteiler wird nicht automatisch gestartet, dies geschieht erst bei
     * startAll() oder startDav()
     *
     * @param name          Name des Datenverteilers
     * @param remoteDavName Name des Datenverteilers, an dem die Konfiguration läuft
     *
     * @return Objekt, dass den Datenverteiler kapselt (Benutzung ist nicht nötig)
     *
     * @throws Exception -
     */
    public MultiDavStarter createDavWithRemoteConfiguration(final String name, final String remoteDavName) throws Exception {
        return createDavWithRemoteConfiguration(name, remoteDavName, AccessControlMode.Disabled);
    }

    /**
     * Erstellt einen Datenverteiler mit entfernter Konfiguration. Der Datenverteiler wird nicht automatisch gestartet, dies geschieht erst bei
     * startAll() oder startDav()
     *
     * @param name          Name des Datenverteilers
     * @param remoteDavName Name des Datenverteilers, an dem die Konfiguration läuft
     * @param passivePort   Port für passiven Verbindungsaufbau
     *
     * @return Objekt, dass den Datenverteiler kapselt (Benutzung ist nicht nötig)
     *
     * @throws Exception -
     */
    public MultiDavStarter createDavWithRemoteConfiguration(final String name, final String remoteDavName, final int passivePort) throws Exception {
        return createDavWithRemoteConfiguration(name, remoteDavName, passivePort, AccessControlMode.Disabled);
    }

    /**
     * Erstellt einen Datenverteiler mit entfernter Konfiguration. Der Datenverteiler wird nicht automatisch gestartet, dies geschieht erst bei
     * startAll() oder startDav()
     *
     * @param name                 Name des Datenverteilers
     * @param remoteDavName        Name des Datenverteilers, an dem die Konfiguration läuft
     * @param accessControlType    Soll die Zugriffssteuerung aktiviert werden? (Standardmäßig nein)
     * @param accessControlPlugIns Zugriffssteuerungs-Plugins
     *
     * @return Objekt, dass den Datenverteiler kapselt (Benutzung ist nicht nötig)
     *
     * @throws Exception -
     */
    public MultiDavStarter createDavWithRemoteConfiguration(final String name, final String remoteDavName, final AccessControlMode accessControlType,
                                                            final String... accessControlPlugIns) throws Exception {
        return createDavWithRemoteConfiguration(name, remoteDavName, 0, accessControlType, accessControlPlugIns);
    }

    /**
     * Erstellt einen Datenverteiler mit entfernter Konfiguration. Der Datenverteiler wird nicht automatisch gestartet, dies geschieht erst bei
     * startAll() oder startDav()
     *
     * @param name                 Name des Datenverteilers
     * @param remoteDavName        Name des Datenverteilers, an dem die Konfiguration läuft
     * @param passivePort          Port für passiven Verbindungsaufbau
     * @param accessControlType    Soll die Zugriffssteuerung aktiviert werden? (Standardmäßig nein)
     * @param accessControlPlugIns Zugriffssteuerungs-Plugins
     *
     * @return Objekt, dass den Datenverteiler kapselt (Benutzung ist nicht nötig)
     *
     * @throws Exception -
     */
    public MultiDavStarter createDavWithRemoteConfiguration(final String name, final String remoteDavName, final int passivePort,
                                                            final AccessControlMode accessControlType, final String... accessControlPlugIns)
        throws Exception {
        if (_hasActivated) {
            throw new IllegalStateException("Konfiguration bereits aktiviert");
        }
        final int nextAppPort = nextFreePort();
        final MultiDavStarter starter =
            new MultiDavStarter(name, accessControlType, accessControlPlugIns, nextAppPort, nextAppPort - 1, passivePort, remoteDavName);
        _starterMap.put(name, starter);
        return starter;
    }

    /**
     * Unterbricht die angegebene Verbindung zwischen den beiden Datenverteilern
     *
     * @param name  Datenverteiler1-Name
     * @param name2 Datenverteiler2-Name
     */
    public void interruptConnection(final String name, final String name2) {
        final int port = getExistingDav(name2).getDavPort();
        final int port2 = getExistingDav(name).getDavPort();
        LocalDavDavConnection.disableConnection(name, port);
        LocalDavDavConnection.disableConnection(name2, port2);
    }

    /**
     * Stellt die angegebene Verbindung wieder her
     *
     * @param name  Datenverteiler1-Name
     * @param name2 Datenverteiler2-Name
     */
    public void restoreConnection(final String name, final String name2) {
        final int port = getExistingDav(name2).getDavPort();
        final int port2 = getExistingDav(name).getDavPort();
        LocalDavDavConnection.enableConnection(name, port);
        LocalDavDavConnection.enableConnection(name2, port2);
    }

    /**
     * Erstellt eine Datenverteiler-Verbindung zwischen 2 Datenverteilern. Diese muss vor dem Starten der Datenverteiler festgelegt werden.
     *
     * @param name  Datenverteiler1-Name
     * @param name2 Datenverteiler2-Name
     */
    public void createDavConnection(final String name, final String name2) {
        createDavConnection(name, name2, "AB");
    }

    /**
     * Erstellt eine Datenverteiler-Verbindung zwischen 2 Datenverteilern. Diese muss vor dem Starten der Datenverteiler festgelegt werden.
     *
     * @param name       Datenverteiler1-Name
     * @param name2      Datenverteiler2-Name
     * @param originator entweder "A", "B", "AB", oder "-" zum bestimmen, welcher Datenverteiler die Verbindung initiieren soll
     */
    public void createDavConnection(final String name, final String name2, final String originator) {
        createDavConnection(name, name2, originator, 1);
    }

    /**
     * Erstellt eine Datenverteiler-Verbindung zwischen 2 Datenverteilern. Diese muss vor dem Starten der Datenverteiler festgelegt werden.
     *
     * @param name       Datenverteiler1-Name
     * @param name2      Datenverteiler2-Name
     * @param originator entweder "A", "B", "AB", oder "-" zum bestimmen, welcher Datenverteiler die Verbindung initiieren soll
     */
    public void createDavConnection(final String name, final String name2, final String originator, final String user1, final String user2,
                                    @Nullable final String... replacementConnections) {
        createDavConnection(name, name2, originator, user1, user2, 1, replacementConnections);
    }

    /**
     * Erstellt eine Datenverteiler-Verbindung zwischen 2 Datenverteilern. Diese muss vor dem Starten der Datenverteiler festgelegt werden.
     *
     * @param name                   Datenverteiler1-Name
     * @param name2                  Datenverteiler2-Name
     * @param originator             entweder "A", "B", "AB", oder "-" zum bestimmen, welcher Datenverteiler die Verbindung initiieren soll
     * @param weight                 Konfiguriertes Gewicht der Verbindung, zur Ermittlung der besten Wege
     * @param replacementConnections Ersatzverbindungen. null: Automatisch bestimmen. Andernfalls eine Folge von jeweils 2 Datenverteiler-Namen, die
     *                               eine Verbindung repräsentieren z.B. "Dav1", "Dav2" für eine Verbindung von "Dav1" nach "Dav2".
     */
    public void createDavConnection(final String name, final String name2, final String originator, final int weight,
                                    @Nullable final String... replacementConnections) {
        createDavConnection(name, name2, originator, "Tester", "Tester", weight, replacementConnections);
    }

    /**
     * Erstellt eine Datenverteiler-Verbindung zwischen 2 Datenverteilern. Diese muss vor dem Starten der Datenverteiler festgelegt werden.
     *
     * @param name                   Datenverteiler1-Name
     * @param name2                  Datenverteiler2-Name
     * @param originator             entweder "A", "B", "AB", oder "-" zum bestimmen, welcher Datenverteiler die Verbindung initiieren soll
     * @param user1
     * @param user2
     * @param weight                 Konfiguriertes Gewicht der Verbindung, zur Ermittlung der besten Wege
     * @param replacementConnections Ersatzverbindungen. null: Automatisch bestimmen. Andernfalls eine Folge von jeweils 2 Datenverteiler-Namen, die
     *                               eine Verbindung repräsentieren
     */
    public void createDavConnection(final String name, final String name2, final String originator, final String user1, final String user2,
                                    final int weight, @Nullable final String... replacementConnections) {
        if (_hasActivated) {
            throw new IllegalStateException("Konfiguration bereits aktiviert");
        }
        List<String> repConn;
        if (replacementConnections == null) {
            repConn = null;
        } else {
            repConn = new ArrayList<>();
            for (int i = 0; i < replacementConnections.length; i += 2) {
                repConn.add("davcon.test." + replacementConnections[i] + "." + replacementConnections[i + 1]);
            }
        }

        _davConnections.add(new DavDavConnection(name, name2, originator, weight, repConn, user1, user2));
    }

    /**
     * Erstellt eine Datenverteiler-Verbindung zwischen 2 Datenverteilern. Diese muss vor dem Starten der Datenverteiler festgelegt werden.
     *
     * @param name                   Datenverteiler1-Name
     * @param name2                  Datenverteiler2-Name
     * @param originator             entweder "A", "B", "AB", oder "-" zum bestimmen, welcher Datenverteiler die Verbindung initiieren soll
     * @param weight                 Konfiguriertes Gewicht der Verbindung, zur Ermittlung der besten Wege
     * @param transmissionDelay      Simulierte minimale Übertragungszeit eines Pakets in Millisekunden zwischen den beiden Datenverteilern zum
     *                               simulieren von "hohen Pings" zwischen den Datenverteilern. Standardmäßig 0.
     * @param maxFlowRate            Simulierte maximale Datenrate in Bytes/Sekunde zwischen den beiden Datenverteilern zum simulieren von langsamen
     *                               Verbindungen zwischen den Datenverteilern. Standardmäßig Double.POSITIVE_INFINITY.
     * @param replacementConnections Ersatzverbindungen. null: Automatisch bestimmen. Andernfalls eine Folge von jeweils 2 Datenverteiler-Namen, die
     *                               eine Verbindung repräsentieren z.B. "Dav1", "Dav2" für eine Verbindung von "Dav1" nach "Dav2".
     */
    public void createDavConnection(final String name, final String name2, final String originator, final int weight, final double transmissionDelay,
                                    final double maxFlowRate, final String... replacementConnections) {
        if (_hasActivated) {
            throw new IllegalStateException("Konfiguration bereits aktiviert");
        }
        createDavConnection(name, name2, originator, weight, replacementConnections);
        final int port = getExistingDav(name).getDavPort();
        final int port2 = getExistingDav(name2).getDavPort();
        LocalDavDavConnection.setConnectionDelay(name, port2, (long) (transmissionDelay * 1_000_000.0), maxFlowRate);
        LocalDavDavConnection.setConnectionDelay(name2, port, (long) (transmissionDelay * 1_000_000.0), maxFlowRate);
    }

    /**
     * Startet alle Datenverteiler-Systeme (ggf. inklusive Konfiguration und Parametrierung) ohne zu warten, bis diese hochgefahren sind
     */
    public void startAllDavWithoutWaiting() {
        startAllDavWithoutWaiting(false);
    }

    /**
     * Startet alle Datenverteiler-Systeme (ggf. inklusive Konfiguration und Parametrierung) ohne zu warten, bis diese hochgefahren sind
     *
     * @param staggered Soll zwischen den Starts eine kurze Zeit gewartet werden (um Überlastung zu vermeiden?)
     */
    public void startAllDavWithoutWaiting(final boolean staggered) {
        for (final MultiDavStarter starter : _starterMap.values()) {
            starter.startWithoutWaiting();
            if (staggered) {
                try {
                    Thread.sleep(2000);
                } catch (InterruptedException ignored) {
                }
            }
        }
    }

    /** Beendet alle ClientDaV-Verbindungen */
    public void stopAllConnections() {
        for (final ClientDavInterface connection : _connections) {
            connection.disconnect(false, "ende");
        }
        _connections.clear();
    }

    /**
     * Startet alle zuvor erstellten Datenverteiler und wartet bis diese hochgefahren sind. Startet die Datenverteiler erneut, falls sie bereits
     * gestartet sind (ohne die Konfiguration etc. zu löschen)
     *
     * @throws Exception -
     */
    public void startAll() throws Exception {
        startAll(false);
    }

    /**
     * Startet alle zuvor erstellten Datenverteiler und wartet bis diese hochgefahren sind. Startet die Datenverteiler erneut, falls sie bereits
     * gestartet sind (ohne die Konfiguration etc. zu löschen)
     *
     * @param staggered Soll zwischen den Starts eine kurze Zeit gewartet werden (um Überlastung zu vermeiden?)
     *
     * @throws Exception -
     */
    public void startAll(final boolean staggered) throws Exception {
        final int size = _starterMap.size();
        _semaphore = new Semaphore(size);
        _semaphore.acquire(size);
        startAllDavWithoutWaiting(staggered);
        _semaphore.acquire(size);
        _semaphore = null;
    }

    /**
     * Beendet alle laufende Datenverteiler
     */
    public void stopAll() {
        for (final MultiDavStarter starter : _starterMap.values()) {
            try {
                starter.stop();
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * Beendet alle laufende Datenverteiler ohne die angelegten Dateien zu löschen
     */
    @Deprecated
    public void stopAllWithoutFileDeletion() {
        for (final MultiDavStarter starter : _starterMap.values()) {
            starter.stopWithoutFileDeletion();
        }
    }

    /**
     * Kopiert den KV von einem Datenverteiler-System zu einem anderen. Ist nach dem Erstellen aller Datenverteiler und Verbindungen auszuführen, aber
     * vor dem Aufruf von startAll() o.ä.
     *
     * @param targetDav Datenverteiler zu dem der KV hinzugefügt wird.
     * @param sourceDav Datenverteiler von dem der KV kopiert wird
     *
     * @throws Exception -
     */
    public void addConfigurationAuthority(final String targetDav, final String sourceDav) throws Exception {
        final String pid = "kb.kv." + sourceDav;
        copyConfigurationAreas(targetDav, sourceDav, pid);
    }

    /**
     * Vermerkt, dass ein Konfigurationsbereich beim Initialisieren der Konfiguration importiert und aktiviert werden soll
     *
     * @param dav Datenverteilersystem, dass dem KB importieren soll
     * @param pid Pid des Konfigurationsbereichs
     */
    public void addImportFile(String dav, String pid) {
        final MultiDavStarter starter = getExistingDav(dav);
        starter.addImportFile(pid);
    }

    /**
     * Importiert und Aktiviert einen Konfigurationsbereich nach dem Initialisieren der Konfiguration
     *
     * @param dav Datenverteilersystem, dass dem KB importieren soll
     * @param pid Pid des Konfigurationsbereichs
     */
    public void activateConfigurationArea(String dav, String pid) throws Exception {
        final MultiDavStarter starter = getExistingDav(dav);
        starter.activateAndReleaseForActivation(pid);
    }

    /**
     * Kopiert einen vorher mit {@link #addImportFile(String, String)} importierten Konfigurationsbereich zu einem anderen Datenverteiler. Damit der
     * KV des Bereichs bekannt ist, muss vermutlich auch {@link #addConfigurationAuthority(String, String)} vorher aufgerufen werden.
     *
     * @param targetDav Datenverteiler zu dem der KB hinzugefügt wird.
     * @param sourceDav Datenverteiler von dem der KB kopiert wird
     * @param areaPids  Pid des Konfigurationsbereichs
     *
     * @throws Exception -
     */
    public void copyConfigurationAreas(final String targetDav, final String sourceDav, final String... areaPids) throws Exception {
        final Path targetDir = getExistingDav(targetDav).getWorkingDirectory();
        final Path sourceDir = getExistingDav(sourceDav).getWorkingDirectory();

        for (String areaPid : areaPids) {
            Files.copy(sourceDir.resolve(areaPid + ".config"), targetDir.resolve(areaPid + ".config"), StandardCopyOption.REPLACE_EXISTING);
        }

        final MultiDavStarter starter = getExistingDav(targetDav);
        starter.addConfigurationFiles(areaPids);
    }

    /**
     * Startet den angegebenen Datenverteiler ohne zu warten, bis dieser hochgefahren ist
     *
     * @param name Datenverteiler-name
     *
     * @return Objekt, dass den Datenverteiler kapselt (Benutzung ist nicht nötig)
     */
    public MultiDavStarter startDav(final String name) {
        final MultiDavStarter starter = getExistingDav(name);
        starter.startWithoutWaiting();
        return starter;
    }

    /**
     * Startet eine Verbindung zum angegebenen Datenverteiler und gibt diese zurück. Der Datenverteiler braucht nicht komplett hochgefahren zu sein,
     * aber er muss mindestens mit startDav() gestartet worden sein.
     *
     * @param name Datenverteiler-name
     *
     * @return Verbindung
     */
    public ClientDavInterface getConnection(final String name) {
        final MultiDavStarter starter = getExistingDav(name);
        return starter.connect();
    }

    /**
     * Gibt den angegebenen Datenverteiler-Starter zurück
     *
     * @param name Datenverteiler-name
     *
     * @return MultiDavStarter
     */
    public MultiDavStarter getDav(final String name) {
        return _starterMap.get(name);
    }

    /**
     * Gibt den angegebenen Datenverteiler-Starter zurück und wirft eine Exception, wenn er nicht existiert.
     *
     * @param name Datenverteiler-name
     *
     * @return MultiDavStarter
     */
    @NotNull
    public MultiDavStarter getExistingDav(final String name) {
        final MultiDavStarter starter = getDav(name);
        if (starter == null) {
            throw new IllegalArgumentException("Es gibt keinen Datenverteiler " + name);
        }
        return starter;
    }

    /**
     * Beendet den angegebenen Datenverteiler
     *
     * @param name Datenverteiler-name
     *
     * @return Objekt, dass den Datenverteiler kapselt (Benutzung ist nicht nötig)
     */
    public MultiDavStarter stopDav(final String name) {
        final MultiDavStarter starter = getExistingDav(name);
        starter.stop();
        return starter;
    }

    /**
     * Beendet den angegebenen Datenverteiler ohne die Dateien zu löschen
     *
     * @param name Datenverteiler-name
     *
     * @return Objekt, dass den Datenverteiler kapselt (Benutzung ist nicht nötig)
     */
    @Deprecated
    public MultiDavStarter stopDavWithoutFileDeletion(final String name) {
        final MultiDavStarter starter = getExistingDav(name);
        starter.stopWithoutFileDeletion();
        return starter;
    }

    /**
     * Wartet, bis Verbindungen zu einem Datenverteiler aufgebaut werden können
     *
     * @param name Datenverteiler
     */
    public void waitUntilRunning(final String name) {
        final MultiDavStarter starter = getExistingDav(name);
        starter.waitUntilReady();
    }

    /**
     * Wartet, bis die 2 angegeben Datenverteiler miteinander verbunden sind bzw. kommunizieren
     *
     * @param dav1 Datenverteiler 1
     * @param dav2 Datenverteiler 2
     *
     * @throws Exception -
     */
    public void waitUntilConnected(final String dav1, final String dav2) throws Exception {
        waitUntilConnected(getConnection(dav1), dav2);
    }

    /**
     * Wartet, bis die bestehende Applikationsverbindung den angebenenen Datenverteiler erreicht
     *
     * @param connection Verbindung
     * @param otherDav   Anderer Datenverteiler
     *
     * @throws Exception -
     */
    public void waitUntilConnected(final ClientDavInterface connection, final String otherDav) throws Exception {
        final Receiver receiver = new Receiver();
        final DataModel dataModel = connection.getDataModel();
        synchronized (receiver) {
            connection.subscribeReceiver(receiver, dataModel.getObject("dav.test." + otherDav),
                                         new DataDescription(dataModel.getAttributeGroup("atg.telegrammLaufzeiten"),
                                                             dataModel.getAspect("asp.messwerte")), ReceiveOptions.normal(), ReceiverRole.receiver());
            long startTime = System.nanoTime();
            while (!receiver._connected) {
                receiver.wait(1000);
                if (TimeUnit.NANOSECONDS.toSeconds(System.nanoTime() - startTime) > 60) {
                    dumpThreads(System.err);
                    throw new AssertionError("Datenverteiler nicht nach 1 Minute verbunden");
                }
            }
        }
        connection.disconnect(false, "");
    }

    /**
     * Prüft, ob die 2 angegeben Datenverteiler miteinander verbunden sind bzw. kommunizieren
     *
     * @param dav1 Datenverteiler 1
     * @param dav2 Datenverteiler 2
     *
     * @throws Exception -
     */
    public boolean isConnected(final String dav1, final String dav2) throws Exception {
        return isConnected(getConnection(dav1), dav2);
    }

    /**
     * Prüft, ob die bestehende Applikationsverbindung den angebenenen Datenverteiler erreicht
     *
     * @param connection Verbindung
     * @param otherDav   Anderer Datenverteiler
     *
     * @return true wenn Verbindung steht, sonst false
     *
     * @throws Exception -
     */
    public boolean isConnected(final ClientDavInterface connection, final String otherDav) throws Exception {
        try {
            final Receiver receiver = new Receiver();
            final DataModel dataModel = connection.getDataModel();
            synchronized (receiver) {
                connection.subscribeReceiver(receiver, dataModel.getObject("dav.test." + otherDav),
                                             new DataDescription(dataModel.getAttributeGroup("atg.telegrammLaufzeiten"),
                                                                 dataModel.getAspect("asp.messwerte")), ReceiveOptions.normal(),
                                             ReceiverRole.receiver());
                long startTime = System.nanoTime();
                while (!receiver._connected) {
                    receiver.wait(1000);
                    if (TimeUnit.NANOSECONDS.toSeconds(System.nanoTime() - startTime) > 1) {
                        return false;
                    }
                }
            }
            return true;
        } finally {
            connection.disconnect(false, "");
        }
    }

    /**
     * Wartet, bis die 2 angegeben Datenverteiler nicht mehr miteinander verbunden sind.
     *
     * @param dav1 Datenverteiler 1
     * @param dav2 Datenverteiler 2
     *
     * @throws Exception -
     */
    public void waitUntilDisconnected(final String dav1, final String dav2) throws Exception {
        final ClientDavInterface clientDavInterface = getConnection(dav1);
        final WaitForNoDataReceiver receiver = new WaitForNoDataReceiver();
        final DataModel dataModel = clientDavInterface.getDataModel();
        synchronized (receiver) {
            clientDavInterface.subscribeReceiver(receiver, dataModel.getObject("dav.test." + dav2),
                                                 new DataDescription(dataModel.getAttributeGroup("atg.distributionspaketReleaseInfo"),
                                                                     dataModel.getAspect("asp.standard")), ReceiveOptions.normal(),
                                                 ReceiverRole.receiver());
            long startTime = System.nanoTime();
            while (!receiver._noDataReceived) {
                receiver.wait(1000);
                if (TimeUnit.NANOSECONDS.toSeconds(System.nanoTime() - startTime) > 60) {
                    throw new AssertionError("Datenverteiler nicht nach 1 Minute getrennt");
                }
            }
        }
        clientDavInterface.disconnect(false, "");
    }

    /**
     * Setzt die Verzögerungszeit, die innerhalb des Datenverteilers gewartet wird, bevor Verbindungen zu anderen Datenverteilern zugelassen bzw.
     * aufgebaut werden.
     *
     * @param davDavConnectDelay Verzögerungszeit in Millisekunden
     */
    public void setDavDavConnectDelay(final int davDavConnectDelay) {
        _davDavConnectDelay = davDavConnectDelay;
    }

    /**
     * Setzt die Verzögerungszeit, die innerhalb des Datenverteilers gewartet wird, bevor Verbindungen zu anderen Datenverteilern zugelassen bzw.
     * aufgebaut werden.
     *
     * @param davDavReconnectDelay Verzögerungszeit in Millisekunden
     */
    public void setDavDavReconnectDelay(final int davDavReconnectDelay) {
        _davDavReconnectDelay = davDavReconnectDelay;
    }

    private synchronized File getConfigurationFile() throws Exception {
        if (_globalConfigurationController == null) {
            _globalConfigurationController = new ConfigurationController(_testClass.getName(), workingDir.resolve("prepare.tmp").toFile().getCanonicalFile());
            _globalConfigurationController.createConfigurationFile("kb.multiDavTestKonfiguration", getConfiguration());
            _globalConfigurationController.startConfiguration("kv.testKonfiguration");
            _globalConfigurationController.importConfigurationArea("kb.multiDavTestKonfiguration");
            _globalConfigurationController.activateConfigurationArea("kb.multiDavTestKonfiguration");
            _globalConfigurationController.reloadConfiguration();
            _globalConfigurationController.releaseConfigurationAreaForActivation("kb.multiDavTestKonfiguration");
            _globalConfigurationController.closeConfiguration();
        }
        return new File(_globalConfigurationController.getConfigurationPath(), "kb.multiDavTestKonfiguration.config");
    }

    private String getRemoteString(final String remoteName) {
        return "localhost:" + getExistingDav(remoteName).getAppPort() + ":Lokale_Konfiguration";
    }

    public int nextFreePort() {
        final int appPort = _appPortCounter;
        _appPortCounter += 2;
        return appPort;
    }

    // TBD: Folgende XML-Generierung verschönern
    private String getConfiguration() {
        final StringBuilder b = new StringBuilder();
        b.append(
            "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>" + "<!DOCTYPE konfigurationsBereich PUBLIC \"-//K2S//DTD Dokument//DE\" \"K2S.dtd\">" +
            "<konfigurationsBereich pid=\"kb.multiDavTestKonfiguration\" name=\"DavTestObjekte\" verantwortlich=\"kv.testKonfiguration\">" +
            "<info>" + "<kurzinfo>test</kurzinfo>" + "</info>" + "<modell>" + "</modell>" + "<objekte>");
        for (final Map.Entry<String, MultiDavStarter> entry : _starterMap.entrySet()) {
            b.append("<konfigurationsObjekt pid=\"dav.test.").
                append(entry.getKey()).
                append("\" name=\"TestDav.").
                append(entry.getKey()).
                append(
                    "\" typ=\"typ.datenverteiler\">" + "<datensatz attributgruppe=\"atg.datenverteilerEigenschaften\" aspekt=\"asp.eigenschaften\">" +
                    "<datum name=\"adresse\" wert=\"localhost\"/>" + "<datum name=\"subAdresse\" wert=\"").append(entry.getValue().getDavPort())
                .append("\"/>" + "</datensatz>" + "<objektMenge name=\"Applikationen\" verwaltung=\"kv.");
            if (entry.getValue().getRemoteConf() == null) {
                b.append(entry.getKey());
            } else {
                b.append(entry.getValue().getRemoteConf());
            }
            b.append("\">").append("</objektMenge>").append("</konfigurationsObjekt>");
        }
        for (final DavDavConnection connection : _davConnections) {
            b.append("<konfigurationsObjekt pid=\"davcon.test.").append(connection.getName()).append(".").append(connection.getName2()).append(
                "\" typ=\"typ" + ".datenverteilerVerbindung\">" +
                "<datensatz attributgruppe=\"atg.datenverteilerTopologie\" aspekt=\"asp.eigenschaften\">" +
                "<datum name=\"datenverteilerA\" wert=\"dav.test.").append(connection.getName())
                .append("\"/>" + "<datum " + "name=\"datenverteilerB\" wert=\"dav.test.").append(connection.getName2())
                .append("\"/>" + "<datum name=\"wichtung\" wert=\"").append(connection.getWeight()).append("\"/>")
                .append("<datum name=\"aktiverDatenverteiler\" wert=\"").append(connection.getOriginator()).append("\"/>")
                .append("<datum name=\"ersatzverbindungsWartezeit\" wert=\"10 Sekunden\"/>")
                .append("<datum name=\"benutzer1\" wert=\"" + connection.getUser1() + "\"/>")
                .append("<datum name=\"benutzer2\" wert=\"" + connection.getUser2() + "\"/>").append("<datenliste name=\"durchsatzPrüfung\">")
                .append("<datum name=\"pufferFüllgrad\" wert=\"75 %\"/>").append("<datum name=\"prüfIntervall\" wert=\"3 Minuten\"/>")
                .append("<datum name=\"mindestDurchsatz\" wert=\"3000 Byte/s\"/>").append("</datenliste>").append("</datensatz>");
            if (connection.getReplacementConnections() != null) {
                b.append("<objektMenge name=\"Ersatzverbindungen\">");
                for (String s : connection.getReplacementConnections()) {
                    b.append("<element pid=\"" + s + "\"/>");
                }
                b.append("</objektMenge>");
            }
            b.append("</konfigurationsObjekt>");
        }
	    b.append("""
			    \t\t<konfigurationsObjekt typ="typ.berechtigungsklasseNeu" name="Test" pid="berechtigungsklasse.test" />
			    \t\t<konfigurationsObjekt typ="typ.berechtigungsklasseNeu" name="Test2" pid="berechtigungsklasse.test2" />
			    \t\t<konfigurationsObjekt typ="typ.berechtigungsklasseNeu" name="Test3" pid="berechtigungsklasse.test3" />
			    \t\t<konfigurationsObjekt typ="typ.berechtigungsklasseNeu" name="Test4" pid="berechtigungsklasse.test4" />
			    
			    \t\t<konfigurationsObjekt typ="typ.zugriffsRolleNeu" name="Testrolle" pid="rolle.test" />
			    \t\t<konfigurationsObjekt typ="typ.zugriffsRolleNeu" name="Testrolle2" pid="rolle.test2" />
			    \t\t<konfigurationsObjekt typ="typ.zugriffsRolleNeu" name="Testrolle3" pid="rolle.test3" />
			    \t\t<konfigurationsObjekt typ="typ.zugriffsRolleNeu" name="Testrolle4" pid="rolle.test4" />
			    
			    \t\t<konfigurationsObjekt typ="typ.zugriffsRegionNeu" name="Testregion" pid="region.test" />
			    \t\t<konfigurationsObjekt typ="typ.zugriffsRegionNeu" name="Testregion2" pid="region.test2" />
			    \t\t<konfigurationsObjekt typ="typ.zugriffsRegionNeu" name="Testregion3" pid="region.test3" />
			    \t\t<konfigurationsObjekt typ="typ.zugriffsRegionNeu" name="Testregion4" pid="region.test4" />""");
        // Objekte für alte Rechteprüfung bei Verwendung von Datenverteilerkopplung zur Ermöglichung der Interaktion von zwei Konfigurationen:
        b.append("		<konfigurationsObjekt pid=\"aktivität.davKopplung.anfrage\" name=\"Basis\" typ=\"typ.zugriffsAktivität\">" +
                 "			<info>" + "				<kurzinfo>Aktivität DavKopplung Anfragen ((gerichtete)" +
                 "					Kommunikation über dem DaV)</kurzinfo>" + "			</info>" +
                 "			<datensatz attributgruppe=\"atg.zugriffsRechte\" aspekt=\"asp.eigenschaften\">" +
                 "				<datum name=\"lesen\" wert=\"Ja\"/>" + "				<datum name=\"schreiben\" wert=\"Ja\"/>" +
                 "				<datum name=\"quelleSenke\" wert=\"Ja\"/>" + "			</datensatz>" +
                 "			<objektMenge name=\"Aspekte\">" + "				<element pid=\"asp.anfrage\"/>" +
                 "				<element pid=\"asp.information\"/>" + "			</objektMenge>" +
                 "			<objektMenge name=\"Attributgruppen\">" +
                 "				<element pid=\"atg.konfigurationsBenutzerverwaltungsAnfrageSchnittstelle\"/>" +
                 "				<element pid=\"atg.konfigurationsAnfrageSchnittstelleSchreibend\"/>" +
                 "				<element pid=\"atg.betriebsMeldung\"/>" + "				<element pid=\"atg.konfigurationsSchreibAnfrage\"/>" +
                 "				<element pid=\"atg.konfigurationsAnfrage\"/>" +
                 "				<element pid=\"atg.konfigurationsAnfrageSchnittstelleLesend\"/>" +
                 "				<element pid=\"atg.konfigurationsBereichsverwaltungsAnfrageSchnittstelle\"/>" + "			</objektMenge>" +
                 "		</konfigurationsObjekt>" +
                 "		<konfigurationsObjekt pid=\"aktivität.davKopplung.antwort\" name=\"Basis\" typ=\"typ.zugriffsAktivität\">" +
                 "			<info>" + "				<kurzinfo>Aktivität DavKopplung Antworten((gerichtete)" +
                 "					Kommunikation über dem DaV)</kurzinfo>" + "			</info>" +
                 "			<datensatz attributgruppe=\"atg.zugriffsRechte\" aspekt=\"asp.eigenschaften\">" +
                 "				<datum name=\"lesen\" wert=\"Ja\"/>" + "				<datum name=\"schreiben\" wert=\"Ja\"/>" +
                 "				<datum name=\"quelleSenke\" wert=\"Ja\"/>" + "			</datensatz>" +
                 "			<objektMenge name=\"Aspekte\">" + "				<element pid=\"asp.antwort\"/>" + "			</objektMenge>" +
                 "			<objektMenge name=\"Attributgruppen\">" +
                 "				<element pid=\"atg.konfigurationsBenutzerverwaltungsAnfrageSchnittstelle\"/>" +
                 "				<element pid=\"atg.konfigurationsAnfrageSchnittstelleSchreibend\"/>" +
                 "				<element pid=\"atg.konfigurationsAntwort\"/>" +
                 "				<element pid=\"atg.konfigurationsAnfrageSchnittstelleLesend\"/>" +
                 "				<element pid=\"atg.konfigurationsBereichsverwaltungsAnfrageSchnittstelle\"/>" +
                 "				<element pid=\"atg.konfigurationsSchreibAntwort\"/>" + "			</objektMenge>" +
                 "		</konfigurationsObjekt>" +
                 "		<konfigurationsObjekt pid=\"aktivität.davKopplung.dav\" name=\"Basis\" typ=\"typ.zugriffsAktivität\">" +
                 "			<info>" + "				<kurzinfo>Aktivität DavKopplung Datenverteiler ((gerichtete)" +
                 "					Kommunikation über dem DaV)</kurzinfo>" + "			</info>" +
                 "			<datensatz attributgruppe=\"atg.zugriffsRechte\" aspekt=\"asp.eigenschaften\">" +
                 "				<datum name=\"lesen\" wert=\"Ja\"/>" + "				<datum name=\"schreiben\" wert=\"Ja\"/>" +
                 "				<datum name=\"quelleSenke\" wert=\"Ja\"/>" + "			</datensatz>" +
                 "			<objektMenge name=\"Aspekte\">" + "				<element pid=\"asp.standard\"/>" +
                 "				<element pid=\"asp.messwerte\"/>" + "			</objektMenge>" +
                 "			<objektMenge name=\"Attributgruppen\">" + "				<element pid=\"atg.distributionspaketReleaseInfo\"/>" +
                 "				<element pid=\"atg.applikationsFertigmeldung\"/>" +
                 "				<element pid=\"atg.datenverteilerRechteprüfung\"/>" + "				<element pid=\"atg.telegrammLaufzeiten\"/>" +
                 "			</objektMenge>" + "		</konfigurationsObjekt>" +
                 "		<konfigurationsObjekt pid=\"berechtigungsklasse.davKopplung\" name=\"Basis\" typ=\"typ.berechtigungsklasse\">" +
                 "			<info>" + "				<kurzinfo>Berechtigungsklasse DavKopplung</kurzinfo>" + "			</info>" +
                 "			<defaultParameter attributgruppe=\"atg.rollenRegionenPaareParameter\" typ=\"typ.berechtigungsklasse\">" +
                 "				<datenliste name=\"Urlasser\">" + "					<datum name=\"BenutzerReferenz\" wert=\"undefiniert\"/>" +
                 "					<datum name=\"Ursache\" wert=\"\"/>" + "					<datum name=\"Veranlasser\" wert=\"\"/>" +
                 "				</datenliste>" + "				<datenfeld name=\"rollenRegionenPaare\">" +
                 "					<datenliste name=\"-\">" + "						<datum name=\"rolle\" wert=\"rolle.davKopplung\"/>" +
                 "						<datum name=\"region\" wert=\"region.root\"/>" + "					</datenliste>" +
                 "				</datenfeld>" + "			</defaultParameter>" + "		</konfigurationsObjekt>" +
                 "		<konfigurationsObjekt pid=\"rolle.davKopplung\" name=\"Basis\" typ=\"typ.zugriffsRolle\">" + "			<info>" +
                 "				<kurzinfo>Rolle DavKopplung</kurzinfo>" + "			</info>" + "			<objektMenge name=\"Aktivitäten\">" +
                 "				<element pid=\"aktivität.davKopplung.dav\"/>" + "				<element pid=\"aktivität.davKopplung.antwort\"/>" +
                 "				<element pid=\"aktivität.davKopplung.anfrage\"/>" + "			</objektMenge>" +
                 "		</konfigurationsObjekt>");
        b.append("</objekte>" + "</konfigurationsBereich>");
        _hasActivated = true;
        return b.toString();
    }

    /**
     * Erstellt für einen KV seinen leeren Konfigurationsbereich
     *
     * @param name            Datenverteiler-Name
     * @param authorityCoding Kodierung
     * @param additionalUsers Zusätzlich zu erstellende Benutzerobjekte
     *
     * @return XML-String
     */
    private String getEmptyConfigurationArea(final String name, final int authorityCoding, final Set<String> additionalUsers) {
        final StringBuilder b = new StringBuilder();
        b.append(
            "<?xml version=\"1.0\" encoding=\"ISO-8859-1\"?>" + "<!DOCTYPE konfigurationsBereich PUBLIC \"-//K2S//DTD Dokument//DE\" \"K2S.dtd\">" +
            "<konfigurationsBereich pid=\"kb.kv.").append(name).append("\" name=\"DavTestObjekte\" verantwortlich=\"kv.").append(name)
            .append("\">" + "<info>" + "<kurzinfo>test</kurzinfo>" + "</info>" + "<modell>" + "</modell>" + "<objekte>");

        b.append("\t\t<konfigurationsObjekt pid=\"kv.").append(name).append("\" name=\"TestKonfiguration.").append(name).append(
				        """
						        " typ="typ.autarkeOrganisationsEinheit">
						        \t\t\t<info>
						        \t\t\t\t<kurzinfo>Konfigurationsverantwortlicher zum Test des Systems</kurzinfo>
						        \t\t\t</info>
						        \t\t\t<datensatz attributgruppe="atg.konfigurationsVerantwortlicherEigenschaften" aspekt="asp.eigenschaften">
						        \t\t\t\t<datum name="kodierung" wert=\"""").append(authorityCoding)
		        .append("""
				        "/>
				        \t\t\t\t<datenfeld name="defaultBereich">
				        \t\t\t\t\t<datum name="0" wert="kb.kv.""").append(name)
		        .append("""
				        "/>
				        \t\t\t\t</datenfeld>
				        \t\t\t</datensatz>
				        \t\t\t<objektMenge name="Ereignisse">
				        \t\t\t</objektMenge>
				        \t\t\t<objektMenge name="SimulationsStrecken">
				        \t\t\t</objektMenge>
				        \t\t\t<objektMenge name="PuaSkripte">
				        \t\t\t</objektMenge>
				        \t\t\t<objektMenge name="Simulationen">
				        \t\t\t</objektMenge>
				        \t\t\t<objektMenge name="Meldungen">
				        \t\t\t</objektMenge>
				        \t\t\t<objektMenge name="SystemKalenderEinträge">
				        \t\t\t</objektMenge>
				        \t\t\t<objektMenge name="EreignisTypen">
				        \t\t\t</objektMenge>
				        \t\t</konfigurationsObjekt>""");

        for (String additionalUser : additionalUsers) {
            b.append("<konfigurationsObjekt typ=\"typ.benutzer\" name=\"").append(additionalUser).append("\" pid=\"benutzer.").append(additionalUser)
                .append("\" />");
        }

        b.append("</objekte>" + "</konfigurationsBereich>");
        return b.toString();
    }

    @Override
    public String toString() {
        return "MultiDavTestEnvironment{" + "_applicationClassName='" + _testClass + '\'' + ", _configurationDebugLevel='" +
               _configurationDebugLevel + '\'' + ", _transmitterDebugLevel='" + _transmitterDebugLevel + '\'' + ", _paramDebugLevel='" +
               _paramDebugLevel + '\'' + '}';
    }

    private static class Receiver implements ClientReceiverInterface {

        boolean _connected;

        public synchronized void update(final ResultData[] results) {
            if (results[results.length - 1].hasData()) {
                _connected = true;
            }
            notifyAll();
        }
    }

    private static class WaitForNoDataReceiver implements ClientReceiverInterface {

        boolean _noDataReceived;

        public synchronized void update(final ResultData[] results) {
            if (!results[results.length - 1].hasData()) {
                _noDataReceived = true;
            }
            notifyAll();
        }
    }

    private static class DavDavConnection {

        final List<String> _replacementConnections;
        private final String _name;
        private final String _name2;
        private final String _originator;
        private final int _weight;
        private final String _user1;
        private final String _user2;

        public DavDavConnection(final String name, final String name2, final String originator, final int weight,
                                @Nullable final List<String> replacementConnections, final String user1, final String user2) {
            _name = name;
            _name2 = name2;
            _originator = originator;
            _weight = weight;
            _replacementConnections = replacementConnections;
            _user1 = user1;
            _user2 = user2;
        }

        public String getName() {
            return _name;
        }

        public String getName2() {
            return _name2;
        }

        public String getOriginator() {
            return _originator;
        }

        @Override
        public String toString() {
            return "DavDavConnection{" + "_name='" + _name + '\'' + ", _name2='" + _name2 + '\'' + ", _originator='" + _originator + '\'' + '}';
        }

        public int getWeight() {
            return _weight;
        }

        @Nullable
        public List<String> getReplacementConnections() {
            return _replacementConnections;
        }

        public String getUser1() {
            return _user1;
        }

        public String getUser2() {
            return _user2;
        }
    }

    /**
     * Objekt das einen Datenverteiler im {@link MultiDavTestEnvironment} starten kann.
     */
    public final class MultiDavStarter extends SingleDavStarter {

        private final Set<String> _additionalUsers = new LinkedHashSet<>();
        private final Set<String> _importFiles = new HashSet<>();
        private final int _authorityCoding;

        private final Map<String, Long> _davIdMap = new HashMap<>();
        private Long overrideTransmitterId;

        private MultiDavStarter(final String name, final AccessControlMode accessControlType, final String[] accessControlPlugIns, final int appPort,
                                final int davPort) throws Exception {
            this(name, accessControlType, accessControlPlugIns, appPort, davPort, 0, null);
        }

        private MultiDavStarter(final String name, final AccessControlMode accessControlType, final String[] accessControlPlugIns, final int appPort,
                                final int davPort, final int passivePort, final String remoteConf) {
            super(name, remoteConf, accessControlPlugIns, accessControlType, appPort, davPort, passivePort, _testClass, workingDir.resolve(name));
            _authorityCoding = getAppPort() - 7000;
            setConfigurationDebugLevel(_configurationDebugLevel);
            setTransmitterDebugLevel(_transmitterDebugLevel);
            setParamDebugLevel(_paramDebugLevel);
        }

        /**
         * Erstellt einen Dav-Starter
         *
         * @return DavStarter
         *
         * @throws Exception Allgemeine Exception, wird nur für Testfälle gebraucht.
         */
        protected DaVStarter createDavStarter() throws Exception {
            if (getRemoteConf() == null) {
                FileCopy.copyFile(getConfigurationFile(),
                                  new File(_configurationController.getConfigurationPath(), "kb.multiDavTestKonfiguration.config"), false);
                final ManagementFile managementFile = getManagementFile();
                if (managementFile.getConfigurationAreaManagementInfo("kb.multiDavTestKonfiguration") == null) {
                    managementFile.addConfigurationAreaManagementInfo("kb.multiDavTestKonfiguration");
                    managementFile.close();

                    _configurationController
                        .createConfigurationFile("kb.kv." + getName(), getEmptyConfigurationArea(getName(), getAuthorityCoding(), _additionalUsers));
                    _configurationController.startConfiguration(getConfigurationAuthority());
                    _configurationController.importConfigurationArea("kb.kv." + getName());
                    _configurationController.activateConfigurationAreas(Arrays.asList("kb.multiDavTestKonfiguration", "kb.kv." + getName()));
                    _configurationController.reloadConfiguration();
                    List<String> pids = _configurationController.importConfigurationAreasWithVersion(_importFiles);
                    _configurationController.activateConfigurationAreas(pids);
                    _configurationController.reloadConfiguration();
                    _configurationController.releaseConfigurationAreasForActivation(pids);
                    _configurationController.releaseConfigurationAreaForActivation("kb.kv." + getName());
                    final DataModel dataModel = _configurationController.getDataModel();
                    for (final MultiDavStarter multiDavStarter : _starterMap.values()) {
                        _davIdMap.put(multiDavStarter.getName(), dataModel.getObject("dav.test." + multiDavStarter.getName()).getId());
                    }
                } else {
                    managementFile.close();
                }
            } else {
                final MultiDavStarter remoteDav = getExistingDav(getRemoteConf());
                remoteDav.waitUntilReady();
                _davIdMap.putAll(remoteDav.getDavIdMap());
            }
            long davId = getDavId();
            if(overrideTransmitterId != null) {
                davId = overrideTransmitterId;
            }
            
            final DaVStarter daVStarter = _configurationController
                .getDaVStarter(_configurationDebugLevel, _transmitterDebugLevel, _paramDebugLevel, getAppPort(), getDavPort(), davId,
                               getRemoteConf() == null ? null : getRemoteString(getRemoteConf()), getAccessControlType(), getAccessControlPlugIns());

            daVStarter.setPassivePort(getPassivePort());
            daVStarter.setName(getName());
            daVStarter.setProtocolClass(LocalDavDavConnectionServer.class);
            Collection<MultiDavStarter> values = new ArrayList<>();
            for (String s : _davIdMap.keySet()) {
                MultiDavStarter dav = getExistingDav(s);
                values.add(dav);
            }
            MultiDavProtocolParameter multiDavProtocolParameter = new MultiDavProtocolParameter(values.toArray(new DavInformation[0]), getName());
            daVStarter.setProtocolParameter(multiDavProtocolParameter);
            daVStarter.setDavDavConnectDelay(_davDavConnectDelay);
            daVStarter.setDavDavReconnectDelay(_davDavReconnectDelay);
            daVStarter.setDebugName("");
            daVStarter.setClassPath(getClassPath());
            daVStarter.setReleaseVersion(getReleaseVersion());

            return configureDaVStarter(daVStarter);
        }

        /**
         * Gibt die Id des Datenverteiler-Systemobjekts zurück
         *
         * @return Id des Datenverteiler-Systemobjekts
         */
        public long getDavId() {
            return _davIdMap.get(getName());
        }

        /**
         * Gibt die Kodierung des Konfigurationsverantwortlichen zurück
         *
         * @return Kodierung des Konfigurationsverantwortlichen
         */
        public int getAuthorityCoding() {
            return _authorityCoding;
        }

        /**
         * Gibt eine Map mit allen bekannten Datenverteiler-IDs dieser Konfiguration zurück
         *
         * @return Kodierung des Konfigurationsverantwortlichen
         */
        private Map<String, Long> getDavIdMap() {
            return Collections.unmodifiableMap(_davIdMap);
        }

        private void addConfigurationFiles(final String... pids) throws IOException, ConfigurationChangeException {
            final ManagementFile managementFile = getManagementFile();
            for (String pid : pids) {
                if (managementFile.getConfigurationAreaManagementInfo(pid) == null) {
                    managementFile.addConfigurationAreaManagementInfo(pid);
                }
            }
            managementFile.close();
            _configurationController.startConfiguration(getConfigurationAuthority());
            ConsistencyCheckResultInterface result = _configurationController.activate(pids);
            if (!result.getLocalErrors().isEmpty()) {
                throw new ConfigurationChangeException("Lokale Fehler bei der Konsistenzprüfung");
            }
            if (!result.getInterferenceErrors().isEmpty()) {
                throw new ConfigurationChangeException("Interferenzfehler bei der Konsistenzprüfung");
            }

            _configurationController.closeConfiguration();
        }

        @Override
        public ClientDavInterface connect(final String user, final ClientCredentials password, final ClientDavParameters clientDavParameters) {
            ClientDavInterface connection = super.connect(user, password, clientDavParameters);
            _connections.add(connection);
            connection.addConnectionListener(_connections::remove);
            return connection;
        }

        @Override
        public String getConfigurationAuthority() {
            return "kv." + getName();
        }

        @Override
        public void onSuccessfulStart() {
            if (_semaphore != null) {
                _semaphore.release();
            }
        }

        public void addUsers(final String... userName) {
            _additionalUsers.addAll(ImmutableList.copyOf(userName));
        }

        public void addImportFile(final String pid) {
            _importFiles.add(pid);
        }

        public void setOverrideTransmitterId(Long overrideTransmitterId) {
            this.overrideTransmitterId = overrideTransmitterId;
        }

        public Long getOverrideTransmitterId() {
            return overrideTransmitterId;
        }
    }

}
