/*
 * Copyright 2018-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.testumg.
 *
 * de.kappich.pat.testumg is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.testumg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.testumg.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.testumg.util;

import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableMap;
import de.bsvrz.dav.daf.accessControl.AccessControlMode;
import de.bsvrz.dav.daf.main.ClientDavConnection;
import de.bsvrz.dav.daf.main.ClientDavInterface;
import de.bsvrz.dav.daf.main.ClientDavParameters;
import de.bsvrz.dav.daf.main.NormalCloser;
import de.bsvrz.dav.daf.main.authentication.ClientCredentials;
import de.bsvrz.dav.daf.main.config.management.consistenycheck.ConsistencyCheckResultInterface;
import de.bsvrz.dav.dav.main.Transmitter;
import de.bsvrz.puk.config.main.managementfile.ManagementFile;
import de.bsvrz.sys.funclib.application.StandardApplication;
import de.bsvrz.sys.funclib.commandLineArgs.ArgumentList;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.operatingMessage.MessageSender;
import de.kappich.pat.testumg.util.connections.DavInformation;
import java.io.BufferedWriter;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.net.JarURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.StandardCharsets;
import java.nio.file.FileSystemNotFoundException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.StandardCopyOption;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Properties;
import java.util.function.Function;
import java.util.jar.JarFile;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import org.junit.Assert;

/**
 * Klasse, die das Testen von Applikationen und sonstigem Code innerhalb einer Datenverteilerumgebung erlaubt.
 * <p>
 * Ein Beispiel für einen einfachen Testfall ist de.kappich.pat.testumg.util.TestSample.
 *
 * @author Kappich Systemberatung
 */
public class SingleDavStarter implements DavInformation, DafApplicationEnvironment {
    protected final ConfigurationController _configurationController;
    private final int _appPort;
    private final int _davPort;
    private final int _passivePort;
    private final String _remoteConf;
    private final String _name;
    private final AccessControlMode _accessControlType;
    private final String[] _accessControlPlugIns;
    private final Object _lock = new Object();
    private int[] _activePorts = new int[0];
    private String[] _classPath;
    private ReleaseVersion _releaseVersion = ReleaseVersion.Current;
    private ImmutableMap<String, ClientCredentials> _authenticationFile;
    private ImmutableList<UserAccount> _userAccounts;
    private String _davUser = "Tester";
    private String _configUser = "configuration";
    private volatile DaVStarter _davStarter;
    private ParamAppType _paramAppType = ParamAppType.NoParamApp;
    private volatile boolean _running;

    private String _configurationDebugLevel = DaVStarter.DEFAULT_CONFIGURATION_DEBUG;
    private String _transmitterDebugLevel = DaVStarter.DEFAULT_TRANSMITTER_DEBUG;
    private String _paramDebugLevel = DaVStarter.DEFAULT_PARAM_DEBUG;

    /**
     * Erstellt eine neue Testumgebung mit einem einzelnen Datenverteiler und mit Standard-Parametern
     * <p>
     * Ein Beispiel für einen einfachen Testfall ist `de.kappich.pat.testumg.util.TestSample`.
     */
    public SingleDavStarter() {
        this(AccessControlMode.Disabled);
    }

    /**
     * Erstellt eine neue Testumgebung mit der angegebenen Rechteprüfung
     *
     * @param accessControlType    Art der Rechteprüfung
     * @param accessControlPlugIns Rechteprüfungs-Plugins (Optional, Klassennamen)
     */
    public SingleDavStarter(final AccessControlMode accessControlType, final String... accessControlPlugIns) {
        this("", null, accessControlPlugIns, accessControlType, 8083 + DaVStarter.getDavPortNumberOffset(), -1, 0);
    }

    /**
     * Erstellt eine neue Testumgebung
     *
     * @param accessControlType    Art der Rechteprüfung
     * @param accessControlPlugIns Rechteprüfungs-Plugins (Optional, Klassennamen)
     * @param appPort              Applikations-Port
     * @param davPort              Datenverteiler-Port
     * @param passivePort          Port for passiven Verbindungaufbau (oder 0 bei aktivem Verbindungsaufbau)
     * @param name                 Name der Testumgebung
     * @param remoteConf           Remote-Konfiguration (wird vom {@link de.kappich.pat.testumg.util.MultiDavTestEnvironment.MultiDavStarter} bei
     *                             Bedarf gesetzt, falls true wird keine eigene Konfiguration gestartet)
     */
    public SingleDavStarter(final String name, @Nullable final String remoteConf, final String[] accessControlPlugIns,
                            final AccessControlMode accessControlType, final int appPort, final int davPort, final int passivePort) {
        this(name, remoteConf, accessControlPlugIns, accessControlType, appPort, davPort, passivePort, getTestClass(SingleDavStarter.class),
             TempDirectoryCreator.createTemporaryDirectory().resolve(getTestClass(SingleDavStarter.class).getSimpleName()).resolve(name));
    }

    /**
     * Erstellt eine neue Testumgebung
     *
     * @param name                 Name der Testumgebung
     * @param remoteConf           Remote-Konfiguration (wird vom {@link MultiDavTestEnvironment.MultiDavStarter} bei Bedarf gesetzt, falls true wird
     *                             keine eigene Konfiguration gestartet)
     * @param accessControlPlugIns Rechteprüfungs-Plugins (Optional, Klassennamen)
     * @param accessControlType    Art der Rechteprüfung
     * @param appPort              Applikations-Port
     * @param davPort              Datenverteiler-Port
     * @param passivePort          Port for passiven Verbindungaufbau (oder 0 bei aktivem Verbindungsaufbau)
     * @param testClass            Klasse des Testfalls
     * @param workingDir           Arbeitsverzeichnis
     */
    public SingleDavStarter(final String name, @Nullable final String remoteConf, final String[] accessControlPlugIns,
                            final AccessControlMode accessControlType, final int appPort, final int davPort, final int passivePort,
                            final Class<?> testClass, final Path workingDir) {
        _name = name;
        _remoteConf = remoteConf;
        _accessControlPlugIns = accessControlPlugIns.clone();
        _accessControlType = accessControlType;
        _appPort = appPort;
        _davPort = davPort;
        _passivePort = passivePort;
        _configurationController = new ConfigurationController(testClass.getName(), workingDir.toFile().getAbsoluteFile());
    }

    @NotNull
    public static Class<?> getTestClass(final Class<?> testerClass) {
        final StackTraceElement[] stackTraceElements = Thread.currentThread().getStackTrace();
        Class<?> testClass = testerClass;
        for (int i = stackTraceElements.length - 1; i >= 0; i--) {
            StackTraceElement stackTraceElement = stackTraceElements[i];
            try {
                Class<?> elementClass = Class.forName(stackTraceElement.getClassName());
                if (elementClass.equals(testerClass)) {
                    break;
                }
                testClass = elementClass;
            } catch (ClassNotFoundException ignored) {
            }
        }
        return testClass;
    }

    /**
     * Lädt die Kernsoftware-Distributionspakete mit der angegebenen Version herunter (interne Funktion zum Testen von verschiedenen
     * Kernsoftware-Versionen)
     *
     * @param version Version
     *
     * @return InputStream der Zip-Datei
     *
     * @throws IOException
     */
    private static InputStream fetchReleaseZip(final ReleaseVersion version) throws IOException {
        Path cacheFile = Paths.get("cache", version.toString() + ".zip");
        Files.createDirectories(cacheFile.getParent());
        if (!Files.exists(cacheFile)) {
            URL url = version.getUrl();
            if (url == null) {
                throw new IllegalArgumentException("ReleaseVersion ohne URL");
            }
            System.out.println("Lade " + url);
            Files.copy(url.openStream(), cacheFile);
        } else {
            System.out.println("Verwende gecachte Version von " + version);
        }
        return Files.newInputStream(cacheFile);
    }

    /**
     * Ergänzt rekursiv die Resource-Namen aller Dateien im angegeben Verzeichnis in eine Liste.
     *
     * @param resourceNames      Liste mit Resource-Namen der im Verzeichnis oder in Unterverzeichnissen enthaltenen Dateien
     * @param directory          Zu untersuchendes Verzeichnis.
     * @param resourceNamePrefix Resource-Namen-Präfix des angegebenen Verzeichnisses (mit "/" am Ende)
     */
    private static void addResourceNames(final ArrayList<String> resourceNames, final Path directory, final String resourceNamePrefix) {
        Stream<Path> files;
        try {
            files = Files.list(directory);
        } catch (IOException e) {
            throw new IllegalArgumentException(e);
        }
        files.forEach(subFile -> {
            final String resourceName = resourceNamePrefix + subFile.getFileName();
            if (Files.isDirectory(subFile)) {
                addResourceNames(resourceNames, subFile, resourceName + "/");
            } else {
                resourceNames.add(resourceName);
            }
        });
    }

    /**
     * Gibt den Debug-Level der Konfiguration (puk.config) zurück
     *
     * @return den Debug-Level
     *
     * @see #setConfigurationDebugLevel(String)
     */
    public final String getConfigurationDebugLevel() {
        return _configurationDebugLevel;
    }

    /**
     * Setzt den Konfigurations-Debug-Level. Diese Methode muss vor dem Start der Umgebung aufgerufen werden, um wirksam zu sein.
     *
     * @param configurationDebugLevel Einer der folgenden Werte: "ERROR" "WARNING" "CONFIG" "INFO" "FINE" "FINER" "FINEST" "ALL".
     */
    public final void setConfigurationDebugLevel(final String configurationDebugLevel) {
        _configurationDebugLevel = configurationDebugLevel;
    }

    /**
     * Gibt den Debug-Level des Datenverteilers
     *
     * @return den Debug-Level
     *
     * @see #setTransmitterDebugLevel(String)
     */
    public final String getTransmitterDebugLevel() {
        return _transmitterDebugLevel;
    }

    /**
     * Setzt den Datenverteiler-Debug-Level. Diese Methode muss vor dem Start der Umgebung aufgerufen werden, um wirksam zu sein.
     *
     * @param transmitterDebugLevel Einer der folgenden Werte: "ERROR" "WARNING" "CONFIG" "INFO" "FINE" "FINER" "FINEST" "ALL".
     */
    public final void setTransmitterDebugLevel(final String transmitterDebugLevel) {
        _transmitterDebugLevel = transmitterDebugLevel;
    }

    /**
     * Gibt den Debug-Level der Parameterierung zurück
     *
     * @return den Debug-Level
     *
     * @see #setParamDebugLevel(String)
     */
    public final String getParamDebugLevel() {
        return _paramDebugLevel;
    }

    /**
     * Setzt den Parametrierungs-Debug-Level. Diese Methode muss vor dem Start der Umgebung aufgerufen werden, um wirksam zu sein.
     *
     * @param paramDebugLevel Einer der folgenden Werte: "ERROR" "WARNING" "CONFIG" "INFO" "FINE" "FINER" "FINEST" "ALL".
     */
    public final void setParamDebugLevel(final String paramDebugLevel) {
        _paramDebugLevel = paramDebugLevel;
    }

    /**
     * Gibt zurück, ob der Datenverteiler gestartet wurde, also entweder gerade startet oder schon läuft.
     *
     * @return Ob der Datenverteiler gestartet wurde
     */
    public final boolean isRunning() {
        return _running;
    }

    /**
     * Gibt den internen {@link DaVStarter} zurück und erstellt ihn falls noch nicht geschehen.
     *
     * @return DaVStarter
     */
    private DaVStarter getDavStarter() {
        if (_davStarter == null) {
            try {
                _davStarter = createDavStarter();
            } catch (Exception e) {
                throw new AssertionError(e);
            }
        }
        return _davStarter;
    }

    /**
     * Erstellt einen Dav-Starter und initialisiert ihn mit den Test-Einstellungen
     *
     * @return DavStarter
     *
     * @throws Exception Allgemeine Exception, wird nur für Testfälle gebraucht.
     */
    protected DaVStarter createDavStarter() throws Exception {
        final DaVStarter daVStarter = _configurationController
            .getDaVStarter(_configurationDebugLevel, _transmitterDebugLevel, _paramDebugLevel, _appPort, _davPort, 10001, null,
                           getAccessControlType(), getAccessControlPlugIns());

        daVStarter.setPassivePort(_passivePort);
        daVStarter.setName(getName());
        daVStarter.setDebugName("");
        daVStarter.setClassPath(getClassPath());
        daVStarter.setReleaseVersion(getReleaseVersion());
        return configureDaVStarter(daVStarter);
    }

    /**
     * Konfiguriert den DaV-Starter. Dabei werden Classpath, Passwortdatei und benutzerverwaltung.xml geschrieben.
     *
     * @param daVStarter Zu konfigurierendes Objekt
     *
     * @return Konfiguriertes Objekt (identisch zum Parameter)
     *
     * @throws IOException IO-Fehler
     */
    @NotNull
    protected final DaVStarter configureDaVStarter(final DaVStarter daVStarter) throws IOException {
        if (getReleaseVersion() != ReleaseVersion.Current) {
            prepareClassPath(daVStarter);
        }

        if (_authenticationFile != null) {
            Properties properties = new Properties();
            for (Map.Entry<String, ClientCredentials> entry : _authenticationFile.entrySet()) {
                properties.setProperty(entry.getKey(), entry.getValue().toString());
            }
            properties.store(Files.newBufferedWriter(daVStarter.getWorkingDirectory().toPath().resolve("passwd.properties")), null);
        }

        if (_userAccounts != null) {
            try (BufferedWriter bufferedWriter = Files
                .newBufferedWriter(daVStarter.getWorkingDirectory().toPath().resolve("benutzerverwaltung.xml"), StandardCharsets.ISO_8859_1)) {
                bufferedWriter.write("<?xml version=\"1.0\" encoding=\"ISO-8859-1\" standalone=\"no\"?>\n");
                bufferedWriter.write("<!DOCTYPE benutzerkonten PUBLIC \"-//K2S//DTD Authentifizierung//DE\" \"authentication.dtd\">\n");
                bufferedWriter.write("<benutzerkonten>\n");
                for (UserAccount account : _userAccounts) {
                    bufferedWriter.write(
                        "    <benutzeridentifikation admin=\"" + (account.isAdmin() ? "ja" : "nein") + "\" name=\"" + account.getName() +
                        "\" passwort=\"" + account.getPassword() + "\">\n");
                    final List<String> oneTimePasswords = account.getOneTimePasswords();
                    for (int i = 0; i < oneTimePasswords.size(); i++) {
                        final String s = oneTimePasswords.get(i);
                        bufferedWriter
                            .write("        <autorisierungspasswort passwort=\"" + s + "\" passwortindex=\"" + i + "\" gueltig=\"ja\" />\n");
                    }
                    bufferedWriter.write("    </benutzeridentifikation>\n");
                }
                bufferedWriter.write("</benutzerkonten>\n");
            }
        }

        daVStarter.setUserNameTransmitter(_davUser);
        daVStarter.setUserNameConfiguration(_configUser);
        daVStarter.setActivePorts(Arrays.stream(getActivePorts()).boxed().collect(Collectors.toList()));
        return daVStarter;
    }

    /**
     * Entpackt die Release-Zip und ergänzt den Classpath um alle enthaltenen runtime.jar-Dateien
     *
     * @param daVStarter Dav-Starter
     *
     * @throws IOException IO-Fehler
     */
    private void prepareClassPath(final DaVStarter daVStarter) throws IOException {
        Path releaseDir = Files.createDirectories(Paths.get("releases", getReleaseVersion().toString()));
        List<String> paths;
        // Release.zip entpacken, enthält für jedes Distributionspaket ein Zip
        try (ZipInputStream outerStream = new ZipInputStream(fetchReleaseZip(getReleaseVersion()))) {
            ZipEntry outerEntry;
            paths = new ArrayList<>();
            if (getClassPath() != null) {
                Collections.addAll(paths, getClassPath());
            }
            while ((outerEntry = outerStream.getNextEntry()) != null) {
                if (outerEntry.isDirectory()) {
                    continue;
                }
                ZipInputStream innerStream = new ZipInputStream(outerStream);
                ZipEntry innerEntry;
                while ((innerEntry = innerStream.getNextEntry()) != null) {
                    // Jede enthaltenene Datei auspacken
                    Path innerFilePath = releaseDir.resolve(innerEntry.getName());
                    if (innerEntry.isDirectory()) {
                        Files.createDirectories(innerFilePath);
                    } else {
                        Files.copy(innerStream, innerFilePath, StandardCopyOption.REPLACE_EXISTING);
                        if (innerFilePath.toString().endsWith("runtime.jar")) {
                            // Classpath ergänzen, falls es sich um eine runtime-jar handelt
                            paths.add(innerFilePath.toString());
                        }
                    }
                }
            }
        }
        daVStarter.setClassPath(paths.toArray(new String[0]));
    }

    /**
     * Diese Funktion erlaubt es, eine Instanz einer Dav-Applikation zu erzeugen und diese dann im laufenden Betrieb zu testen und fernzusteuern.
     *
     * @param creator Vom Anwender der Klasse festzulegender Ausdruck, der eine Instanz der Applikation erstellt. Der Parameter der Funktion sind
     *                dabei die Aufrufargumente. Die Rückgabe der Funktion ist ein beliebiges Objekt, zum Beispiel die Main-Klasse oder Hauptklasse
     *                der Applikation. Sie kann später abgefragt werden.
     * @param args    Zusätzliche applikationsspezifische Start-Argumente (Standard-Argumente wie "-benutzer" können weggelassen werden und werden
     *                automatisch erzeugt)
     * @param <T>     Beliebiger, für den Test relevanter Typ der Applikation
     *
     * @return Objekt, das die zu testende Applikation kapselt und Befehle wie Starten und Stoppen ermöglicht.
     */
    public <T> DafApplication<T> createApplication(final Function<String[], T> creator, final List<String> args) {
        return createApplication(creator, args, "Tester");
    }

    /**
     * Diese Funktion erlaubt es, eine Instanz einer Dav-Applikation zu erzeugen und diese dann im laufenden Betrieb zu testen und fernzusteuern.
     *
     * @param creator Vom Anwender der Klasse festzulegender Ausdruck, der eine Instanz der Applikation erstellt. Der Parameter der Funktion sind
     *                dabei die Aufrufargumente. Die Rückgabe der Funktion ist ein beliebiges Objekt, zum Beispiel die Main-Klasse oder Hauptklasse
     *                der Applikation. Sie kann später abgefragt werden.
     * @param args    Zusätzliche applikationsspezifische Start-Argumente (Standard-Argumente wie "-benutzer" können weggelassen werden und werden
     *                automatisch erzeugt)
     * @param user    Benutzer der Applikation (muss in passwd und Benutzerverwaltung vorhanden sein, siehe {@link #setAuthenticationFile(Map)} und
     *                {@link #setUserAccounts(UserAccount...)}).
     * @param <T>     Beliebiger, für den Test relevanter Typ der Applikation
     *
     * @return Objekt, das die zu testende Applikation kapselt und Befehle wie Starten und Stoppen ermöglicht.
     */
    public <T> DafApplication<T> createApplication(final Function<String[], T> creator, final List<String> args, final String user) {
        return new DafApplication<>(this, creator, args, user);
    }

    /**
     * Diese Funktion erlaubt es, eine Instanz einer Dav-Applikation zu erzeugen und diese dann im laufenden Betrieb zu testen und fernzusteuern.
     *
     * @param application Neue, bisher nicht initialisierte Instant einer {@link StandardApplication}.
     * @param args        Zusätzliche applikationsspezifische Start-Argumente (Standard-Argumente wie "-benutzer" können weggelassen werden und werden
     *                    automatisch erzeugt)
     * @param <T>         Implementierugn von StandardApplication
     *
     * @return Objekt, das die zu testende Applikation kapselt und Befehle wie Starten und Stoppen ermöglicht.
     */
    public <T extends StandardApplication> DafApplication<T> createStandardApplication(final T application, final List<String> args) {
        return createStandardApplication(application, args, "Tester");
    }

    /**
     * Diese Funktion erlaubt es, eine Instanz einer Dav-Applikation zu erzeugen und diese dann im laufenden Betrieb zu testen und fernzusteuern.
     *
     * @param application Neue, bisher nicht initialisierte Instant einer {@link StandardApplication}.
     * @param args        Zusätzliche applikationsspezifische Start-Argumente (Standard-Argumente wie "-benutzer" können weggelassen werden und werden
     *                    automatisch erzeugt)
     * @param <T>         Implementierugn von StandardApplication
     * @param user        Benutzer der Applikation (muss in passwd und Benutzerverwaltung vorhanden sein, siehe {@link #setAuthenticationFile(Map)}
     *                    und {@link #setUserAccounts(UserAccount...)}).
     *
     * @return Objekt, das die zu testende Applikation kapselt und Befehle wie Starten und Stoppen ermöglicht.
     */
    public <T extends StandardApplication> DafApplication<T> createStandardApplication(final T application, final List<String> args,
                                                                                       final String user) {
        return createApplication(strings -> {
            // Dies entspricht um wesentlichen dem Aufruf in StandardApplicationRunner, mit dem
            // Unterschied, dass die Debugs nicht erneut gesetzt werden und dass bei einem Fehler nicht die VM beendet wird,
            // sondern eine Exception geworfen wird.
            try {
                final ArgumentList argumentList = new ArgumentList(strings);
                // ArgumentListe wird in ClientDavParameters konvertiert
                final ClientDavParameters parameters = new ClientDavParameters(argumentList);
                parameters.setApplicationTypePid("typ.applikation");
                parameters.setApplicationName("");
                // zuerst darf die Applikation die ArgumentListe durcharbeiten
                application.parseArguments(argumentList);
                argumentList.ensureAllArgumentsUsed();
                final ClientDavInterface connection = new ClientDavConnection(parameters);
                // Fertigmeldung für Start/Stop wird explizit selbst übernommen
                connection.setCloseHandler(new NormalCloser());
                connection.enableExplicitApplicationReadyMessage();
                connection.connect();
                connection.login();
                MessageSender.getInstance().init(connection, "", "");
                application.initialize(connection);
                // Fertigmeldung wird gesendet
                connection.sendApplicationReadyMessage();
                return application;
            } catch (Exception e) {
                throw new AssertionError(e);
            }
        }, args, user);
    }

    @Override
    public String getAddress() {
        return "localhost.localdomain";
    }

    @Override
    public final int getPort() {
        return getDavPort();
    }

    /**
     * Gibt den Anwendungs-Port zurück
     *
     * @return Anwendungs-Port
     */
    @Override
    public final int getAppPort() {
        return _appPort;
    }

    /**
     * Gibt den Datenverteiler-Port zurück
     *
     * @return Datenverteiler-Port
     */
    public int getDavPort() {
        return _davPort;
    }

    /**
     * Gibt den Namen dieses Datenverteilers zurück
     *
     * @return Name
     */
    @Override
    public String getName() {
        return _name;
    }

    /**
     * Startet den Datenverteiler ohne auf Fertigstellung zu warten
     */
    public void startWithoutWaiting() {
        if (_running) {
            throw new IllegalStateException("Datenverteiler läuft bereits");
        }
        final Thread callingThread = Thread.currentThread();
        final Thread thread = new Thread(_name + "_starter") {
            @Override
            public void run() {
                try {
                    synchronized (_lock) {
                        _running = true;
                        _lock.notifyAll();
                        if (_davStarter == null) {
                            _davStarter = getDavStarter();
                        }
                        if (getClassPath() != null || getReleaseVersion() != ReleaseVersion.Current) {
                            _davStarter.startTransmitter();
                            if (getRemoteConf() == null) {
                                _davStarter.startConfiguration();
                            }
                            _davStarter.startParam(_paramAppType);
                        } else {
                            _davStarter.startTransmitterInSameProcess(getRemoteConf() == null, _paramAppType);
                        }

                        _lock.notifyAll();
                        onSuccessfulStart();
                    }
                } catch (Exception e) {
                    e.printStackTrace();
                    callingThread.interrupt();
                }
            }
        };
        thread.setDaemon(true);
        thread.start();
    }

    protected void onSuccessfulStart() {
    }

    /** Stoppt den Datenverteiler (ohne das Verzeichnis zu löschen) */
    public void stop() {
        _running = false;
        if (_davStarter != null) {
            _davStarter.stopDavWithoutSleep(false);
        }
        _davStarter = null;
    }

    /** Stoppt den Datenverteiler ohne das Verzeichnis zu löschen */
    @Deprecated
    public void stopWithoutFileDeletion() {
        stop();
    }

    /**
     * Erstellt eine Verbindung, wartet gegebenenfalls auf das Laden des Datenverteilers. Er muss aber vorher gestartet worden sein.
     *
     * @return Verbindung
     */
    @Override
    public ClientDavInterface connect() {
        return connect("Tester", ClientCredentials.ofString("geheim"));
    }

    public ClientDavInterface connect(ClientDavParameters clientDavParameters) {
        return connect("Tester", ClientCredentials.ofString("geheim"), clientDavParameters);
    }

    /**
     * Erstellt eine Verbindung, wartet gegebenenfalls auf das Laden des Datenverteilers. Er muss aber vorher gestartet worden sein.
     *
     * @param user     Benutzername zur Authentifizierung
     * @param password Passwort zur Authentifizierung
     *
     * @return Verbindung
     */
    public ClientDavInterface connect(final String user, final ClientCredentials password) {
        return connect(user, password, null);
    }

    /**
     * Erstellt eine Verbindung, wartet gegebenenfalls auf das Laden des Datenverteilers. Er muss aber vorher gestartet worden sein.
     *
     * @param user                Benutzername zur Authentifizierung
     * @param password            Passwort zur Authentifizierung
     * @param clientDavParameters
     *
     * @return Verbindung
     */
    public ClientDavInterface connect(final String user, final ClientCredentials password, @Nullable final ClientDavParameters clientDavParameters) {
        if (!isRunning()) {
            throw new IllegalStateException("Datenverteiler wurde noch nicht gestartet.");
        }
        synchronized (_lock) {
            while (_davStarter == null) {
                try {
                    _lock.wait();
                } catch (InterruptedException e) {
                    throw new IllegalStateException(e);
                }
            }
            return new CreateClientDavConnection(user, password, null, clientDavParameters, "localhost", _davStarter.getDavAppPort()).getConnection();
        }
    }

    /**
     * Gibt das Transmitter-Objekt des Datenverteilers zurück
     *
     * @return Transmitter-Objekt oder null falls nicht vorhanden
     */
    @Nullable
    public Transmitter getTransmitter() {
        if (_davStarter == null) {
            return null;
        }
        return _davStarter.getTransmitterObject();
    }

    /**
     * Gibt die Fake-Parametrierung zurück. Der Datenverteiler muss vorher erzeugt worden sein.
     *
     * @return FakeParamApp oder null falls nicht mit {@link #setParamAppType(ParamAppType)} aktiviert.
     */
    @Override
    public FakeParamApp getFakeParamApp() {
        synchronized (_lock) {
            return getDavStarter().getFakeParamApp();
        }
    }

    @Override
    public String getConfigurationAuthority() {
        return "kv.testKonfiguration";
    }

    /** Wartet bis der Datenverteiler fertig geladen ist. */
    public void waitUntilReady() {
        // Der folgende for-Loop ist etwas tricky, da gewartet werden muss, bis der starter-Thread `_running` auf true setzt,
        // gleichzeitig aber eine Fehlermeldung erfolgen soll, falls der Thread noch nicht gestartet wurde.
        for (int i = 0; i < 10; i++) {
            synchronized (_lock) {
                try {
                    _lock.wait(100);
                } catch (InterruptedException e) {
                    throw new AssertionError(e);
                }
                if (isRunning()) {
                    break;
                }
            }
        }
        if (!isRunning()) {
            throw new IllegalStateException("Datenverteiler wurde noch nicht gestartet.");
        }
        synchronized (_lock) {
            while (_davStarter == null) {
                try {
                    _lock.wait();
                } catch (InterruptedException e) {
                    throw new AssertionError(e);
                }
            }
        }
    }

    @Override
    public String toString() {
        return "SingleDavStarter{}";
    }

    @Override
    public Path getWorkingDirectory() {
        try {
            return getDavStarter().getWorkingDirectory().toPath();
        } catch (Exception e) {
            throw new AssertionError(e);
        }
    }

    @Override
    public Path getRootDir() {
        return getWorkingDirectory().getParent();
    }

    /**
     * Setzt den Typ der zu startenden Parametrierung.
     *
     * @param paramAppType Typ der Parametrierung.
     */
    @Override
    public void setParamAppType(final ParamAppType paramAppType) {
        if (paramAppType == null) {
            throw new IllegalArgumentException("paramAppType ist null");
        }
        _paramAppType = paramAppType;
    }

    /**
     * Gibt die Verwaltungsdaten zurück
     *
     * @return verwaltungsdaten
     */
    public ManagementFile getManagementFile() {
        return new ManagementFile(getAdminFile());
    }

    /**
     * Gibt die Datei der Verwaltungsdaten zurück
     *
     * @return Verwaltungsdaten-Datei
     */
    private File getAdminFile() {
        return _configurationController.getAdminFile();
    }

    /**
     * Erweitert den Classpath um die angegebenen Dateien. Bei wiederholten Aufruf dieser Methode werden die vorherigen Erweiterungen ersetzt.
     *
     * @param classPath ClassPath-Erweiterungen
     *
     * @return this
     */
    public SingleDavStarter withClassPath(final String... classPath) {
        _classPath = classPath.clone();
        return this;
    }

    /**
     * Setzt die Version von Datenverteiler, Konfiguration und ggf. Parametrierung für diesen Datenverteiler
     *
     * @param version Version
     *
     * @return this
     */
    public SingleDavStarter withReleaseVersion(ReleaseVersion version) {
        Objects.requireNonNull(version, "version == null");
        _releaseVersion = version;
        return this;
    }

    /**
     * Setzt die Art der Parametrierung an diesem Datenverteiler
     *
     * @param paramAppType Art
     *
     * @return this
     */
    public SingleDavStarter withParam(final ParamAppType paramAppType) {
        if (paramAppType == null) {
            throw new IllegalArgumentException("paramAppType ist null");
        }
        _paramAppType = paramAppType;
        return this;
    }

    /**
     * Gibt die Inhalte der Passwort-Datei zurück
     *
     * @return die Inhalte der Passwort-Datei, falls null wird die Standard-Passwortdatei benutzt
     *
     * @see #setAuthenticationFile(Map)
     */
    public ImmutableMap<String, ClientCredentials> getAuthenticationFile() {
        return _authenticationFile;
    }

    /**
     * Setzt die Inhalte der Passwort-Datei (passwd.properties)
     *
     * @param authenticationFile Inhalte. Der Key ist der Benutzername, der Value das Passwort bzw. der verschlüsselte SRP-Login-Token
     */
    public void setAuthenticationFile(final Map<String, ClientCredentials> authenticationFile) {
        _authenticationFile = ImmutableMap.copyOf(authenticationFile);
    }

    /**
     * Gibt die Inhalte der Benutzerverwaltung zurück
     *
     * @return die Inhalte der Benutzerverwaltung, falls null wird die Standard-Benutzerverwaltung benutzt
     */
    public List<UserAccount> getUserAccounts() {
        return _userAccounts;
    }

    /**
     * Setzt die Inhalte der Benutzerverwaltung.xml
     *
     * @param userAccounts Liste mit Benutzerkonten (Accounts)
     */
    public void setUserAccounts(final UserAccount... userAccounts) {
        _userAccounts = ImmutableList.copyOf(userAccounts);
    }

    /**
     * Gibt den Datenverteiler-Benutzer zurück
     *
     * @return den Datenverteiler-Benutzer
     */
    public String getDavUser() {
        return _davUser;
    }

    /**
     * Setzt den Benutzer, mit dem der Datenverteiler gestartet wird
     *
     * @param davUser Dav-Benutzer
     */
    public void setDavUser(final String davUser) {
        _davUser = davUser;
    }

    /**
     * Gibt den Konfigurationsbenutzer zurück
     *
     * @return den Konfigurationsbenutzer
     */
    public String getConfigUser() {
        return _configUser;
    }

    /**
     * Setzt den Benutzer, mit dem die Konfiguration gestartet wird
     *
     * @param configUser Konfigurations-Benutzer
     */
    public void setConfigUser(final String configUser) {
        _configUser = configUser;
    }

    /**
     * Startet den Datenverteiler, sowie Konfiguration und Parameterierung (wenn entsprechend konfiguriert).
     *
     * @param configurationAreas Optional vor dem Start zu aktivierende Konfigurationsbereiche. Die Konfigurationsbereiche können mit Versionsnummern
     *                           versehen sein, das Verhalten ist in der {@link #activate(String...) activate}-Methode beschrieben.
     */
    public void start(final String... configurationAreas) {
        getDavStarter();
        if (configurationAreas.length > 0) {
            activate(configurationAreas);
        }
        startWithoutWaiting();
        waitUntilReady();
    }

    /**
     * Importiert und aktiviert die angegebenen Konfigurationsbereiche in den angegebenen Versionen.
     * <p>
     * Die Konfigurationsbereiche müssen sich als Resource im Package des Testfalls befinden. Es kann entweder nur die Pid angegeben werden, dann wird
     * die Datei einfach importiert. Z. B. die Angabe von "kb.testObjekte" importiert den Bereich mit der Pid "kb.testObjekte" aus der Datei
     * "kb.testObjekte.xml".
     * <p>
     * Bei manchen Tests ist es sinnvoll nacheinander verschiedene Versionen eines Bereiches zu aktivieren, dafür kann zusätzlich eine Versionsangabe
     * getrennt mit einem Doppelpunkt nach der Pid gemacht werden. Beispiel:
     * <ul>
     * <li>Die Angabe von "kb.testObjekte:1" importiert den Bereich mit der id "kb.testObjekte" aus der Datei "kb.testObjekte1.xml"</li>
     * <li>Die Angabe von "kb.testObjekte:2" importiert den Bereich mit der Pid "kb.testObjekte" aus der Datei "kb.testObjekte2.xml"</li>
     * <li>usw.</li>
     * </ul>
     * <p>
     * Die Versionsangabe korreliert nicht notwendigerweise mit der resultierten Version der Konfigurationsdatei, es kann also auch
     * z. B. erst "kb.testObjekte:2" importiert und aktiviert werden und danach "kb.testObjekte:1".
     * <p>
     *     Für manche Testfälle können die Konfigurationsdateien nicht gleichzeitig importiert werden, sondern müssen nacheinander
     *     oder Blockweise importiert werden. Durch die Einfügung eines {@code null}-Elements können verschiedene Blöcke getrennt werden. Beispiel:
     *     {@code activate("kb.test1", "kb.test2", null, "kb.test3");} importiert und aktiviert zuerst gemeinsam
     *     die Bereiche "test1" und "test2", startet dann die Konfiguration neu und importiert und aktiviert "test3.
     *
     * @param configurationAreaPids die Pids der Konfigurationsbereiche, optional mit Version. Die String sollten (wenn sie eine Version enthalten)
     *                              das Format "kb.bereich:2" oder ähnlich haben, also die Version mit einem Doppelpunkt getrennt enthalten.
     *                              Null-Elemente sorgen für einen Neustart der Konfiguration nach Import und aktivierung der bisherigen Elemente.
     */
    public void activate(final String... configurationAreaPids) {
        activate(false, configurationAreaPids);
    }

    /**
     * Diese Methode tut das selbe wie {@link #activate(String...)}, mit dem Unterschied, dass die Bereiche nicht nur aktiviert, sondern auch für die
     * Aktivierung durch andere freigegeben werden.
     *
     * @param configurationAreaPids die Pids der Konfigurationsbereiche, optional mit Version. Die String sollten (wenn sie eine Version enthalten)
     *                              das Format "kb.bereich:2" oder ähnlich haben, also die Version mit einem Doppelpunkt getrennt enthalten.
     *                              Null-Elemente sorgen für einen Neustart der Konfiguration nach Import und aktivierung der bisherigen Elemente.
     */
    public void activateAndReleaseForActivation(final String... configurationAreaPids) {
        activate(true, configurationAreaPids);
    }

    /**
     * Hilfsmethode zu {@link #activate(String...)} und {@link #activateAndReleaseForActivation(String...)}
     *
     * @param release               Zur Aktivierung freigeben?
     * @param configurationAreaPids Konfigurationsbereiche
     */
    private void activate(final boolean release, final String... configurationAreaPids) {
        getDavStarter(); // Sicherstellen, dass Basis-Konfiguration initialisiert ist

        ConfigurationController configurationController = _configurationController;
        final List<String> pidsWithVersion = new ArrayList<>();
        for (int i = 0; i < configurationAreaPids.length; i++) {
            String configurationAreaPid = configurationAreaPids[i];
            if (configurationAreaPid != null) {
                pidsWithVersion.add(configurationAreaPid);
            }
            if (configurationAreaPid == null || i == configurationAreaPids.length - 1 && !pidsWithVersion.isEmpty()) {
                try {
                    System.out.println("Konfigurationsbereiche werden importiert und aktiviert: " + pidsWithVersion);
                    configurationController.startConfiguration(getConfigurationAuthority());
                    final List<String> pids = configurationController.importConfigurationAreasWithVersion(pidsWithVersion);
                    ConsistencyCheckResultInterface result = configurationController.activateConfigurationAreas(pids);
                    if (result.interferenceErrors() || result.localError()) {
                        Assert.fail("Fehler beim aktivieren der Konfigurationsbereiche:\n" + result.toString());
                    }
                    if (release) {
                        configurationController.reloadConfiguration();
                        configurationController.releaseConfigurationAreasForActivation(pids);
                    }
                } catch (Exception e) {
                    throw new AssertionError(e);
                } finally {
                    configurationController.closeConfiguration();
                }
            }
            if (configurationAreaPid == null) {
                pidsWithVersion.clear();
            }
        }
    }

    /**
     * Gibt die im Konstruktor übergebene Remote-Konfiguration zurück, mit der sich der datenverteiler verbinden soll
     *
     * @return Remote-Konfiguration
     *
     * @see MultiDavTestEnvironment
     */
    public String getRemoteConf() {
        return _remoteConf;
    }

    /**
     * Gibt die Art der Rechteprüfung zurück
     *
     * @return die Art der Rechteprüfung
     */
    public AccessControlMode getAccessControlType() {
        return _accessControlType;
    }

    /**
     * Gibt die verwendeten Rechteprüfungs-Plugins zurück
     *
     * @return die verwendeten Rechteprüfungs-Plugins
     */
    public String[] getAccessControlPlugIns() {
        return _accessControlPlugIns.clone();
    }

    /**
     * Gibt den zusätzlichen ClassPath zurück
     *
     * @return den zusätzlichen ClassPath
     *
     * @see #withClassPath(String...)
     */
    @Nullable
    public String[] getClassPath() {
        return _classPath == null ? null : _classPath.clone();
    }

    /**
     * Gibt das verwendete Release der Kernsoftware zurück
     *
     * @return Kernsoftware-Release
     */
    public ReleaseVersion getReleaseVersion() {
        return _releaseVersion;
    }

    /**
     * Kopiert Resourcedateien mit einem angegebenen Präfix in ein neues Verzeichnis. Die Resourcedateien müssen im Classpath auffindbar sein. Die
     * Methode kann sowohl mit wirklichen Dateien, als auch mit Resourcen in JAR-Dateien umgehen. Dabei geht die Methode davon aus, dass alle
     * Resource-Dateien in der gleichen JAR-Datei enthalten sind.
     *
     * @param resourceNamePrefix       Prefix der zu kopierenden Resourcen. Resourcenamen entsprechen Packagenamen, bei denen die Punkte durch Slashes
     *                                 (/) ersetzt wurden. Beispielsweise: 'de/kappich/pat/testumg/util'.
     * @param destinationDirectoryName Name des zu erzeugenden Verzeichnis, das innerhalb eines temporären Verzeichnisses (siehe @{link
     *                                 #getTemporaryDirectory}) angelegt wird.
     *
     * @return Erzeugtes Verzeichnis mit den kopierten Resource-Dateien
     */
    public Path copyResources(String resourceNamePrefix, String destinationDirectoryName) {
        if (!resourceNamePrefix.endsWith("/")) {
            resourceNamePrefix = resourceNamePrefix + "/";
        }
        try {
            final Path destinationDirectory = getWorkingDirectory().resolve(destinationDirectoryName);
            final ArrayList<String> resourceNames = new ArrayList<>();
            final URL startResource = DavTestEnvironment.class.getClassLoader().getResource(resourceNamePrefix);
            Path path = null;
            try {
                path = Paths.get(startResource.toURI());
            } catch (FileSystemNotFoundException ignored) {
            }
            if (path != null && Files.exists(path)) {
                if (Files.isDirectory(path)) {
                    addResourceNames(resourceNames, path, resourceNamePrefix);
                } else {
                    throw new IllegalArgumentException(
                        "Kopieren von Resourcen nicht möglich, da die angegebenen Resource '" + resourceNamePrefix + "' kein Verzeichnis ist.");
                }
            } else {
                URLConnection connection = startResource.openConnection();
	            if (connection instanceof JarURLConnection jarURLConnection) {
                    JarFile jarFile = jarURLConnection.getJarFile();
                    Enumeration<java.util.jar.JarEntry> e = jarFile.entries();
                    while (e.hasMoreElements()) {
                        ZipEntry entry = e.nextElement();
                        String entryname = entry.getName();
                        if (!entry.isDirectory() && entryname.startsWith(resourceNamePrefix)) {
                            resourceNames.add(entryname);
                        }
                    }
                } else {
                    throw new IllegalArgumentException("Kopieren von Resourcen nicht möglich, da '" + connection + "' keine JarURLConnection ist.");
                }
            }
            final int resourceNamePrefixLength = resourceNamePrefix.length();
            for (String resourceName : resourceNames) {
                final URL resource = DavTestEnvironment.class.getClassLoader().getResource(resourceName);
                try (InputStream sourceInput = resource.openStream()) {
                    final String localName = resourceName.substring(resourceNamePrefixLength);
                    final Path destinationFile = destinationDirectory.resolve(localName);
                    Files.createDirectories(destinationFile.getParent());
                    Files.copy(sourceInput, destinationFile, StandardCopyOption.REPLACE_EXISTING);
                }
            }
            return destinationDirectory;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    protected void finalize() throws Throwable {
        super.finalize();
        if (_running) {
            stop();
            System.err.println(this + " wurde nicht korrekt vom Test gestoppt.");
        }
    }

    /**
     * Gibt den Port zurück, dem dem der Datenverteiler auf den aktiven Applikations-Verbindungsaufbau eines anderen Datenverteilers wartet. Fall 0,
     * wird die datenverteilerinterne Applikations-Verbindung zum anderen Datenverteiler aktiv aufgebaut.
     *
     * @return Port-Nummer oder 0
     */
    public int getPassivePort() {
        return _passivePort;
    }

    /**
     * Gibt die Ports zurück, zu denen der Datenverteiler aktiv Applikationsverbindungen aufbaut
     *
     * @return Ports
     */
    public int[] getActivePorts() {
        return _activePorts.clone();
    }

    /**
     * Setzt die aktiven Ports
     *
     * @param activePorts Ports, zu denen der Datenverteiler aktiv Applikationsverbindungen aufbaut
     */
    public void setActivePorts(final int... activePorts) {
        _activePorts = activePorts.clone();
    }
}
