/*
 * Copyright 2018-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.testumg.
 *
 * de.kappich.pat.testumg is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.testumg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.testumg.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.testumg.util;

import java.io.IOException;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.List;
import java.util.concurrent.CopyOnWriteArrayList;

/**
 * Die Klasse erlaubt die Erzeugung von temporären Verzeichnissen um für Tests beliebigte Test-Dateien abzulegen (z. B. eine Beispiel-Konfiguration).
 * Die temporären Verzeichnisse werden standardmäßig im TEMP-Ordner des Betriebssystems abgelegt und beim Beenden der JVM automatisch gelöscht.
 *
 * @author Kappich Systemberatung
 */
public final class TempDirectoryCreator {

    private static final List<Path> _temporaryDirectories = new CopyOnWriteArrayList<>();
    private static final Thread _cleaner = new Thread() {
        @Override
        public void run() {
            for (Path temporaryDirectory : _temporaryDirectories) {
                try {
                    deletePath(temporaryDirectory);
                } catch (Throwable e) {
                    System.err.println("Konnte " + temporaryDirectory + " nicht löschen.");
                    e.printStackTrace();
                }
            }
        }
    };

    static {
        Runtime.getRuntime().addShutdownHook(_cleaner);
    }

    private TempDirectoryCreator() {
        throw new UnsupportedOperationException();
    }

    /**
     * Erstellt ein temporäres Verzeichnis udn gibt es als {@link Path} zurück.
     *
     * @return Path
     */
    public static Path createTemporaryDirectory() {
        Path path = createTemporaryDirectoryInternal();
        _temporaryDirectories.add(path);
        return path;
    }

    private static Path createTemporaryDirectoryInternal() {
        try {
            String basePath = System.getProperty("testbasepath", null);
            if (basePath == null) {
                return Files.createTempDirectory("test");
            } else {
                return Files.createTempDirectory(Paths.get(basePath), "test");
            }
        } catch (IOException e) {
            throw new AssertionError(e);
        }
    }

    /**
     * Löscht ein Verzeichnis inklusive enthaltener Dateien.
     *
     * @param path Verzeichnis
     *
     * @throws IOException Mögliche Exception
     */
    private static void deletePath(final Path path) throws IOException {
        if (!Files.exists(path)) {
            return;
        }
	    Files.walkFileTree(path, new SimpleFileVisitor<>() {
            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Files.delete(file);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFileFailed(Path file, IOException exc) throws IOException {
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                Files.delete(dir);
                return FileVisitResult.CONTINUE;
            }
        });
    }

}
