/*
 * Copyright 2016-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.testumg.
 *
 * de.kappich.pat.testumg is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.testumg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.testumg.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.testumg.util;

import com.google.common.collect.ImmutableList;
import de.bsvrz.dav.daf.communication.srpAuthentication.SrpVerifierData;
import java.util.Arrays;
import java.util.List;

/**
 * Ein Benutzer für die Benutzerverwaltung der Konfiguration (benutzerverwaltung.xml).
 * <p>
 * Diese Klasse wird für die Konfiguration der Benutzer im {@link SingleDavStarter} verwendet.
 *
 * @author Kappich Systemberatung
 */
public class UserAccount {
    private final String _name;
    private final String _password;
    private final boolean _isAdmin;
    private final List<String> _oneTimePasswords;

    /**
     * Erstellt einen Benutzer für die Benutzerverwaltung der Konfiguration (benutzerverwaltung.xml)
     *
     * @param name             Name des Benutzers
     * @param password         Passwort des Benutzers
     * @param isAdmin          Ist der Benutzer ein Admin?
     * @param oneTimePasswords Einmalpasswörter des Benutzers (Sollten vom Typ String sein für normale Passwörter oder vom Typ {@link SrpVerifierData}
     *                         für SRP-Überprüfungscodes.
     */
    public UserAccount(String name, String password, boolean isAdmin, Object... oneTimePasswords) {
        _name = name;
        _password = password;
        _isAdmin = isAdmin;
        _oneTimePasswords = Arrays.stream(oneTimePasswords).map(Object::toString).collect(ImmutableList.toImmutableList());
    }

    /**
     * Erstellt einen Benutzer für die Benutzerverwaltung der Konfiguration (benutzerverwaltung.xml)
     *
     * @param name             Name des Benutzers
     * @param verifier         SRP-Überprüfungscode des Benutzers
     * @param isAdmin          Ist der Benutzer ein Admin?
     * @param oneTimePasswords Einmalpasswörter des Benutzers (Sollten vom Typ String sein für normale Passwörter oder vom Typ {@link SrpVerifierData}
     *                         für SRP-Überprüfungscodes.
     */
    public UserAccount(String name, SrpVerifierData verifier, boolean isAdmin, Object... oneTimePasswords) {
        this(name, verifier.toString(), isAdmin, oneTimePasswords);
    }

    /**
     * Gibt den Benutzernamen zurück
     *
     * @return den Benutzernamen
     */
    public String getName() {
        return _name;
    }

    /**
     * Gibt das Passwort oder den SRP-Verifier zurück
     *
     * @return das Passwort oder den SRP-Verifier
     */
    public String getPassword() {
        return _password;
    }

    /**
     * Gibt {@code true} zurück, wenn der Benutzer ein Admin sein soll
     *
     * @return {@code true}, wenn der Benutzer ein Admin sein soll, sonst {@code false}
     */
    public boolean isAdmin() {
        return _isAdmin;
    }

    /**
     * Gibt die Liste mit den Einmalpasswörtern zurück
     *
     * @return die Liste mit den Einmalpasswörtern
     */
    public List<String> getOneTimePasswords() {
        return _oneTimePasswords;
    }
}
