/*
 * Copyright 2013-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.testumg.
 *
 * de.kappich.pat.testumg is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.testumg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.testumg.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.testumg.util.connections;

import java.io.IOException;
import java.io.OutputStream;
import java.util.Arrays;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.TimeUnit;

/**
 * {@link OutputStream}, der die Daten an einen weiteren {@link OutputStream} weitergibt und dabei die Übertragungsrate begrenzt.
 *
 * @author Kappich Systemberatung
 */
public class DelayOutputStream extends OutputStream {
    /**
     * Mit diesem Flag kann das Ausbremsen temporär deaktiviert werden (z. B. während des Setup-Vorgang eines Tests)
     */
    public static volatile boolean DISABLE_DELAY;
    private final OutputStream _inner;
    private final ArrayBlockingQueue<ByteArray> _queue = new ArrayBlockingQueue<>(128);
    private final int _maxPacketSize;
    private final long _transmissionDelayNanos;
    private long _minPacketDelayMillis;
    private Thread _thread;

    /**
     * Erstellt eine neue Instanz
     *
     * @param outputStream           Gekapselter {@link OutputStream}
     * @param transmissionDelayNanos Feste minimale Verzögerung zwischen Senden und Empfang in Nanosekunden (Ping)
     * @param maxFlowRate            Maximaler Durchsatz in Bytes/Sekunde
     */
    public DelayOutputStream(OutputStream outputStream, final long transmissionDelayNanos, final double maxFlowRate) {
        _inner = outputStream;
        _transmissionDelayNanos = transmissionDelayNanos;
        if (maxFlowRate >= 1000000000) {
            _minPacketDelayMillis = 0;
            _maxPacketSize = Integer.MAX_VALUE;
        } else {
            _minPacketDelayMillis = Math.max(TimeUnit.NANOSECONDS.toMillis(transmissionDelayNanos), 4);
            double packetSizeTmp = maxFlowRate * _minPacketDelayMillis / 1000.0;
            while (packetSizeTmp < 128) {
                _minPacketDelayMillis *= 2;
                packetSizeTmp = maxFlowRate * _minPacketDelayMillis / 1000.0;
            }
            _maxPacketSize = (int) packetSizeTmp;
        }
        _thread = new Thread("DelayOutputStream[" + outputStream + "]") {
            @SuppressWarnings("BusyWait")
            @Override
            public void run() {
                try {
                    while (!isInterrupted()) {
                        ByteArray byteArray = _queue.take();
                        if (!DISABLE_DELAY) {
                            long d = byteArray.getExpires() - System.nanoTime();
                            while (d > 0) {
                                Thread.sleep(TimeUnit.NANOSECONDS.toMillis(d));
                                d = byteArray.getExpires() - System.nanoTime();
                            }
                        }
                        _inner.write(byteArray.getBytes());
                        if (!DISABLE_DELAY) {
                            Thread.sleep(_minPacketDelayMillis);
                        }
                    }
                } catch (InterruptedException | IOException ignored) {
                    // Interrupts oder Schreibfehler, falls der Empfänger die Verbindung geschlossen hat, ignorieren
                } finally {
                    close();
                }
            }
        };
        _thread.start();
    }

    @Override
    public void close() {
        try {
            _inner.close();
            _thread.interrupt();
        } catch (IOException ignored) {
        }
    }

    @Override
    public void write(final int b) throws IOException {
        byte[] bytes = {(byte) b};
        enqueue(bytes);
    }

    @Override
    public void write(final byte[] b) throws IOException {
        enqueue(b.clone());
    }

    @Override
    public void write(final byte[] b, final int off, final int len) throws IOException {
        enqueue(Arrays.copyOfRange(b, off, off + len));
    }

    protected void enqueue(final byte[] bytes) throws IOException {
        if (bytes.length < _maxPacketSize) {
            try {
                _queue.put(new ByteArray(bytes));
            } catch (InterruptedException e) {
                throw new IOException(e);
            }
        } else {
            for (int s = 0; s < bytes.length; s += _maxPacketSize) {
                try {
                    _queue.put(new ByteArray(Arrays.copyOfRange(bytes, s, Math.min(s + _maxPacketSize, bytes.length))));
                } catch (InterruptedException e) {
                    throw new IOException(e);
                }
            }
        }
    }

    private class ByteArray {
        private final byte[] _bytes;
        private final long _expires;

        public ByteArray(final byte[] bytes) {
            _bytes = bytes;
            _expires = System.nanoTime() + _transmissionDelayNanos;
        }

        private byte[] getBytes() {
            return _bytes;
        }

        private long getExpires() {
            return _expires;
        }
    }
}
