/*
 * Copyright 2011-2020 by Kappich Systemberatung, Aachen
 *
 * This file is part of de.kappich.pat.testumg.
 *
 * de.kappich.pat.testumg is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.kappich.pat.testumg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.kappich.pat.testumg.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * Kappich Systemberatung
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 240
 * mail: <info@kappich.de>
 */

package de.kappich.pat.testumg.util.connections;

import de.bsvrz.dav.daf.communication.lowLevel.ConnectionInterface;
import de.bsvrz.dav.daf.communication.tcpCommunication.TCP_IP_Communication;
import de.bsvrz.dav.daf.main.ConnectionException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArraySet;

/**
 * Erweitert die {@link TCP_IP_Communication}-Verbindung um die Möglichkeit, Verbindungen (temporär) zu unterbrechen oder auszubremsen.
 *
 * @author Kappich Systemberatung
 */
public class LocalDavDavConnection implements ConnectionInterface {

    /**
     * Alle Instanzen dieser Klasse (static, damit andere Klassen die zu unterbrechenden Verbindungen anhand der Datenverteilernamen finden können)
     */
    static final Set<LocalDavDavConnection> _localDavDavConnections = new CopyOnWriteArraySet<>();
    /**
     * Aktuell unterbrochene Verbindungen
     */
    private static final Set<String> DISABLED_CONNECTIONS = new CopyOnWriteArraySet<>();
    /**
     * Aktuell ausgebremste Verbindungen
     */
    private static final Map<String, ConnectionDelayInformation> CONNECTION_DELAYS = new ConcurrentHashMap<>();
    /**
     * Interne Kommunikation
     */
    private final TCP_IP_Communication _communication;
    /**
     * Parameter (enthält Verbindungsdetails zu den anderen Datenverteilern und erlaubt eine Zuordnung von Datenverteilername zu Adresse/Port, ohne
     * Zugriff auf die Konfiguration)
     */
    private final MultiDavProtocolParameter _parameters;
    /**
     * Portnummer
     */
    private int _subAdressNumber;

    /**
     * Erstellt ein neues LocalDavDavConnection-Objekt auf Server-Seite
     *
     * @param socket     Socket (Verbindung zu Client)
     * @param parameters Parameter
     */
    public LocalDavDavConnection(final Socket socket, final String parameters) {
        _parameters = new MultiDavProtocolParameter(parameters);
        _communication = new TCP_IP_Communication(socket);
        _localDavDavConnections.add(this);
    }

    /**
     * Erstellt ein neues LocalDavDavConnection-Objekt auf Client-Seite
     *
     * @param parameters Parameter
     */
    public LocalDavDavConnection(final String parameters) {
        _parameters = new MultiDavProtocolParameter(parameters);
        _communication = new TCP_IP_Communication();
        _localDavDavConnections.add(this);
    }

    /**
     * Erstellt ein neues LocalDavDavConnection-Objekt mit Standardparametern
     */
    public LocalDavDavConnection() {
        _parameters = new MultiDavProtocolParameter("");
        _communication = new TCP_IP_Communication();
    }

    /**
     * Kappt die angegebene Verbindung
     *
     * @param davName    Datenverteiler-Name
     * @param targetPort Port
     */
    public static void disableConnection(final String davName, final int targetPort) {
        DISABLED_CONNECTIONS.add(makeAdressKey(davName, targetPort));
        for (final LocalDavDavConnection connection : _localDavDavConnections) {
            connection.refreshEnabled();
        }
    }

    /**
     * Stelle die angegebene Verbindung wieder her
     *
     * @param davName    Datenverteiler-Name
     * @param targetPort Port
     */
    public static void enableConnection(final String davName, final int targetPort) {
        DISABLED_CONNECTIONS.remove(makeAdressKey(davName, targetPort));
        for (final LocalDavDavConnection connection : _localDavDavConnections) {
            connection.refreshEnabled();
        }
    }

    /**
     * Setzt die Geschwindigkeit einer Verbindung (zum manuellen Ausbremsen)
     *
     * @param maxFlowRate       Maximale Datenrate in Bytes/Sekunde (ungefähr)
     * @param transmissionDelay Zeit in Nanosekunden, die Daten auf jeden Fall für die Strecke vom Sender zum Empfänger benötigen
     * @param fromDav           Von-Datenverteiler
     * @param toPort            Bis-Port
     */
    public static void setConnectionDelay(final String fromDav, final int toPort, final long transmissionDelay, final double maxFlowRate) {
        CONNECTION_DELAYS.put(fromDav + ":" + toPort, new ConnectionDelayInformation(maxFlowRate, transmissionDelay));
    }

    private static String makeAdressKey(final String davName, final int targetPort) {
        return davName + ":" + targetPort;
    }

    /**
     * Aktiviert alle deaktivierten Verbindungen und setzt diese Klasse zurück (am Ende eines Tests)
     */
    public static void enableAll() {
        DISABLED_CONNECTIONS.clear();
        for (final LocalDavDavConnection connection : _localDavDavConnections) {
            connection.refreshEnabled();
        }
        _localDavDavConnections.clear();
    }

    @Override
    public String getMainAdress() {
        return _communication.getMainAdress();
    }

    @Override
    public int getSubAdressNumber() {
        return _communication.getSubAdressNumber();
    }

    @Override
    public OutputStream getOutputStream() {
        ConnectionDelayInformation delayInformation = CONNECTION_DELAYS.get(_parameters.getMyName() + ":" + _communication.getSubAdressNumber());
        if (delayInformation == null) {
            return _communication.getOutputStream();
        }
        return new DelayOutputStream(_communication.getOutputStream(), delayInformation.getTransmissionDelay(), delayInformation.getMaxFlowRate());
    }

    @Override
    public String toString() {
        return _parameters.getMyName() + ">" + _parameters.getOtherName(_communication.getMainAdress(), _communication.getSubAdressNumber());
    }

    @Override
    public InputStream getInputStream() {
        return _communication.getInputStream();
    }

    @Override
    public void connect(final String mainAddress, final int subAddressNumber) throws ConnectionException {
        _subAdressNumber = subAddressNumber;

        if (DISABLED_CONNECTIONS.contains(makeAdressKey(_parameters.getMyName(), subAddressNumber))) {
            throw new ConnectionException("Verbindung derzeit deaktiviert");
        } else {
            _communication.connect(mainAddress, subAddressNumber);
        }
    }

    @Override
    public void disconnect() {
        _communication.disconnect();
    }

    @Override
    public boolean isConnected() {
        return _communication.isConnected();
    }

    @Override
    public boolean isLoopback() {
        return _communication.isLoopback();
    }

    /**
     * Kappt die Verbindung, wenn sie zum testen deaktiviert wurde
     */
    private void refreshEnabled() {
        final boolean disabled = DISABLED_CONNECTIONS.contains(makeAdressKey(_parameters.getMyName(), _subAdressNumber));
        if (disabled && _communication.isConnected()) {
            _communication.disconnect();
        }
    }

    /**
     * Informationen über die geschwindigkeit einer Verbindung. Kann manuell gesetzt werden um Verbindungen testweise auszubremsen, etc.
     */
    private static final class ConnectionDelayInformation {
        private final long _transmissionDelay;
        private final double _maxFlowRate;

        /**
         * @param maxFlowRate       Maximale Datenrate in Bytes/Sekunde (ungefähr)
         * @param transmissionDelay Zeit in Nanosekunden, die Daten für die Strecke vom Sender zum Empfänger benötigen
         */
        private ConnectionDelayInformation(final double maxFlowRate, final long transmissionDelay) {
            _maxFlowRate = maxFlowRate;
            _transmissionDelay = transmissionDelay;
        }

        /**
         * Immer vorhandene Verzögerung der Übertragung in Nanosekunden eines Pakets (Ping)
         *
         * @return Verzögerung
         */
        private long getTransmissionDelay() {
            return _transmissionDelay;
        }

        /**
         * Maximale Datenrate in Bytes/s
         *
         * @return Datenrate
         */
        private double getMaxFlowRate() {
            return _maxFlowRate;
        }
    }
}
