/*
 * Copyright 2011-2021 Connect2id Ltd. and others
 * Copyright 2021 by DTV-Verkehrsconsult, Aachen
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.sys.funclib.srp6;


import java.math.BigInteger;
import java.security.MessageDigest;


/**
 * Custom routine interface for computing the password key 'x'. Calculation of 
 * this value is required by the verifier 'v' generator as well as by the 
 * authenticating {@link SRP6ClientSession client session}.
 *
 * <p>If you don't want to employ the {@link SRP6Routines#computeX default routine} 
 * for computing 'x' you can use this interface to define your own. Remember to 
 * make sure that exactly the same routine is used to generate the verifier 'v' 
 * and by client authentication sessions later, else authentication will fail.
 *
 * <p>For a sample implementation see {@link XRoutineWithUserIdentity} which
 * computes x = H(s | H(I | ":" | P))
 *
 * @author Vladimir Dzhuvinov
 */
public interface XRoutine {


	/**
	 * Computes the password key 'x'.
	 *
	 * <p>Tip: To convert a string to a byte array you can use 
	 * {@code String.getBytes()} or 
	 * {@code String.getBytes(java.nio.charset.Charset)}. To convert a big
	 * integer to a byte array you can use {@code BigInteger.toByteArray()}.
	 *
	 * @param digest   The hash function 'H'. To enforce a particular hash
	 *                 algorithm, e.g. "SHA-1", you may perform a check 
	 *                 that throws an {@code IllegalArgumentException} or
	 *                 ignore this argument altogether.
	 * @param salt     The salt 's'. This is considered a mandatory
	 *                 argument in computation of 'x'.
	 * @param username The user identity 'I'. It may be ignored if the 
	 *                 username is allowed to change or if a user may 
	 *                 authenticate with multiple alternate identities,
	 *                 such as name and email address.
	 * @param password The user password 'P'. This is considered a
	 *                 mandatory argument in the computation of 'x'.
	 *
	 * @return The resulting 'x' value.
	 */
	public BigInteger computeX(final MessageDigest digest, 
	                           final byte[] salt, 
				   final byte[] username, 
				   final byte[] password);
}
