/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2019 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt;

import de.bsvrz.ars.ars.mgmt.simulation.SimulationResultData;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.DataDescription;
import de.bsvrz.dav.daf.main.config.*;
import de.bsvrz.sys.funclib.dataIdentificationSettings.DataIdentification;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.losb.util.K;

import java.util.*;

/**
 * Diese Klasse wertet die Parametrierung für Simulationen aus. Der Auswerte-Algorithmus ist von der Klasse {@link de.bsvrz.sys.funclib.dataIdentificationSettings.SettingsManager} der Kernsoftware
 * übernommen.
 *
 * @author beck et al. projects GmbH
 * @author Alexander Schmidt
 * @version $Revision$ / $Date$ / ($Author$)
 */
public class SimVarParamExtracter {

	/**
	 * Liest Einstellungen für Simulation
	 *
	 * @param result        Daten
	 * @param configuration Datenmodell
	 * @param newSettings   Hier werden die Einstellungen gespeichert (Ergebnis der Methode)
	 * @param simVar        Simulationsvariante
	 */
	public static void extractSettings(SimulationResultData result, DataModel configuration, Map<DataIdentification, Data> newSettings, short simVar) {
		Data data = result.getData();
		Debug.getLogger().finer("Parametrierte Daten fuer Simulationsvariante " + result.getSimVar() + " = " + data);
		if(data == null) return;
		Data parameterArray = data.getItem("ParameterSatz");
		//Schleife über die einzelnen Elemente des Arrays ParameterSatz
		for(Data parameter : parameterArray) {
			SystemObject[] specifiedConfigAreas = parameter.getReferenceArray("Bereich").getSystemObjectArray();
			specifiedConfigAreas = removeNullReferences(specifiedConfigAreas);
			Data ack = parameter.getItem(K.QUITTIEREN);
			for(Data dataSpec : parameter.getItem("Eingangsdaten")) {
				SystemObject[] specifiedObjects = dataSpec.getReferenceArray("Objekt").getSystemObjectArray();
				specifiedObjects = removeNullReferences(specifiedObjects);
				SystemObject[] specifiedAtgs = dataSpec.getReferenceArray("AttributGruppe").getSystemObjectArray();
				specifiedAtgs = removeNullReferences(specifiedAtgs);
				// Sortierung ist notwendig für das später binäre Suchen in diesem Array s.u.
				Arrays.sort(specifiedAtgs);
				SystemObject[] specifiedAspects = dataSpec.getReferenceArray("Aspekt").getSystemObjectArray();
				// Sortierung ist notwendig für das später binäre Suchen in diesem Array s.u.
				specifiedAspects = removeNullReferences(specifiedAspects);
				Arrays.sort(specifiedAspects);

				List<SystemObject> objectList;
				if(specifiedObjects.length == 0) {
					// keine Objekte sind alle Objekte
					// d.h. alle dynamischen Objekte und alle Konfigurationsobjekte

					List<SystemObject> dynamicObjects = Objects.requireNonNull(configuration.getType(Pid.Type.DYNAMIC_OBJECT), "typ.dynamischesObjekt unbekannt").getObjects();
					List<SystemObject> configurationObjects = Objects.requireNonNull(configuration.getType(Pid.Type.CONFIGURATION_OBJECT), "typ.konfigurationsObjekt unbekannt").getObjects();
					objectList = new ArrayList<>(dynamicObjects.size() + configurationObjects.size());
					objectList.addAll(dynamicObjects);
					objectList.addAll(configurationObjects);
				}
				else {
					objectList = new ArrayList<>(specifiedObjects.length);
					for(SystemObject object : specifiedObjects) {
						if (object instanceof SystemObjectCollection collection) {
							// Typen und Mengen werden durch die jeweils enthaltenen Elemente ersetzt
							objectList.addAll(collection.getElements());
						}
						else {
							objectList.add(object);
						}
					}
				}

				// Einschränkung auf ausgewählte Konfigurationsbereiche
				if(specifiedConfigAreas.length > 0) {
					List<SystemObject> specifiedConfigAreaList = Arrays.asList(specifiedConfigAreas);
					List<SystemObject> newObjectList = new LinkedList<>();
					for(SystemObject systemObject : objectList) {
						ConfigurationArea configurationArea = systemObject.getConfigurationArea();
						if(specifiedConfigAreaList.contains(configurationArea)) {
							newObjectList.add(systemObject);
						} // alle anderen werden aus der Objektliste gelöscht
					}
					objectList.clear();
					objectList.addAll(newObjectList);
				}

				// Folgende Map Enthält als Schlüssel die zu betrachtenden Typen und als Wert jeweils ein Set mit den zu
				// betrachtenden Objekten des Typs:
				Map<SystemObjectType, Collection<SystemObject>> type2objectSetMap = new TreeMap<>();

				// Erzeugen der Map, d.h. sortieren nach Typen und sicherstellen, dass jedes Objekt nur einmal enthalten ist.
				for(SystemObject object : objectList) {
					SystemObjectType type = object.getType();
					type2objectSetMap.computeIfAbsent(type, k -> new LinkedList<>()).add(object);
				}

				objectList.clear();

				
				for(Map.Entry<SystemObjectType, Collection<SystemObject>> entry : type2objectSetMap.entrySet()) {
					SystemObjectType type = entry.getKey();
					Collection<? extends SystemObject> objectCollection = entry.getValue();
					// Im folgenden werden alle Attributgruppen des Typs betrachtet und diese eventuell weiter
					// eingeschränkt, dadurch werden unsinnige Spezifikation ignoriert.
					List<AttributeGroup> typeAtgs = type.getAttributeGroups();
					for(AttributeGroup typeAtg : typeAtgs) {

						// Wenn keine Attributgruppen spezifiziert wurden, dann werden alle des Typs weiter betrachtet,
						// ansonsten werden nur die Attributgruppen des Typs weiter betrachtet, die auch explizit
						// spezifiziert wurden.
						if(specifiedAtgs.length > 0 && Arrays.binarySearch(specifiedAtgs, typeAtg) < 0) continue;

						Collection<AttributeGroupUsage> atgUsages = typeAtg.getAttributeGroupUsages();

						for(AttributeGroupUsage atgUsage : atgUsages) {

							if(atgUsage.isConfigurating()) continue;

							// Wenn kein Aspekt spezifiziert wurde, dann werden alle Aspekte der Attributgruppe weiter betrachtet,
							// ansonsten werden nur die Aspekte weiter betrachtet, die auch explizit
							// spezifiziert wurden.
							if(specifiedAspects.length > 0 && Arrays.binarySearch(specifiedAspects, atgUsage.getAspect()) < 0) continue;

							// Für alle spezifizierten Objekte des betrachteten Typs die spezifizierte Einstellung merken
							for(SystemObject object : objectCollection) {
								DataIdentification key = new DataIdentification(object, new DataDescription(typeAtg, atgUsage.getAspect(), simVar));
								newSettings.put(key, ack);
							}
						}
					}
				}
			}
		}
	}


	private static SystemObject[] removeNullReferences(SystemObject[] objects) {
		int numberOfNonNullObjects = 0;
		for(final SystemObject object1 : objects) {
			if(object1 != null) numberOfNonNullObjects++;
		}
		if(numberOfNonNullObjects != objects.length) {
			SystemObject[] trimmedObjects = new SystemObject[numberOfNonNullObjects];
			int trimmedIndex = 0;
			for(final SystemObject object : objects) {
				if(object != null) trimmedObjects[trimmedIndex++] = object;
			}
			return trimmedObjects;
		}
		else {
			return objects;
		}
	}
}
