/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.commands;

import de.bsvrz.ars.ars.mgmt.ArchiveManager;
import de.bsvrz.ars.ars.mgmt.datatree.DataIdentNode;
import de.bsvrz.ars.ars.mgmt.datatree.DataIdentTree;
import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.dav.daf.main.config.DataModel;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.losb.util.Util;
import de.bsvrz.sys.funclib.losb.util.cmdinterface.Command;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Vector;

/**
 * Kommando zum Ausgeben von Informationen einer bestimmten Datenidentifikation aus dem {@link de.bsvrz.ars.ars.mgmt.datatree.DataIdentTree}
 *
 * @author beck et al. projects GmbH
 * @author Thomas Mueller
 * @version $Revision$ / $Date$ / ($Author$)
 */
public final class DataIdentInfo extends Command {

	private static final String PAGER = " (vor: [enter], zurueck: <[enter])";

	private final ArchiveManager archMgr;

	private final DataIdentTree tree;

	private DataModel model;

	private boolean stopCmd;

	private static final int linesPerPage = 40;

	private static final int maxPerLine = 77;

	@Nullable
	private List<SystemObject> _objects;

	private int _selectedObject = -1;

	@Nullable
	private List<SystemObject> _atgs;

	private int _selectedAtg = -1;

	@Nullable
	private List<SystemObject> _asps;

	private int _selectedAsp = -1;

	@Nullable
	private int[] _simVars;

	private int _selectedSimVar = -2;

	/**
	 * simple constructor
	 *
	 * @param am Archivmanager
	 */
	public DataIdentInfo(ArchiveManager am) {
		super("Informationen zu einer Datenidentifikation", "Detailierte Informationen zu einer bestimmten Datenidentifikation abrufen.");
		archMgr = am;
		tree = archMgr.getPersistenceManager().getDataIdentTree();
		resetInput();
	}

	@Override
	public void execute() throws Exception {
		resetInput();

		println("");
		printlnPlain("    Bedienung:");
		printlnPlain("      <Zahl>           Auswahl");
		printlnPlain("      obj,atg,asp,sv   Direktauswahl (ID oder PID)");
		printlnPlain("      q                Abbrechen");
		printlnPlain("      ^                Neue Auswahl");
		printBlank(2);
		printlnPlain("Weiter mit [Enter]");

		// Datenverteiler initialisieren
		model = archMgr.getDataModel();
		if (model == null) {
			printlnPlain("Abbruch (Datenverteiler nicht bereit).");
			stopCmd = true;
		}

		// Vorselektion (übersprungen mit [Enter]) setzt obj, atg, asp, simVar
		String input = getUserInput();
		try {
			getDirectNavigation(input);
		}
		catch(Exception confEx) {
			printlnPlain("Datenidentifikation ist teilweise ungueltig,");
		}

		// Objekt auswählen
		if(!stopCmd && _selectedObject == -1) {
			_objects = sortArray(tree.getObjects(model));
			_selectedObject = getObject(_objects);
			if(_selectedObject == -1) stopCmd = true;
		}

		// Attributgruppe auswählen
		if(!stopCmd && _selectedAtg == -1) {
			_atgs = sortArray(tree.getAtgs(_objects.get(_selectedObject).getId(), model));
			_selectedAtg = getAtg(_atgs);
			if(_selectedAtg == -1) stopCmd = true;
		}

		// Aspekt auswählen
		if(!stopCmd && _selectedAsp == -1) {
			_asps = sortArray(tree.getAsps(_objects.get(_selectedObject).getId(), _atgs.get(_selectedAtg).getId(), model));
			_selectedAsp = getAsp(_asps);
			if(_selectedAsp == -1) stopCmd = true;
		}

		// Simulationsvariante auswählen
		if(!stopCmd && _selectedSimVar == -2) {
			_simVars = tree.getSimVars(_objects.get(_selectedObject).getId(), _atgs.get(_selectedAtg).getId(), _asps.get(_selectedAsp).getId());
			_selectedSimVar = getSelectedSimVar();
			if(_selectedSimVar == -2) stopCmd = true;
		}

		// Informationen zum Objekt in Klartext
		Vector<String[]> info = new Vector<>(4);
		if(!stopCmd) {
			info.add(new String[]{_objects.get(_selectedObject).getName(), _objects.get(_selectedObject).getPid(), String.valueOf(_objects.get(_selectedObject).getId())});
			info.add(new String[]{_atgs.get(_selectedAtg).getName(), _atgs.get(_selectedAtg).getPid(), String.valueOf(_atgs.get(_selectedAtg).getId())});
			info.add(new String[]{_asps.get(_selectedAsp).getName(), _asps.get(_selectedAsp).getPid(), String.valueOf(_asps.get(_selectedAsp).getId())});
			info.add(new String[]{String.valueOf(_selectedSimVar)});
		}

		// Informationen für die Datenidentifikation ausgeben
		if(!stopCmd) printInfo(_objects.get(_selectedObject).getId(), _atgs.get(_selectedAtg).getId(), _asps.get(_selectedAsp).getId(), _selectedSimVar, info);
	}

	/** Felder für Objekte, Attributgruppen, Aspekte, SimulationsVarianten zurücksetzen */
	public void resetInput() {
		stopCmd = false;
		_objects = null;
		_selectedObject = -1;
		_atgs = null;
		_selectedAtg = -1;
		_asps = null;
		_selectedAsp = -1;
		_simVars = null;
		_selectedSimVar = -2;
	}

	/**
	 * Direktnavigation mit kommaseparierten Werten
	 *
	 * @param input Benutzereingabe über Telnet, eine Kommaseparierte Liste von IDs oder Pids
	 */
	public void getDirectNavigation(@Nullable String input) {
		if(input != null) {
			String[] di = input.split(",");
			// Objekt vorselektieren
			if(!di[0].isEmpty()) {
				long objId;
				try {
					objId = Long.parseLong(di[0]);
				}
				catch(NumberFormatException nfEx) {
					objId = model.getObject(di[0]).getId();
				}
				if(objId != 0) {
					_objects = tree.getObjects(model);
					for(int i = 0; i < _objects.size(); i++) {
						if(_objects.get(i).getId() == objId) _selectedObject = i;
					}
				}
			}
			// Attributgruppe vorselektieren
			if(di.length > 1 && !di[1].isEmpty()) {
				long atgId;
				try {
					atgId = Long.parseLong(di[1]);
				}
				catch(NumberFormatException nfEx) {
					atgId = model.getObject(di[1]).getId();
				}
				if(atgId != -10) {
					_atgs = tree.getAtgs(_objects.get(_selectedObject).getId(), model);
					for(int i = 0; i < _atgs.size(); i++) {
						if(_atgs.get(i).getId() == atgId) _selectedAtg = i;
					}
				}
			}
			// Aspekt voreselektieren
			if(di.length > 2 && !di[2].isEmpty()) {
				long aspId;
				try {
					aspId = Long.parseLong(di[2]);
				}
				catch(NumberFormatException nfEx) {
					aspId = model.getObject(di[2]).getId();
				}
				if(aspId != -10) {
					_asps = tree.getAsps(_objects.get(_selectedObject).getId(), _atgs.get(_selectedAtg).getId(), model);
					for(int i = 0; i < _asps.size(); i++) {
						if(_asps.get(i).getId() == aspId) _selectedAsp = i;
					}
				}
			}
			// Simulationsvariante vorselektieren
			if(di.length > 3) {
				try {
					_selectedSimVar = Integer.parseInt(di[3]);
				}
				catch(NumberFormatException nfEx) {
					if(di[3].isEmpty()) _selectedSimVar = 0;
				}
				if(_selectedSimVar != -2) {
					_simVars = tree.getSimVars(_objects.get(_selectedObject).getId(), _atgs.get(_selectedAtg).getId(), _asps.get(_selectedAsp).getId());
					for(int i = 0; i < _simVars.length; i++) {
						if(_simVars.length == _selectedSimVar) _selectedSimVar = i;
					}
				}
			}
		}
	}


	/**
	 * Objekt aus allen möglichen des DataIdentTree auswählen
	 *
	 * @param objects Alle Objekte
	 *
	 * @return die Id des Objekts, -1 bei Abbruch durch User
	 *
	 * @throws Exception Beliebiger Fehler  
	 */
	public int getObject(List<SystemObject> objects) throws Exception {
		printlnPlain("Objekt waehlen (" + objects.size() + "):");
		int obj = -1;
		int page = 0;
		for(int i = 0; i < objects.size() && !stopCmd; i++) {
			String objOut = "  " + i + ":\t" + objects.get(i).getId() + ", " + objects.get(i).getPid() + ", " + objects.get(i).getName();
			objOut = objOut.length() > maxPerLine ? objOut.substring(0, maxPerLine - 3) + "..." : objOut;
			printlnPlain(objOut);
			if(((i + 1) % linesPerPage) == 0 || (i + 1) == objects.size()) {
				page++;
				if((1 + ((objects.size() - 1) / linesPerPage)) > 1) {
					printlnPlain("Seite " + page + " von " + (1 + ((objects.size() - 1) / linesPerPage)) + PAGER);
				}
				String input = getUserInput();
				if(input == null) {
					break;
				}
				else if(input.equals("<")) {
					page = (i > linesPerPage) ? page - 2 : 0;
					i = (i > linesPerPage) ? (i - 2 * linesPerPage) : -1;
				}
				else if(input.equals(">") || input.isEmpty()) {
					if(objects.size() == 1) obj = 0;
				}
				else if(input.equals("^")) {
					execute();
					stopCmd = true;
				}
				else {
					try {
						obj = Integer.parseInt(input);
						if(obj >= objects.size()) obj = getObject(objects);
						break;
					}
					catch(NumberFormatException nfEx) {
						/* noop */
					}
				}
			}
			if(i == objects.size() - 1 && obj == -1) {
				i = 0;
				page = 0;
			}
		}
		return obj;
	}

	/**
	 * Attributgruppe aus allen möglichen für das zuvor ausgewählte Objekt wählen
	 *
	 * @param atgs Attributgruppen
	 *
	 * @return die Id der Attributgruppe, -1 bei Abbruch durch User
	 *
	 * @throws Exception Beliebige Exception
	 */
	public int getAtg(List<SystemObject> atgs) throws Exception {
		printlnPlain("Attributgruppe waehlen (" + atgs.size() + "):");
		int atg = -1;
		int page = 0;
		for(int i = 0; i < atgs.size() && !stopCmd; i++) {
			String objOut = "  " + i + ":\t" + atgs.get(i).getId() + ", " + atgs.get(i).getPid() + ", " + atgs.get(i).getName();
			objOut = objOut.length() > maxPerLine ? objOut.substring(0, maxPerLine - 3) + "..." : objOut;
			printlnPlain(objOut);
			if(((i + 1) % linesPerPage) == 0 || (i + 1) == atgs.size()) {
				page++;
				if((1 + ((atgs.size() - 1) / linesPerPage)) > 1) printlnPlain("Seite " + page + " von " + (1 + ((atgs.size() - 1) / linesPerPage)) + PAGER);
				String input = getUserInput();
				if(input == null) {
					break;
				}
				else if(input.equals("<")) {
					page = (i > linesPerPage) ? page - 2 : 0;
					i = (i > linesPerPage) ? (i - 2 * linesPerPage) : -1;
				}
				else if(input.equals(">") || input.isEmpty()) {
					if(atgs.size() == 1) atg = 0;
				}
				else if(input.equals("^")) {
					execute();
					stopCmd = true;
				}
				else {
					try {
						atg = Integer.parseInt(input);
						if(atg >= atgs.size()) atg = getAtg(atgs);
						break;
					}
					catch(NumberFormatException nfEx) {
						/* noop */
					}
				}
			}
			if(i == atgs.size() - 1 && atg == -1) {
				i = 0;
				page = 0;
			}
		}
		return atg;
	}

	/**
	 * Aspekt für das zuvor gewählte Objekt und Attributgruppe wählen
	 *
	 * @param asps mögliche Aspekte
	 *
	 * @return die Id des gewählten Aspekts
	 *
	 * @throws Exception Beliebige Exception
	 */
	public int getAsp(List<SystemObject> asps) throws Exception {
		printlnPlain("Aspekt waehlen (" + asps.size() + "):");
		int asp = -1;
		int page = 0;
		for(int i = 0; i < asps.size() && !stopCmd; i++) {
			String objOut = "  " + i + ":\t" + asps.get(i).getId() + ", " + asps.get(i).getPid() + ", " + asps.get(i).getName();
			objOut = objOut.length() > maxPerLine ? objOut.substring(0, maxPerLine - 3) + "..." : objOut;
			printlnPlain(objOut);
			if(((i + 1) % linesPerPage) == 0 || (i + 1) == asps.size()) {
				page++;
				if((1 + ((asps.size() - 1) / linesPerPage)) > 1) printlnPlain("Seite " + page + " von " + (1 + ((asps.size() - 1) / linesPerPage)) + PAGER);
				String input = getUserInput();
				if(input == null) {
					break;
				}
				else if(input.equals("<")) {
					page = (i > linesPerPage) ? page - 2 : 0;
					i = (i > linesPerPage) ? (i - 2 * linesPerPage) : -1;
				}
				else if(input.equals(">") || input.isEmpty()) {
					if(asps.size() == 1) asp = 0;
				}
				else if(input.equals("^")) {
					execute();
					stopCmd = true;
				}
				else {
					try {
						asp = Integer.parseInt(input);
						if(asp >= asps.size()) asp = getAsp(asps);
						break;
					}
					catch(NumberFormatException nfEx) {
						/* noop */
					}
				}
			}
			if(i == asps.size() - 1 && asp == -1) {
				i = -1;
				page = 0;
			}
		}
		return asp;
	}

	/**
	 * Simulationsvariante für die ansonsten spezifizierte Datenidentifikation wählen
	 *
	 * @return die Simulationsvariante, -1 bei Abbruch durch User
	 */
	public int getSelectedSimVar() throws Exception {
		assert  _simVars != null;
		printlnPlain("Simulationsvariante waehlen (" + _simVars.length + "):");
		int simVar = -2;
		int page = 0;
		for(int i = 0; i < _simVars.length && !stopCmd; i++) {
			printlnPlain("  " + i + ":\tSimVar " + _simVars[i]);
			if(((i + 1) % linesPerPage) == 0 || (i + 1) == _simVars.length) {
				page++;
				if((1 + ((_simVars.length - 1) / linesPerPage)) > 1) {
					printlnPlain("Seite " + page + " von " + (1 + ((_simVars.length - 1) / linesPerPage)) + PAGER);
				}
				String input = getUserInput();
				if(input == null) {
					break;
				}
				else if(input.equals("<")) {
					page = (i > linesPerPage) ? page - 2 : 0;
					i = (i > linesPerPage) ? (i - 2 * linesPerPage) : -1;
				}
				else if(input.equals(">") || input.isEmpty()) {
					if(_simVars.length == 1) simVar = 0;
				}
				else if(input.equals("^")) {
					execute();
					stopCmd = true;
				}
				else {
					try {
						simVar = Integer.parseInt(input);
						if(simVar >= _simVars.length) simVar = getSelectedSimVar();
						break;
					}
					catch(NumberFormatException nfEx) {
						/* noop */
					}
				}
			}
			if(i == _simVars.length - 1 && simVar == -2) {
				i = 0;
				page = 0;
			}
		}
		return simVar;
	}

	/**
	 * gibt das ausgewählte Objekt auf StdOut aus
	 *
	 * @param objId    Objekt-ID
	 * @param atgId    Attributgruppen-ID
	 * @param aspId    Aspekt-ID
	 * @param simVar   Simulationsvariante
	 * @param info     Namen, Pids und IDs von Objekt, Attributgruppe und Aspekt
	 *
	 * @throws Exception Fehler bei der Ausgabe
	 */
	public void printInfo(long objId, long atgId, long aspId, int simVar, Vector<String[]> info) throws Exception {
		printlnPlain("Datenidentifikation:");
		try {
			// Infos zur Datenidentifikation
			int p = 20;
			printlnPlain(Util.sr("  Objekt", p) + info.get(0)[0]);
			printlnPlain(Util.sr("", p) + info.get(0)[2]);
			printlnPlain(Util.sr("", p) + info.get(0)[1]);
			printlnPlain(Util.sr("  AttGrp", p) + info.get(1)[0]);
			printlnPlain(Util.sr("", p) + info.get(1)[2]);
			printlnPlain(Util.sr("", p) + info.get(1)[1]);
			printlnPlain(Util.sr("  Aspekt", p) + info.get(2)[0]);
			printlnPlain(Util.sr("", p) + info.get(2)[2]);
			printlnPlain(Util.sr("", p) + info.get(2)[1]);
			printlnPlain(Util.sr("  SimVar", p) + info.get(3)[0]);

			// Infos zum din
			DataIdentNode din = tree.get(new IdDataIdentification(objId, atgId, aspId, simVar));
			printlnPlain(Util.sr("  Archivieren", p) + (din.arSParamIsArchivieren() ? "ja" : "nein"));
			if(din.arSParamIsNachfordern()) {
				printlnPlain(Util.sr("  Nachfordern", p) + din.arSParamGetNachfordern(0));
				for(int i = 1; i < din.arSParamGetAnzNachfordern(); i++) {
					printlnPlain(Util.sr("", p) + din.arSParamGetNachfordern(i));
				}
			}
			else {
				printlnPlain(Util.sr("  Nachfordern", p) + "nein");
			}

			if(din.arSParamIsQuittieren()) {
				SystemObject object = model.getObject(din.arSParamGetQuittieren());
				if (object == null) {
					printlnPlain(Util.sr("  Quittieren", p) + din.arSParamGetQuittieren() + " (unbekannt!)");
				} else {
					String quitAspPid = object.getPid();
					printlnPlain(Util.sr("  Quittieren", p) + din.arSParamGetQuittieren() + " (" + quitAspPid + ")");
					if (din.arSParamIsArchivieren()) {
						printlnPlain(Util.sr("    gueltig", p) + (din.arSParamIsQuittierenValid() ? "ja" : "nein"));
					}
				}
			}
			else {
				printlnPlain(Util.sr("  Quittieren", p) + "nein");
			}

			printlnPlain(Util.sr("  Parametriert", p) + (din.isArSParameterized() ? "ja" : "nein"));
			printlnPlain("");
			printlnPlain("Beenden mit [Enter], Neustart mit [^].");
			if(readln().equalsIgnoreCase("^")) execute();
		}
		catch(Exception e) {
			printlnPlain("Fehler: (" + e.getMessage() + ")");
		}
	}


	/**
	 * Sortiert eine Array von Objekten nach der Pid mit Quicksort
	 *
	 * @param in Zu sortierende Objektliste
	 *
	 * @return Sortierte Objekte
	 */
	private static ArrayList<SystemObject> sortArray(Collection<? extends SystemObject> in) {
		ArrayList<SystemObject> result = new ArrayList<>(in);
		result.sort(
				(a, b) -> a.getPid().compareToIgnoreCase(b.getPid())
		);
		return result;
	}


	/**
	 * liest die benutzereingabe oder setzt das Flag zum Abbrechen, falls "e" eingegeben wird
	 *
	 * @return the inputString
	 *
	 * @throws Exception Fehler beim Lesen der Benutzereingabe
	 */
	public String getUserInput() throws Exception {
		String input = readln();
		if(input.equals("q")) {
			stopCmd = true;
			return null;
		}
		return input;
	}


	/**
	 * prints blank lines
	 *
	 * @param count number of blank lines
	 * @throws Exception Fehler beim Lesen der Benutzereingabe
	 */
	public void printBlank(int count) throws Exception {
		for (int i = 0; i < count; i++) {
			printlnPlain("");
		}
	}

}
