/*
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */
package de.bsvrz.ars.ars.mgmt.commands;

import com.google.common.base.Splitter;
import com.google.common.collect.Range;
import com.google.common.collect.RangeSet;
import com.google.common.collect.TreeRangeSet;
import de.bsvrz.ars.ars.persistence.directories.mgmt.range.TimeDomain;
import de.bsvrz.ars.ars.persistence.directories.mgmt.range.TimeRange;
import de.bsvrz.ars.ars.persistence.directories.mgmt.range.WeekDomain;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.losb.util.cmdinterface.Command;

import java.text.ParseException;
import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.time.format.DateTimeFormatter;
import java.time.format.DateTimeParseException;
import java.time.format.ResolverStyle;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;

/**
 * Hilfsklasse, um Datumsbereiche von der Konsole zu Lesen
 */
public class DomainUtil {

	private static final DateTimeFormatter _formatter;
	private static final DateTimeFormatter _parserAlt0;
	private static final DateTimeFormatter _parserAlt1;
	private static final DateTimeFormatter _parserAlt2;

	static {
		_formatter = DateTimeFormatter.ofPattern("dd. MMMM yyyy").withLocale(Locale.GERMANY).withResolverStyle(ResolverStyle.LENIENT);
		_parserAlt0 = DateTimeFormatter.ofPattern("d.[ ]MMMM yyyy").withLocale(Locale.GERMANY).withResolverStyle(ResolverStyle.LENIENT);
		_parserAlt1 = DateTimeFormatter.ofPattern("d.[ ]M.[ ]yy").withLocale(Locale.GERMANY).withResolverStyle(ResolverStyle.LENIENT);
		_parserAlt2 = DateTimeFormatter.ofPattern("d.[ ]M.[ ]yyyy").withLocale(Locale.GERMANY).withResolverStyle(ResolverStyle.LENIENT);
	}


	/**
	 * Liest einen Text und wandelt ihn in eine Mange von Zeitbereichen um. Der Text hat ein Format wie
	 * "10. Januar 1990 bis 12. Dezember 2000; 15. Dezember 2000"
	 *
	 * @param text Text
	 * @param domain Zeitbereichs-Klasse
	 * @return Menge von Datumsbereichen
	 * @throws ParseException Fehler beim Parsen
	 * @param <T> Zeitbereichs-Typ
	 */
	public static <T extends TimeRange<T>> TreeRangeSet<LocalDate> stringToValue(final String text, TimeDomain<T> domain) throws ParseException {
		TreeRangeSet<LocalDate> set = TreeRangeSet.create();
		try {
			Iterable<String> split = Splitter.on(";").trimResults().omitEmptyStrings().split(text);

			for (String pair : split) {
				List<String> foo = Splitter.on("bis").limit(2).trimResults().splitToList(pair);
				if (foo.size() == 2) {
					set.add(
							parseRange(foo.get(0), domain).span(parseRange(foo.get(1), domain))
					);
				} else if (foo.size() == 1) {
					set.add(
							parseRange(foo.get(0), domain)
					);
				}
			}
		} catch (DateTimeParseException e) {
			throw new ParseException(e.getMessage(), e.getErrorIndex());
		} catch (IllegalArgumentException e) {
			throw new ParseException(e.getMessage(), 0);
		}
		return set;
	}

	@NotNull
	private static <T extends TimeRange<T>> Range<LocalDate> parseRange(String s, TimeDomain<T> domain) {
		if (s.matches("\\d{4}")) {
			// Ganzes Jahr auswerfen
			int year = Integer.parseInt(s);
			LocalDate startDate = LocalDate.ofYearDay(year, 1);
			LocalDate endDate = startDate.plusYears(1);
			if (domain instanceof WeekDomain) {
				startDate = getNextSunday(startDate);
			}
			return Range.closedOpen(startDate, endDate);
		}
		return Range.closedOpen(
				parse(s),
				parse(s).plusDays(1)
		);
	}

	/**
	 * Gibt den nächsten Sonntag zurück
	 *
	 * @param date Datum
	 * @return den Sonntag nach dem angegebenen Tag. Wenn der Tag bereits ein Sonntag ist, wird er selbst zurückgegeben.
	 */
	public static LocalDate getNextSunday(LocalDate date) {
		int daysUntilNextSunday = DayOfWeek.SUNDAY.getValue() - date.getDayOfWeek().getValue();

		return date.plusDays(daysUntilNextSunday);
	}

	/**
	 * Fragt den Benutzer nach Zeitbereichen und gibt diese zurück
	 *
	 * @param command Telnet-befehl für IO
	 * @param domain  Zeitbereichs-Typ (bzw. Factory-Interface)
	 * @param <T>     Zeitbereichstyp (z. B. {@link de.bsvrz.ars.ars.persistence.directories.mgmt.range.Week})
	 * @return Menge von Zeitbereichen (also z. B. Wochenverzeichnissen)
	 * @throws Exception Fehler beim Parsen
	 */
	@Nullable
	public static <T extends TimeRange<T>> Set<T> getTimeRangesFromUser(Command command, TimeDomain<T> domain) throws Exception {
		command.printlnPlain("  z. B. \"2012\" oder \"01.01.2020 bis 16.05.2021\"");
		TreeRangeSet<LocalDate> rangeSet = null;
		while (rangeSet == null) {
			String range = command.readln();
			if (range.isBlank()) {
				return null;
			}
			try {
				rangeSet = stringToValue(range, domain);
			} catch (Exception e) {
				command.printlnPlain("Ungültige Eingabe: \"" + range + "\"");
			}
		}
		command.printlnPlain("  Angegebener Zeitbereich:");
		command.printlnPlain("  " + format(rangeSet));
		command.printlnPlain("  Zugehörige Wochenverzeichnisse:");
		final Set<T> timeRanges = new LinkedHashSet<>();
		for (Range<LocalDate> range : rangeSet.asRanges()) {
			var adjustedRange = Range.range(
					range.lowerEndpoint().atStartOfDay().atOffset(ZoneOffset.UTC).toInstant(),
					range.lowerBoundType(),
					range.upperEndpoint().atStartOfDay().atOffset(ZoneOffset.UTC).toInstant(),
					range.upperBoundType()
			);
			timeRanges.addAll(domain.getIntervals(adjustedRange));
		}
		for (T timeRange : timeRanges) {
			command.printlnPlain("  " + domain.getPath(timeRange));
		}
		command.printlnPlain("  Fortfahren? (j/N)");
		String readLn = command.readln();
		if (readLn.equalsIgnoreCase("j")) {
			return timeRanges;
		}
		command.printlnPlain("  Abgebrochen.");
		return null;
	}

	private static LocalDate parse(final String s) {
		try {
			return LocalDate.parse(s, _parserAlt0);
		} catch (DateTimeParseException ignored) {
		}
		try {
			return LocalDate.parse(s, _parserAlt1);
		} catch (DateTimeParseException ignored2) {
		}
		return LocalDate.parse(s, _parserAlt2);
	}

	/**
	 * Formatiert umgekehrt eine Menge von Zeitbereichen in einen String
	 *
	 * @param value Zeitbereiche
	 * @return String
	 */
	@NotNull
	public static String format(final RangeSet<? extends LocalDate> value) {
		Set<? extends Range<? extends LocalDate>> ranges = value.asRanges();
		StringBuilder builder = new StringBuilder();
		for (var iterator = ranges.iterator(); iterator.hasNext(); ) {
			final Range<? extends LocalDate> range = iterator.next();
			if (range.lowerEndpoint().equals(range.upperEndpoint().minusDays(1))) {
				builder.append(range.lowerEndpoint().format(_formatter));
			} else {
				builder.append(range.lowerEndpoint().format(_formatter)).append(" bis ").append(range.upperEndpoint().minusDays(1).format(_formatter));
			}
			if (iterator.hasNext()) {
				builder.append("; ");
			}
		}
		return builder.toString();
	}

}
