/*
 *
 * Copyright 2005-2008 by beck et al. projects GmbH, Munich
 * Copyright 2009-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.datatree;

import de.bsvrz.ars.ars.persistence.IdDataIdentification;
import de.bsvrz.dav.daf.main.Data;
import de.bsvrz.dav.daf.main.archive.ArchiveDataKind;
import de.bsvrz.dav.daf.main.config.SystemObject;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;
import de.bsvrz.sys.funclib.losb.util.K;
import de.bsvrz.sys.funclib.losb.util.Util;

/**
 * Repraesentiert einen Knoten für eine Daten-Identifikation im Baum DataIdentTree.
 *
 * @author beck et al. projects GmbH
 * @author Alexander Schmidt
 * @author Kappich Systemberatung
 * @version $Revision$ / $Date$ / ($Author$)
 */
@SuppressWarnings({"PointlessBitwiseExpression"})
public final class DataIdentNode {

	private static final byte INVALID = -1;

	/** Zeigt an, ob der Quittierungsaspekt ungueltig ist */
	private static final int FLAG_INVALID_ACK_ID = 1 << 0;

	/**
	 * Flag, ob nach einer Anmeldung bereits ein OA-Datum erhalten wurde (wenn nicht, darf ein erhaltener Datenindex ausnahmsweise gleich dem zuletzt archivierten Datenindex sein)
	 */
	private static final int FLAG_FIRST_OA_DATA_AFTER_SUBSCRIPTION = 1 << 1;

	/**
	 * Flag, ob nach einer Anmeldung bereits ein OA-Datum erhalten wurde (wenn nicht, darf ein erhaltener Datenindex ausnahmsweise gleich dem zuletzt archivierten Datenindex sein)
	 */
	private static final int FLAG_FIRST_ON_DATA_AFTER_SUBSCRIPTION = 1 << 2;

	/**
	 * Hier wird vermerkt, ob der letzte während der laufenden Uptime archivierte OA-Datensatz die Kennung "keine Quelle" hatte. Zu Beginn ist der Status nicht
	 * gesetzt, was bedeutet, dass man im Container nachsehen muss (aufwendig).
	 */
	private static final int FLAG_LAST_DATA_NO_SOURCE_ASSIGNED = 1 << 7;
	private static final int FLAG_LAST_DATA_NO_SOURCE = 1 << 8;

	/**
	 * Flag, ob gerade eine gültige Quelle für Onlinedaten besteht
	 */
	private static final int FLAG_HAS_VALID_DATA = 1 << 9;

	//
	// ************ Interne Informationen zu Datenidentifikation:
	//

	/** Bitfeld zu den einzelnen Flags je Datenidentifikation */
	private int _flags;

	private final IdDataIdentification _dataIdentification;

	private long _unsubscriptionTime = -1;

	/** Parametrierung dieser Datenidentifikation (falls parametriert) */
	@Nullable
	private Data _arsParams;
	
	/** 
	 * Erstellt einen neuen DataIdentNode
	 * @param dataIdentification Datenidentifikation
	 */
	public DataIdentNode(final IdDataIdentification dataIdentification) {
		_dataIdentification = dataIdentification;
	}

	// ++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

	/**
	 * @return Wahr, falls der keine-Quelle-Status des letzten während der laufenden Uptime archivierten OA-Datensatzes gesetzt wurde. Falls er nicht gesetzt
	 * wurde, muss im Container nachgesehen werden (aufwendig)
	 */
	public boolean lastDataNoSourceAssigned() {
		return getFlag(FLAG_LAST_DATA_NO_SOURCE_ASSIGNED);
	}

	/**
	 * @return Wahr, falls der letzte während der laufenden Uptime archivierte online-aktuelle Datensatz die Kennung "keine Quelle" hatte. Dies wird hier
	 * vermerkt, damit nicht jedesmal im Container nachgesehen werden muss.
	 */
	public boolean lastDataWasNoSource() {
		return getFlag(FLAG_LAST_DATA_NO_SOURCE);
	}

	/**
	 * Setzt die Markierung, ob der letzte während der laufenden Uptime archivierte OA-Datensatz die Kennung "keine Quelle" hatte. (In diesem Fall muss kein
	 * Datenlücken-Datensatz erzeugt werden.)
	 *
	 * @param lastOADataWasNoSource War der letzte Datensatz ein "Keine Quelle"-Datensatz?
	 */
	public void setLastOAWasNoSource(boolean lastOADataWasNoSource) {
		setFlag(FLAG_LAST_DATA_NO_SOURCE_ASSIGNED, true);
		setFlag(FLAG_LAST_DATA_NO_SOURCE, lastOADataWasNoSource);
	}

	/**
	 * Setzt die Markierung, ob seit der letzten Anmeldung bereits ein Datum der genannten Datensatzart erhalten wurde ({@link #isFirstDataAfterSubscription(ArchiveDataKind)}
	 *
	 * @param adk  Datensatzart (OA oder ON)
	 * @param value Soll die Markierung gesetzt oder entfernt werden?
	 */
	public void setFirstDataAfterSubscription(ArchiveDataKind adk, boolean value) {
		if(adk == ArchiveDataKind.ONLINE) {
			setFlag(FLAG_FIRST_OA_DATA_AFTER_SUBSCRIPTION, value);
		}
		else if(adk == ArchiveDataKind.ONLINE_DELAYED) {
			setFlag(FLAG_FIRST_ON_DATA_AFTER_SUBSCRIPTION, value);
		}
	}

	private void setFlag(final int flag, final boolean value) {
		if(value) {
			_flags |= flag;
		}
		else {
			_flags &= ~flag;
		}
	}

	private boolean getFlag(final int flag) {
		return (_flags & flag) != 0;
	}

	/**
	 * Sagt, ob nach einer Anmeldung bereits ein Datum der genannten Datensatzart erhalten wurde. Wenn dies nicht der Fall ist, darf der Datenindex gleich dem
	 * letzten archvierten Datenindex sein.
	 *
	 * @param adk Datensatzart (OA oder ON)
	 * @return Wahr oder falsch
	 */
	public boolean isFirstDataAfterSubscription(ArchiveDataKind adk) {
		if(adk == ArchiveDataKind.ONLINE) {
			return getFlag(FLAG_FIRST_OA_DATA_AFTER_SUBSCRIPTION);
		}
		else if(adk == ArchiveDataKind.ONLINE_DELAYED) {
			return getFlag(FLAG_FIRST_ON_DATA_AFTER_SUBSCRIPTION);
		}
		else {
			return false;
		}
	}

	/**
	 * Setzt einen Zeiger auf den Datensatz mit der Parametrierung für diese Datenidentifikation
	 *
	 * @param params Parameter-Data-Objekt
	 */
	public void arsParamSetParams(@Nullable Data params) {
		_arsParams = params;
	}

	/**
	 * Liefert Kennzeichen, ob die Daten zu archivieren sind.
	 *
	 * @return Kennzeichen
	 */
	public boolean arSParamIsArchivieren() {
		if(_arsParams == null) {
			return false;
		}
		else {
			try {
				return _arsParams.getUnscaledValue(K.ARCHIVIEREN).intValue() > 0;
			}
			catch(Exception e) {
				return true;    // Das Data-Objekt stammt von einer Simulation (die immer archiviert wird).
			}
		}
	}

	/**
	 * Liefert Kennzeichen, ob bei potentiell erkannten Datenlücken bei anderen Archivsystemen nachgefordert werden soll.
	 *
	 * @return Kennzeichen
	 */
	public boolean arSParamIsNachfordern() {
		if(_arsParams == null) {
			return false;
		}
		else {
			try {
				return _arsParams.getArray(K.NACHFORDERN).getLength() > 0;
			}
			catch(Exception e) {
				return false;    // Das Data-Objekt stammt von einer Simulation (die niemals nachfordern).
			}
		}
	}

	/**
	 * Liefert die Anzahl der Archivsysteme, bei denen nachgefordert werden soll.
	 *
	 * @return Anzahl Archivsysteme
	 */
	public int arSParamGetAnzNachfordern() {
		if(_arsParams == null) {
			return 0;
		}
		else {
			try {
				return _arsParams.getArray(K.NACHFORDERN).getLength();
			}
			catch(Exception e) {
				return 0;    // Das Data-Objekt stammt von einer Simulation (die niemals nachfordern).
			}
		}
	}

	/**
	 * Liefert aus der Liste der Archivsysteme, bei denen bei potentiell erkannten Datenlücken nachgefordert werden soll, das Archivsystem mit dem angegebenen
	 * nullbasierten Index.
	 *
	 * @param at Index
	 * @return Archivsystem
	 */
	@Nullable
	public String arSParamGetNachfordern(int at) {
		if(arSParamIsNachfordern()) {
			assert _arsParams != null;
			return _arsParams.getArray(K.NACHFORDERN).getItem(at).asTextValue().getValueText();
		}
		else {
			return null;
		}
	}

	/**
	 * Liefert den AspektID unter dem quittiert werden soll, -1 falls nicht quittiert werden soll.
	 *
	 * @return Quittierungs-AspektID.
	 */
	public long arSParamGetQuittieren() {
		if(_arsParams == null) {
			return INVALID;
		}
		else {
			try {
				// Dies funktionier für normale Archiv-Konfig und für Simulationen:
				Data.Array qt = _arsParams.getArray(K.QUITTIEREN); // Array mit max. Länge 1
				if(qt == null || qt.getLength() <= 0) {
					return INVALID;
				}
				else {
					SystemObject so = qt.getItem(0).asReferenceValue().getSystemObject();
					// DAV laesst fehlerhafterweise hier Nullwerte zu:
					return so == null ? INVALID : so.getId();
				}
			}
			catch(IllegalArgumentException e) {
				return INVALID;    // Das Data-Objekt stammt von einer Simulation (die niemals sichert) und kein solches Array hat.
			}
		}
	}

	/** @return ob Datensätze dieser Datenidentifikation zu quittieren sind. */
	public boolean arSParamIsQuittieren() {
		return arSParamGetQuittieren() != INVALID;
	}

	/** @return ob quittiert werden soll und der Quittierungsaspekt erfolgreich angemeldet werden konnte. */
	public boolean arSParamIsQuittierenValid() {
		return arSParamIsQuittieren() && !getFlag(FLAG_INVALID_ACK_ID);
	}

	/**
	 * Setzt den "Quittieren ungültig"-Flag
	 *
	 * @param invalid Ungültig?
	 */
	public void arSParamMarkQuittierenInvalid(boolean invalid) {
		setFlag(FLAG_INVALID_ACK_ID, invalid);
	}


	/**
	 * Liefert Kennzeichen, ob der DataIdentNode über die Archivparametrierung/Simulation angelegt wurde oder wegen einer Archivanfrage.
	 *
	 * @return wahr: über Archivparametrierung, falsch: wegen Archivanfrage
	 */
	public boolean isArSParameterized() {
		return _arsParams != null;
	}

	/**
	 * Setzt, ob aktuell ein gültiger Online-Datensatz vorliegt, (d.h. eine Quelle existiert).
	 * <p>
	 * Diese Information wird benötigt, um beim Beenden des Archivsystems zu vermerken, ob der zuletzt archivierte Datensatz
	 * zum Zeitpunkt des Beenden noch gültig war.
	 *
	 * @param value Ob aktuell eine Quelle existiert
	 */
	public void setValidData(boolean value) {
		setFlag(FLAG_HAS_VALID_DATA, value);
	}

	/**
	 * Gibt zurück ob es aktuell eine Quelle für die Online-Daten dieser Datenidentifikation gibt.
	 * @return true wenn es eine Quelle gibt, sonst false.
	 */
	public boolean hasValidData() {
		return getFlag(FLAG_HAS_VALID_DATA);
	}

	/**
	 * Gibt die Zeit zurück, bei der zuletzt bekannt war, dass eine gültige Quelle für die Online-Daten existierte, oder -1 wenn aktuell ein gültiger Datensatz vorliegt.
	 * <p>
	 * Diese Zeit wird über einen Neustart des Archivsystems gemerkt, damit der Zeitstempel der Datenlücke entsprechend gebildet werden kann (
	 *
	 * @return Zeitstempel in Millisekunden seit Epoch, bei der die Quelle abgemeldet wurde oder -1 falls eine Quelle gerade existiert
	 * @see #hasValidData()
	 */
	public long getUnsubscriptionTime() {
		return _unsubscriptionTime;
	}

	/**
	 * Setzt die Zeit, ab der keine Online-Daten mehr für diese Datenidentifikation existieren
	 * @param value Zeitstempel in Millisekunden seit Epoch
	 */
	public void setUnsubscriptionTime(long value) {
		_unsubscriptionTime = value;
	}

	@Override
	public String toString() {
		String erg = "DataIdentNode "+ _dataIdentification + " mit ArchivParam: a=" + (arSParamIsArchivieren() ? "T" : "F");
		StringBuilder list = new StringBuilder();
		if(arSParamIsNachfordern()) {
			for(int i = 0; i < arSParamGetAnzNachfordern(); i++) {
				list.append(arSParamGetNachfordern(i)).append(",");
			}
		}
		erg += " n=[" + Util.removeLastChar(list.toString()) + "]";
		erg += " q=" + arSParamGetQuittieren();
		return erg;
	}

	/**
	 * Gibt die Datenidentifikation zurück, die dieses Objekt verwaltet
	 *
	 * @return die Datenidentifikation
	 */
	public IdDataIdentification getDataIdentification() {
		return _dataIdentification;
	}
}
