/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.datatree;

import de.bsvrz.ars.ars.mgmt.datatree.synchronization.SyncKey;
import de.bsvrz.ars.ars.persistence.IdContainerFileDir;
import de.bsvrz.ars.ars.persistence.LockedContainerDirectory;
import de.bsvrz.ars.ars.persistence.RebuildResult;
import de.bsvrz.ars.ars.persistence.directories.PersistenceDirectory;
import de.bsvrz.ars.ars.persistence.index.*;
import de.bsvrz.ars.ars.persistence.index.backend.management.BaseIndex;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;
import de.bsvrz.sys.funclib.kappich.annotations.Nullable;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

/**
 * Diese Klasse verwaltet die vom aktuellen Thread geöffneten Indexe. Wenn {@linkplain SyncKey die Synchronisierung} auf eine Datenidentifikation beendet wird, werden automatisch alle Indexe hier freigegeben und können dann von anderen Threads in Anspruch genommen werden.
 *
 * @author Kappich Systemberatung
 */
public final class IndexTree {
	
	private static final Debug _debug = Debug.getLogger();

	/**
	 * Limit ab dem der User über zu viele Containerdateien in einer DID gewarnt wird.
	 * 32000 ist schon recht hoch, es gibt aber viele "volle" Archivsysteme, und wir wollen keinen Debug-Spam erzeugen.
	 */
	public static final long ENTRIES_WARN_LIMIT = Short.MAX_VALUE; // Halbes Dateilimit pro Ordner für FAT32, willkürlich gewählt

	/**
	 * Cache mit offenen Indexen je Thread. ThreadLocal wird hauptsächlich deswegen verwendet, damit die HashMap nicht synchronisiert werden muss (langsam/deadlock-Gefahr).
	 * Das ThreadLocal ist hier <b>nicht</b> static, damit Testfälle mehrere Archivsysteme gleichzeitig starten können und sich die Caches nicht in die Quere kommen.
	 * <p>
	 *     Der Cache enthält das NO_INDEX-Objekt für den Fall, dass kein Index existiert.
	 * </p>
	 */
	private final ThreadLocal<Map<IndexId, Object>> _cache = ThreadLocal.withInitial(HashMap::new);

	/**
	 * Markierungsobjekt für "Kein Index"
	 */
	private static final Object NO_INDEX = new Object();

	/**
	 * Persistenz
	 */
	private final PersistenceDirectory _persistenceDirectory;

	/**
	 * Maximale Anzahl Cache-Einträge für Datenzeitindex.
	 */
	private final int _dataTimeSize;

	/**
	 * Maximale Anzahl Cache-Einträge für Datenindex-Index.
	 */
	private final int _dataIndexSize;

	/**
	 * Maximale Anzahl Cache-Einträge für Archivzeit-Index.
	 */
	private final int _archiveTimeSize;

	/**
	 * Maximale Anzahl Cache-Einträge für Verwaltungsdatenindex.
	 */
	private final int _managementIndexSize;

	/**
	 * Konstruktor.
	 *
	 * @param maximumIndexSizeBytes Maximale Anzahl Bytes, die ein Index im Speicher Cachen darf
	 * @param persistenceDirectory  Verwaltung der Persistenz
	 */
	public IndexTree(final int maximumIndexSizeBytes, final PersistenceDirectory persistenceDirectory) {
		_dataTimeSize = Math.min(1, maximumIndexSizeBytes / DataTimeIndexImpl.entrySize());
		_dataIndexSize = Math.min(1, maximumIndexSizeBytes / DataIndexIndexImpl.entrySize());
		_archiveTimeSize = Math.min(1, maximumIndexSizeBytes / ArchiveTimeIndexImpl.entrySize());
		_managementIndexSize = Math.min(1, maximumIndexSizeBytes / ContainerManagementIndex.entrySize());
		_persistenceDirectory = persistenceDirectory;
	}

	/**
	 * Schließt alle Indexe, die vom aktuellen Thread geöffnet wurden.
	 */
	public void closeIndexes() {
		Map<IndexId, Object> indexMap = _cache.get();
		for (Object value : indexMap.values()) {
			if (value instanceof BaseIndex<?> baseIndex) {
				try {
					baseIndex.close();
				} catch (IndexException e) {
					_debug.warning("Fehler beim Schließen eines Index", e);
				}
			}
		}
		indexMap.clear();
	}

	/**
	 * Lädt einen Index von der Festplatte. Dies darf nur ausgeführt werden, wenn die Indexdatei noch nicht geöffnet ist. (Eine Indexdatei zweimal zu öffnen kann zu Datenkorruption führen, da die Inhalte gecacht werden und dann nicht mehr zu den Dateien passen)
	 *
	 * @param key Information, welcher Index geladen werden soll
	 * @return geladener Index
	 * @throws IndexException Index konnte nicht geladen werden. In manchen Fällen wird die Subklasse {@link CorruptIndexException} geworfen.
	 */
	private Optional<? extends BaseIndex<? extends Enum<?>>> loadIndexFromDisk(@NotNull IndexId key, LockedContainerDirectory containerDirectory)
		throws IndexException {
		Optional<? extends BaseIndex<? extends Enum<?>>> index = loadIndexFromDiskHelper(key, containerDirectory);
		if (index.isPresent() && index.get() instanceof ContainerManagementIndex cmi && cmi.numEntries() > ENTRIES_WARN_LIMIT) {
			_persistenceDirectory.warnAboutHugeContainerDirectory(_persistenceDirectory.getLayoutInstance().getContainerDirectory(cmi), cmi.numEntries());
		}
		return index;
	}

	private Optional<? extends BaseIndex<? extends Enum<?>>> loadIndexFromDiskHelper(@NotNull final IndexId key, LockedContainerDirectory containerDirectory) throws IndexException {
		Path indexFile = key.toFile(_persistenceDirectory);
		if(!Files.exists(indexFile)) {
			if (!Files.exists(indexFile.getParent())) {
				return Optional.empty();
			}
			// Index fehlt, wird neu aufgebaut
			return rebuildIndex(key, indexFile, containerDirectory);
		}
		try {
			return Optional.of(openIndex(key, indexFile));
		}
		catch(CorruptIndexException e) {
			// Index ist fehlerhaft, wird neu aufgebaut
			
			// Zuerst alten Index löschen
			try {
				Files.deleteIfExists(indexFile);
			} catch (IOException ioException) {
				IndexException subExp =
					new IndexException("Defekter Index konnte nicht gelöscht werden", indexFile, ioException);
				subExp.addSuppressed(e);
				throw subExp;
			}
			
			// Neuen Index erstellen und zurückgeben
			return rebuildIndex(key, indexFile, containerDirectory);
		}
	}

	/**
	 * Baut einen Index neu auf.
	 * @param indexId   Ort und Typ des Indexes
	 * @param indexFile Anzulegende Indexdatei
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	Datenidentifikation
	 * @return Neuer Index
	 * @throws IndexException Index konnte nicht angelegt werden
	 */
	private Optional<? extends BaseIndex<? extends Enum<?>>> rebuildIndex(final IndexId indexId, final Path indexFile, LockedContainerDirectory containerDirectory) throws IndexException {
		if(indexId.getIndexClass() == IndexImpl.ManagementData) {
			return rebuildContainerHeaderIndex(indexId, indexFile, null);
		}
		return rebuildStandardIndex(indexId, indexFile, containerDirectory);
	}

	/**
	 * Baut einen Standard-Index (Datenzeit, Datenindex, Archivzeit) neu auf.
	 * @param indexId   Ort und Typ des Indexes
	 * @param indexFile Anzulegende Indexdatei
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten
	Synchronisierung auf Datenidentifikation
	 * @return Neuer Index
	 * @throws IndexException Index konnte nicht angelegt werden
	 */
	private Optional<BaseIndex<IndexValues>> rebuildStandardIndex(final IndexId indexId, final Path indexFile, LockedContainerDirectory containerDirectory) throws IndexException {
		// Ein Standardindex wird aus dem Verwaltungsdatenindex erzeugt.
		
		IndexId managementId = new IndexId(indexId.getContainerFileDir(), IndexImpl.ManagementData);
		Optional<ContainerManagementIndex> managementIndex = cacheIndex(managementId, () -> loadIndexFromDisk(managementId, containerDirectory));
		if (managementIndex.isEmpty()) {
			return Optional.empty();
		}
		return rebuildStandardIndex(indexId, indexFile, managementIndex.get(), containerDirectory);
	}

	/**
	 * Baut einen Standard-Index (Datenzeit, Datenindex, Archivzeit) aus einem Verwaltungsdatenindex neu auf.
	 *
	 * @param indexId         Ort und Typ des Indexes
	 * @param indexFile       Anzulegende Indexdatei
	 * @param managementIndex Geladener Verwaltungsdatenindex
	 * @param containerDirectory Referenz auf die gelockte Datenidentifikation und Datenart für den Zugriff auf Containerdaten

	 * @return Neuer Index
	 * @throws IndexException Index konnte nicht angelegt werden
	 */
	private Optional<BaseIndex<IndexValues>> rebuildStandardIndex(IndexId indexId,
	                                                    Path indexFile,
	                                                    ContainerManagementIndex managementIndex,
	                                                    LockedContainerDirectory containerDirectory) throws IndexException {
		deleteIndex(indexId);
		Optional<BaseIndex<IndexValues>> index = cacheIndex(indexId, () -> Optional.of(openIndex(indexId, indexFile)));
		if (index.isPresent()) {
			_persistenceDirectory.rebuildStandardIndex(managementIndex, index.get(), containerDirectory);
		}
		return index;
	}

	/**
	 * Baut einen Verwaltungsdatenindex neu auf.
	 *
	 * @param indexId       Ort und Typ des Indexes
	 * @param indexFile     Anzulegende Indexdatei
	 * @param rebuildResult Hier kann die Methode eine Statistik über Vorgang ablegen. Kann null sein.
	 * @return Neuer Index
	 * @throws IndexException Index konnte nicht angelegt werden
	 */
	private Optional<ContainerManagementIndex> rebuildContainerHeaderIndex(final IndexId indexId, final Path indexFile,
	                                                             @Nullable final RebuildResult rebuildResult) throws IndexException {
		deleteIndex(indexId);
		Optional<ContainerManagementIndex> index = cacheIndex(indexId, () -> Optional.of(openIndex(indexId, indexFile)));

		if (index.isPresent()) {
			// Implementierung in Verwaltungsschicht, da Informationen aus Containern geholt werden.
			_persistenceDirectory.rebuildContainerHeaderIndex(indexId.getContainerFileDir(), index.get(), rebuildResult);
		}

		return index;
	}

	/**
	 * Entfernt einen Index umgehend aus dem Cache und aktualisiert die Indexdatei auf der Festplatte auf de aktuellen gecachten Stand.
	 * @param indexId Index-Typ und -Ort
	 * @throws IndexException Bei einem Schreibfehler
	 */
	private void invalidateNow(final IndexId indexId) throws IndexException {
		Object removed = _cache.get().remove(indexId);
		if (removed instanceof BaseIndex<?> index) {
			index.close();
		}
	}

	/**
	 * Öffnet einen Index von der Festplatte.
	 * @param indexId   Typ und Ort des Index
	 * @param indexFile Indexdatei
	 * @return Geladener Index
	 * @throws CorruptIndexException Falls beim Laden ein Fehler auftritt. Dann ist die Indexdatei vermutlich defekt und muss neu aufgebaut werden.
	 */
	private BaseIndex<? extends Enum<?>> openIndex(final IndexId indexId, final Path indexFile) throws CorruptIndexException {
		switch (indexId.getIndexClass()) {
			case DataTime -> {
				return new DataTimeIndexImpl(_dataTimeSize, indexFile);
			}
			case DataIndex -> {
				// DataIndexIndex alleinstehend nur bei nachgeforderten Daten, bei Online-Daten wird der Archivzeitindex mitbenutzt,
				// der dort auch im Datenindex monoton ist.
				assert indexId.getContainerFileDir().archiveDataKind().isRequested();
				return new DataIndexIndexImpl(_dataIndexSize, indexFile);
			}
			case ArchiveTime -> {
				if (indexId.getContainerFileDir().archiveDataKind().isRequested()) {
					return new ArchiveTimeIndexImpl(_archiveTimeSize, indexFile);
				} else {
					return new DataIndexAndArchiveTimeIndex(_archiveTimeSize, indexFile);
				}
			}
			case ManagementData -> {
				return new ContainerManagementIndex(_managementIndexSize, indexFile);
			}
		}
		throw new IllegalArgumentException(String.valueOf(indexId.getIndexClass()));
	}

	public Optional<? extends ContainerManagementIndex> getContainerManagementIndex(LockedContainerDirectory containerDirectory) throws IndexException {
		assert containerDirectory.lock().isValid();
		IndexId key = new IndexId(containerDirectory, IndexImpl.ManagementData);
		return cacheIndex(key, () -> loadIndexFromDisk(key, containerDirectory));
	}

	public Optional<? extends DataIndexIndex> getDataIndexIndex(LockedContainerDirectory containerDirectory) throws IndexException {
		assert containerDirectory.lock().isValid();
		if (!containerDirectory.archiveDataKind().isRequested()) {
			//noinspection unchecked
			return (Optional<DataIndexAndArchiveTimeIndex>) getArchiveTimeIndex(containerDirectory);
		}
		IndexId key = new IndexId(containerDirectory, IndexImpl.DataIndex);
		return cacheIndex(key, () -> loadIndexFromDisk(key, containerDirectory));
	}

	public Optional<? extends DataTimeIndex> getDataTimeIndex(LockedContainerDirectory containerDirectory) throws IndexException {
		assert containerDirectory.lock().isValid();
		IndexId key = new IndexId(containerDirectory, IndexImpl.DataTime);
		return cacheIndex(key, () -> loadIndexFromDisk(key, containerDirectory));
	}

	public Optional<? extends ArchiveTimeIndex> getArchiveTimeIndex(LockedContainerDirectory containerDirectory) throws IndexException {
		assert containerDirectory.lock().isValid();
		IndexId key = new IndexId(containerDirectory, IndexImpl.ArchiveTime);
		return cacheIndex(key, () -> loadIndexFromDisk(key, containerDirectory));
	}

	@SuppressWarnings("unchecked")
	private <T extends BaseIndex<?>> Optional<T> cacheIndex(IndexId key, IndexCreator indexCreator) throws IndexException {
		Map<IndexId, Object> cacheMap = _cache.get();
		Object index = cacheMap.get(key);
		if (index != null) {
			if (index == NO_INDEX) {
				return Optional.empty();
			}
			return Optional.of((T) index);
		}
		
		// Nicht computeIfAbsent verwenden, da loadIndex ggf. weitere andere Indexe lädt und es sonst zu einer
		// java.util.ConcurrentModificationException kommt 
		if (cacheMap.containsKey(key)) {
			throw new IllegalArgumentException("key mehrfach initialisiert");
		}
		Optional<? extends BaseIndex<?>> indexToCache = indexCreator.create();
		if (indexToCache.isPresent()) {
			cacheMap.put(key, indexToCache.get());
		} else {
			cacheMap.put(key, NO_INDEX);
		}
		return (Optional<T>) indexToCache;
	}

	interface IndexCreator {
		public Optional<? extends BaseIndex<?>> create() throws IndexException;
	}

	/**
	 * Erstellt alle Indexe aus den Containerdaten neu.
	 * @param result          Ergebnis (Statistik) oder null
	 * @throws IndexException Falls die Indexe nicht neu erstellt werden konnten
	 */
	public void recreateIndex(LockedContainerDirectory containerDirectory, @Nullable RebuildResult result) throws IndexException {
		Optional<ContainerManagementIndex> managementIndex = rebuildContainerHeaderIndex(containerDirectory, result);
		if (managementIndex.isEmpty()) {
			return;
		}
		if (containerDirectory.archiveDataKind().isRequested()) {
			rebuildStandardIndex(containerDirectory, IndexImpl.DataIndex, managementIndex.get());
		}
		rebuildStandardIndex(containerDirectory, IndexImpl.ArchiveTime, managementIndex.get());
		rebuildStandardIndex(containerDirectory, IndexImpl.DataTime, managementIndex.get());
	}

	/**
	 * Erstellt den Verwaltungsdatenindex neu.
	 * @param rebuildResult   Ergebnis (Statistik)
	 * @return Neuer Verwaltugnsdatenindex
	 * @throws IndexException Falls der Index nicht erstellt werden konnte
	 */
	private Optional<ContainerManagementIndex> rebuildContainerHeaderIndex(final LockedContainerDirectory containerDirectory, @Nullable RebuildResult rebuildResult) throws IndexException {
		IndexId indexId = new IndexId(containerDirectory, IndexImpl.ManagementData);
		return rebuildContainerHeaderIndex(indexId, indexId.toFile(_persistenceDirectory), rebuildResult);
	}

	/**
	 * Erstellt einen Standard-Index aus dem Verwaltungsdatenindex neu.
	 *
	 * @param indexClass      Art des zu erstellenden Index
	 * @param managementIndex Verwaltungsdatenindex
	 * @return Neuer Index
	 * @throws IndexException Falls der Index nicht erstellt werden konnte
	 */
	@SuppressWarnings("UnusedReturnValue")
	private Optional<BaseIndex<IndexValues>> rebuildStandardIndex(final LockedContainerDirectory containerDirectory, final IndexImpl indexClass,
	                                                              ContainerManagementIndex managementIndex) throws IndexException {
		IndexId indexId = new IndexId(containerDirectory, indexClass);
		return rebuildStandardIndex(indexId, indexId.toFile(_persistenceDirectory), managementIndex, containerDirectory);
	}

	/**
	 * Löscht alle Indexe und Indexdateien in einem Containerverzeichnis.
	 * @param result          Statistik (die Zahl der gelöschten Indexdateien wird in result.indexesDeleted gespeichert)
	 * @throws IndexException Fehler beim Löschen
	 */
	public void deleteIndex(LockedContainerDirectory containerDirectory, @Nullable RebuildResult result) throws IndexException {
		int sum = 0;
		sum += deleteIndex(new IndexId(containerDirectory, IndexImpl.ManagementData));
		if (containerDirectory.archiveDataKind().isRequested()) {
			sum += deleteIndex(new IndexId(containerDirectory, IndexImpl.DataIndex));
		}
		sum += deleteIndex(new IndexId(containerDirectory, IndexImpl.ArchiveTime));
		sum += deleteIndex(new IndexId(containerDirectory, IndexImpl.DataTime));
		if (result != null) {
			result.indexesDeleted += sum;
		}
	}

	/**
	 * Löscht einen Index.
	 * @param indexId ID des Indexes	
	 * @return Anzahl gelöschter Dateien (1 = Index gelöscht, 0 = Index war nicht vorhanden)
	 * @throws IndexException Wenn der Index nicht gelöscht werden konnte (in der Regel Dateisystemproblem)
	 */
	private int deleteIndex(final IndexId indexId) throws IndexException {
		invalidateNow(indexId);
		Path file = indexId.toFile(_persistenceDirectory);
		try {
			return Files.deleteIfExists(file) ? 1 : 0;
		} catch (IOException e) {
			throw new IndexException("Kann vorhandenen Index nicht löschen", file, e);
		}
	}

	/**
	 * Schließt alle vom aktuellen Thread geöffneten Indexe und schreibt die Puffer auf die Festplatte.
	 * @throws IndexException Fehler beim Schreiben der Indexdateien
	 */
	public void closeIndexes(final LockedContainerDirectory containerDirectory) throws IndexException {
		for(IndexImpl value : IndexImpl.values()) {
			invalidateNow(new IndexId(containerDirectory, value));
		}
	}

	/**
	 * Schreibt die Puffer für alle vom aktuellen Thread geöffneten Indexe auf die Festplatte ohne den Index zu schließen.
	 * Fehler werden dabei ignoriert udn führen zu einer Warnung.
	 * Diese Methode wird vom datenkonsistenten Backup benutzt. Hier ist bei Fehlern keine bessere Fehlerbehandlung sinnvoll.
	 */
	public void flushIndexes(final LockedContainerDirectory containerDirectory) {
		for (Map.Entry<IndexId, Object> entry : _cache.get().entrySet()) {
			IdContainerFileDir containerFileDir = entry.getKey().getContainerFileDir();
			if (containerFileDir.dataIdentification().equals(containerDirectory.dataIdentification()) && containerFileDir.archiveDataKind() == containerDirectory.archiveDataKind()) {
				Object baseIndex = entry.getValue();
				if (baseIndex instanceof BaseIndex<?> index) {
					try {
						index.flush();
					} catch (IndexException e) {
						_debug.warning("Fehler beim Speichern eines Index", e);
					}
				}
			}
		}
	}

	/**
	 * Debug-Methode. Wirft eine Exception, wenn der aktuelle Thread aktuell Indexe geöffnet hat.
	 */
	public void ensureNoCached() {
		if(!_cache.get().isEmpty()) {
			throw new IllegalStateException("Cache nicht leer: " + _cache.get());
		}
	}

}
