/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.datatree.synchronization;

import com.google.common.collect.SetMultimap;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;

import java.time.Duration;

/**
 * Delegate auf einen SynchronizationManager, der bei langen Wartedauern Debug-Meldungen erzeugt.
 *
 * @author Kappich Systemberatung
 */
public class DebuggingSynchronizationManager<T> implements SynchronizationManager<T> {
	
	private static final Debug _debug = Debug.getLogger();
	
	private final SynchronizationManager<T> _delegate;

	/** 
	 * Erstellt einen neuen DebuggingSynchronizationManager
	 * @param delegate Delegate-Objekt (!= null)
	 */
	public DebuggingSynchronizationManager(SynchronizationManager<T> delegate) {
		_delegate = delegate;
	}

	@NotNull
	@Override
	public SyncKey<T> acquireWriteKey(final T element) {
		while(true) {
			try {
				return acquireWriteKey(element, Duration.ofSeconds(60));
			}
			catch(SynchronizationFailedException e) {
				_debug.warning("Thread " + Thread.currentThread().getName() + " wartet beim schreibenden Zugriff auf " + element + " da ein anderer Thread den Zugriff blockiert.", e);
			}
		}
	}

	@NotNull
	@Override
	public SyncKey<T> acquireWriteKey(final T element, final Duration timeout) throws SynchronizationFailedException {
		return _delegate.acquireWriteKey(element, timeout);
	}

	@Override
	public SetMultimap<T, SyncKey<T>> getLocks() {
		return _delegate.getLocks();
	}

	@Override
	public String toString() {
		return "DebuggingSynchronizationManager{" +
				"_delegate=" + _delegate +
				'}';
	}
}
