/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.datatree.synchronization;

import com.google.common.collect.SetMultimap;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;

import java.time.Duration;

/**
 * Objekt, das Synchronisierungen auf Elemente vom generischen Typ T verwaltet.
 * <p>
 * Ein Nutzer dieser Klasse kann entweder Lese- oder Schreibzugriff auf ein Element vom Typ K anfordern und
 * erhält dann ein Schlüssel-Objekt. Nach Abschluss der Operation muss das Schlüsselobjekt freigegeben werden,
 * damit weitere Nutzer auf das Element zugreifen können.
 * </p>
 * <p>
 * Diese Klasse verwaltet alle vergebenen Schlüssel und erlaubt für Debug-Zwecke z. B. die vergebenen Schlüssel auszugeben.
 * </p>
 * <p>
 * Der konkrete Typ von K sollte equals und hashCode performant und sinnvoll implementieren.
 * </p>
 * <p>
 * Zurückgegebene {@link SyncKey}-Objekte sind an einen Thread gebunden und dürfen nicht von einem Thread an einen aderen weitergegeben werden. Sie sollten nur innerhalb von try-with-resources-Blöcken genutzt werden. Beispiel:
 * </p>
 * <pre>{@code
 * try(var key = syncManager.acquireWriteKey(someID)) {
 *	 // Mache etwas, das key benötigt
 * }
 * }</pre>
 *
 * @author Kappich Systemberatung
 * @param <T>
 */
public interface SynchronizationManager<T> {

	/**
	 * Fordert exklusiven Zugriff (Schreibzugriff) auf das angegebene Element an. Ein Schreibzugriff darf nur angefordert werden, wenn dieser Thread noch keinen Lesezugriff besitzt. Ansonsten würde dies Deadlocks verursachen. Sollte dieser Thread bereits einen Schreibzugriff besitzen ist das jedoch kein Problem.
	 * @param element Element
	 * @return Schlüssel zum Zugriff auf das Element. Solange der Schlüssel gültig ist, garantiert diese Klasse, kann kein anderer Thread einen anderen Schlüssel auf dieses Element hält.
	 * @throws SynchronizationFailedException Falls der Schlüssel nicht geholt werden konnte (z. B. interrupted beim warten auf Schlüssel)
	 */
	@NotNull
	SyncKey<T> acquireWriteKey(T element) throws SynchronizationFailedException;

	/**
	 * Fordert exklusiven Zugriff (Schreibzugriff) auf das angegebene Element an. Ein Schreibzugriff darf nur angefordert werden, wenn dieser Thread noch keinen Lesezugriff besitzt. Ansonsten würde dies Deadlocks verursachen. Sollte dieser Thread bereits einen Schreibzugriff besitzen ist das jedoch kein Problem.
	 * @param element Element
	 * @param timeout maximale Wartezeit zur synchronisierung
	 * @return Schlüssel zum Zugriff auf das Element. Solange der Schlüssel gültig ist, garantiert diese Klasse, kann kein anderer Thread einen anderen Schlüssel auf dieses Element hält.
	 * @throws SynchronizationFailedException Falls der Schlüssel nicht geholt werden konnte (z. B. timeout oder interrupted beim warten auf Schlüssel)
	 */
	@NotNull
	SyncKey<T> acquireWriteKey(T element, Duration timeout) throws SynchronizationFailedException;

	/** 
	 * Gibt alle aktuell angeforderten Schlüssel zurück. Diese Funktion existiert nur zum Debugging bzw. zur Fehleranalyse um z. B. bei Deadlocks den Zustand ausgeben zu können.
	 * @return alle aktuell angeforderten Schlüssel
	 */
	SetMultimap<T, SyncKey<T>> getLocks();
}
