/*
 * Copyright 2019-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.datatree.synchronization;

import com.google.common.base.FinalizablePhantomReference;
import com.google.common.base.FinalizableReferenceQueue;
import com.google.common.collect.HashMultimap;
import com.google.common.collect.ImmutableSetMultimap;
import com.google.common.collect.SetMultimap;
import com.google.common.collect.Sets;
import de.bsvrz.sys.funclib.debug.Debug;
import de.bsvrz.sys.funclib.kappich.annotations.NotNull;

import java.io.StringWriter;
import java.lang.ref.Reference;
import java.time.Duration;
import java.util.Iterator;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.BooleanSupplier;
import java.util.function.Consumer;

/**
 * Objekt, das Synchronisierungen auf Elemente vom generischen Typ T verwaltet.
 * <p>
 * Ein Nutzer dieser Klasse kann entweder Lese- oder Schreibzugriff auf ein Element vom Typ T anfordern und
 * erhält dann ein Schlüssel-Objekt. Nach Abschluss der Operation muss das Schlüsselobjekt freigegeben werden,
 * damit weitere Nutzer auf das Element zugreifen können.
 * </p>
 * <p>
 * Diese Klasse verwaltet alle vergebenen Schlüssel und erlaubt für Debug-Zwecke z. B. die vergebenen Schlüssel auszugeben.
 * </p>
 * <p>
 * Der konkrete Typ von T sollte equals und hashCode performant und sinnvoll implementieren.
 * </p>
 *
 * @author Kappich Systemberatung
 * @param <T> Typ, der gelockt wird (in der Regel {@link de.bsvrz.ars.ars.persistence.IdDataIdentification}). 
 */
public class SynchronizationManagerImpl<T> implements SynchronizationManager<T> {

	/**
	 * Debug.
	 */
	private static final Debug _debug = Debug.getLogger();

	// Die beiden folgenden Fields und die #attachPhantomReference-Methode sind eigentlich überflüssig. Sie warnen aber bei Programmierfehlern.
	/**
	 * Sammelt {@link SyncKey}-Objekte, die vom garbage-Collector aufgeräumt werden, um bei nicht korrekt geschlossenen Objekten zu benachrichtigen.
	 */
	private static final FinalizableReferenceQueue REFERENCE_QUEUE = new FinalizableReferenceQueue();

	/**
	 * Hält harte Referenzen. Siehe REFERENCE_QUEUE.
	 */
	@SuppressWarnings("MismatchedQueryAndUpdateOfCollection")
	private static final Set<Reference<?>> REFERENCES = Sets.newConcurrentHashSet();
	
	/**
	 * Hier werden alle Schlüssel indexiert gespeichert.
	 */
	private final HashMultimap<T, SyncKey<T>> _keyStorage = HashMultimap.create();

	/**
	 * Lock zum synchronisieren auf _keyStorage.
	 */
	private final ReentrantLock _lock = new ReentrantLock();

	/**
	 * Condition, die für Condition#await genutzt wird.
	 */
	private final Condition _condition = _lock.newCondition();

	/**
	 * Aktion, die ausgeführt wird, wann immer (erstmals) auf ein Element synchronisiert wird.
	 */
	private final Consumer<T> _onOpenAction;

	/**
	 * Aktion, die ausgeführt wird, wenn ein Thread ein Lock komplett aufgibt. (Aufräumaktion o.ä.).
	 * Im Archivsystem werden hier die offenen Indexe geschlossen.
	 */
	private final Consumer<T> _onCloseAction;

	/**
	 * Konstruktor.
	 *
	 * @param onOpenAction Aktion, die ausgeführt wird, wann immer (erstmals) auf ein Element synchronisiert wird.
	 * @param onCloseAction Aktion, die jedes mal ausgeführt wird, wenn ein Thread die Synchronisierung auf ein Element komplett aufgibt, also <b>kein</b> Lock mehr vorhanden ist. Das wird hier benutzt, um alle genutzten Indexe freizugeben.
	 */
	public SynchronizationManagerImpl(Consumer<T> onOpenAction, Consumer<T> onCloseAction) {
		_onCloseAction = onCloseAction;
		_onOpenAction = onOpenAction;
	}

	@NotNull
	@Override
	public SyncKey<T> acquireWriteKey(T element) throws SynchronizationFailedException {
		_lock.lock();
		try {
			while(isWriteBlocked(element)) {
				_condition.await();
			}
			if(_keyStorage.get(element).isEmpty()) {
				_onOpenAction.accept(element);
			}
			SyncKey<T> writeKey = createWriteKey(element);
			_keyStorage.put(element, writeKey);
			attachPhantomReference(writeKey);
			return writeKey;
		}
		catch(InterruptedException e) {
			throw new SynchronizationFailedException(null, element, getLocks(element), e);
		}
		finally {
			_lock.unlock();
		}
	}

	@NotNull
	@Override
	public SyncKey<T> acquireWriteKey(T element, Duration timeout) throws SynchronizationFailedException {
		long nanosRemaining = timeout.toNanos();
		_lock.lock();
		try {
			while(isWriteBlocked(element)) {
				nanosRemaining = _condition.awaitNanos(nanosRemaining);
				if(nanosRemaining <= 0) {
					throw new SynchronizationFailedException(timeout, element, getLocks(element), null);
				}
			}
			if(_keyStorage.get(element).isEmpty()) {
				_onOpenAction.accept(element);
			}
			SyncKey<T> writeKey = createWriteKey(element);
			_keyStorage.put(element, writeKey);
			attachPhantomReference(writeKey);
			return writeKey;
		}
		catch(InterruptedException e) {
			throw new SynchronizationFailedException(timeout, element, getLocks(element), e);
		}
		finally {
			_lock.unlock();
		}
	}

	/**
	 * Debug-Funktionalität, die den Programmierer benachrichtigt, wenn ein Thread nicht korrekt geschlossen wurde.
	 * @param writeKey Schlüssel
	 */
	private void attachPhantomReference(SyncKey<T> writeKey) {
		BooleanSupplier notClosed = writeKey.notClosed();
		Thread thread = writeKey.getThread();
		T element = writeKey.getElement();
		REFERENCES.add(new FinalizablePhantomReference<>(writeKey, REFERENCE_QUEUE) {
			@Override
			public void finalizeReferent() {
				REFERENCES.remove(this);
				if (notClosed.getAsBoolean()) {
					_debug.error("Element " + element + " von Thread " + thread + " wurde nicht korrekt geschlossen.");
				}
			}
		});
	}

	/**
	 * Erstellt einen neuen {@link SyncKey}.
	 * @param element Element (Datenidentifikation)
	 * @return neuer SyncKey
	 */
	private SyncKey<T> createWriteKey(final T element) {
		return new SyncKeyImpl(element);
	}


	/**
	 * Ein Element ist für das Schreiben blockiert, wenn es bereits eine Synchronisierung auf das Element gibt, die nicht dem aktuellen thread zugeordnet ist.
	 * Für den Sonderfall, dass bisher nur Lesezugriffe dem aktuellen thread zugeordnet sind, wird eine {@link SynchronizationFailedException} geworfen, da ein lesezugriff nicht in einen Schreibzugriff upgegradet werden darf, da es dadurch deadlocks geben könnte.
	 * @param element Element
	 * @return true falls Lesen blockiert, false falls gelesen werden darf
	 */
	private boolean isWriteBlocked(final T element) {
		Set<SyncKey<T>> values = getLocks(element);
		if(values.isEmpty()) return false;
		for(Iterator<SyncKey<T>> iterator = values.iterator(); iterator.hasNext(); ) {

			//noinspection resource
			final SyncKey<T> it = iterator.next();
			
			if(!it.getThread().isAlive()) {
				iterator.remove();
				_condition.signalAll();
				_debug.error("SyncKey wurde nicht korrekt geschlossen: " + it);
				continue;
			}
			
			//noinspection ObjectEquality
			if(it.getThread() == Thread.currentThread()) {
				return false;
			}
		}
		return true;
	}

	/** 
	 * Gibt alle Locks für ein Element zurück. (Interne Funktion)
	 * @param element Element (z. B. Datenidentifikation), auf das synchronisiert werden kann
	 * @return alle Locks für ein Element
	 */
	private Set<SyncKey<T>> getLocks(final T element) {
		return _keyStorage.get(element);
	}

	@Override
	public SetMultimap<T, SyncKey<T>> getLocks() {
		_lock.lock();
		try {
			return ImmutableSetMultimap.copyOf(_keyStorage);
		}
		finally {
			_lock.unlock();
		}
	}

	/**
	 * Implementierung von {@link SyncKey}.
	 */
	private class SyncKeyImpl implements SyncKey<T> {
		/**
		 * Datenidentifikation.
		 */
		private final T _element;

		/**
		 * Thread, den dieser Key zugeordnet ist.
		 */
		private final Thread _thread;

		/**
		 * Flag das angibt, ob der Schlüssel noch gültig ist.
		 */
		private final AtomicBoolean _open = new AtomicBoolean(true);

		/** 
		 * Erstellt ein neues Objekt.
		 * @param element Datenidentifikation
		 */
		public SyncKeyImpl(final T element) {
			_element = element;
			_thread = Thread.currentThread();
		}

		@Override
		public T getElement() {
			return _element;
		}

		@Override
		public Thread getThread() {
			return _thread;
		}

		@Override
		public void close() {

			if (!_open.get())
				return;
			
			// erst auf die HashMultimap synchronisieren
			_lock.lock();
			try {
				// Sicherheitshalber nochmal prüfen, der folgende Code soll nicht mehrmals ausgeführt werden.
				if (!_open.get())
					return;
				
				_open.set(false);

				// vorhandenes Set mit Locks ermitteln
				Set<SyncKey<T>> set = getLocks(_element);

				// sich selbst aus Set entfernen
				set.remove(this);

				// wenn das Set leer geworden ist, können andere Threads ggf. wieder auf das Element synchronisieren,
				// also alle wartenden Threads benachrichtigen
				if (set.isEmpty()) {
					_onCloseAction.accept(_element);
					_condition.signalAll();
				}
			} finally {
				_lock.unlock();
			}
		}

		@Override
		public String toString() {
			StringWriter out = new StringWriter();
			return _thread + " (alive: " + _thread.isAlive() + ") e: " + _element + " " + out + " valid: " + isValid();
		}

		@Override
		public boolean isValid() {
			return _open.get() && _thread.equals(Thread.currentThread());
		}

		@Override
		public BooleanSupplier notClosed() {
			return _open::get;
		}
	}

	@Override
	public String toString() {
		return "SynchronizationManagerImpl";
	}
}
