/*
 *
 * Copyright 2017-2020 by Kappich Systemberatung, Aachen
 * Copyright 2023 by DTV-Verkehrsconsult, Aachen
 *
 * This file is part of de.bsvrz.ars.ars.
 *
 * de.bsvrz.ars.ars is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * de.bsvrz.ars.ars is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with de.bsvrz.ars.ars.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact Information:
 * DTV-Verkehrsconsult GmbH
 * Pascalstraße 53
 * 52076 Aachen, Germany
 * phone: +49 2408 7047 0
 * mail: <info@dtv-verkehrsconsult.de>
 */

package de.bsvrz.ars.ars.mgmt.tasks;

import de.bsvrz.ars.ars.mgmt.TaskManager;
import de.bsvrz.ars.ars.mgmt.tasks.tasklistener.TaskStepListener;
import de.bsvrz.ars.ars.persistence.PersistenceManager;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;

/**
 * Abstrakte Basisklasse für die abstrakten Task-Basisklassen {@link de.bsvrz.ars.ars.mgmt.tasks.SingleTask} und {@link
 * de.bsvrz.ars.ars.mgmt.tasks.MultiTask}.
 * <p>
 * Ein Task ist ein Thread, eine bestimmte Aktion ausführt. Mit {@link #terminateTask()} kann diese Aufgabe abgebrochen werden.
 * Während der Ausführung kann der Task mit {@link #shouldTerminate()} prüfen, ob er terminiert wurde und dann die Aufgabe ggf. neu starten.
 *
 * @author beck et al. projects GmbH
 * @author Thomas Schaefer
 * @author Alexander Schmidt
 * @author Kappich Systemberatung
 * @version $Revision$ / $Date$ / ($Author$)
 */
public abstract class AbstractTask extends Task {

	/**
	 * Alle Objekte in dieser Liste werden benachrichtigt, wenn ein Zyklus der Hauptschleife durchlaufen wurde.
	 */
	private final List<TaskStepListener> taskStepListeners = new ArrayList<>();

	/**
	 * Der Name des Tasks bzw. Threads. kann frei gesetzt werden. Standardmäßig der Klassenname.
	 */
	private String _name;

	/**
	 * Der aktuelle Thread. Nachdem eine Exception auftritt wird ggf. ein neuer Thread gestartet.
	 */
	private volatile Thread _thread;

	/**
	 * Wurde der Task gestartet?
	 */
	private volatile boolean _isStarted;

	/**
	 * Wurde der Task schon komplett beendet?
	 */
	private volatile boolean _isTerminated;

	/**
	 * Erstellt einen neuen Task
	 *
	 * @param archiveMgr Archiv-Manager (Kann für Testfälle und ähnliches null sein, allerdings kann es dann passieren, das bestimmte Funktionen nicht
	 *				   funktionieren.)
	 */
	public AbstractTask(final TaskManager archiveMgr) {
		super(archiveMgr);
		_name = getClass().getSimpleName();
	}

	/**
	 * Erstellt einen neuen Task
	 *
	 * @param persistenceManager Archiv-Manager (Kann für Testfälle und ähnliches null sein, allerdings kann es dann passieren, das bestimmte Funktionen nicht
	 *				   funktionieren.)
	 */
	public AbstractTask(final PersistenceManager persistenceManager) {
		super(persistenceManager);
		_name = getClass().getSimpleName();
	}


	/**
	 * Gibt den Zustand den Threads zurück
	 *
	 * @return Zustand
	 */
	public final Thread.State getState() {
		Thread thread = _thread;
		if (thread != null) return thread.getState();
		return Thread.State.NEW;
	}

	/**
	 * Startet den Thread, der den Task ausführt
	 *
	 * @see Thread#start()
	 */
	@Override
	public void start() {
		_thread = new Thread(this::execute, getName());
		_thread.start();
		_isStarted = true;
	}

	/**
	 * Diese Methode muss implementiert werden und definiert die Aktion, die dieser Task ausführt.
	 * <p>
	 * Der Implementierende sollte am ende {@code super.execute()} ausführen.
	 */
	protected void execute() {
		_isTerminated = true;
	}

	/**
	 * Wird garantiert aufgerufen, nachdem der Task vollständig terminiert wurde.
	 * Dieser Code wird im selben Thread ausgeführt, wie der eigentliche Task
	 * <p>
	 * Tut standardmäßig nichts, kann überschrieben werden um Daten aufzuräumen, Abmeldungen durchzuführen usw.
	 */
	protected void cleanUp() {

	}

	/**
	 * Zu Testzwecken kann hiermit ein Listener eingefuegt werden, der am Ende eines jedem Schleifendurchlaufs aufgerufen wird. Auf diese Weise sind z.B. die
	 * Tasks einfacher zu testen. Wenn ein Task abgebrochen wird, kann nicht garantiert werden, dass für die letzte Schleife der {@link TaskStepListener} aufgerufen
	 * wird.
	 *
	 * @param tsl Callback-Interface
	 */
	public void addTaskStepListener(TaskStepListener tsl) {
		synchronized (taskStepListeners) {
			taskStepListeners.add(tsl);
		}
	}

	/**
	 * Entfernt den Listener.
	 *
	 * @param tsl Callback-Interface
	 */
	public void removeTaskStepListener(TaskStepListener tsl) {
		synchronized (taskStepListeners) {
			taskStepListeners.remove(tsl);
		}
	}

	/**
	 * Benachrichtigt darüber, dass ein Schleifendurchlauf abgearbeitet wurde.
	 */
	protected void taskStepDone() {
		synchronized (taskStepListeners) {
			for (TaskStepListener tsl : taskStepListeners) {
				tsl.taskStepDone(this);
			}
		}
	}

	/**
	 * Gibt den Namen zurück
	 *
	 * @return den Namen
	 */
	@Override
	public final String getName() {
		return _name;
	}

	/**
	 * Setzt den Namen, der auch den Namen des Threads definiert und in Debug-Ausgaben benutzt wird.
	 *
	 * @param name neuer Name (nicht null)
	 */
	public final void setName(final String name) {
		_name = Objects.requireNonNull(name);
		Thread thread = _thread;
		if (thread != null) {
			thread.setName(name);
		}
	}

	/**
	 * Wartet auf das Beenden des Tasks nach der Terminierung. Beispielcode:
	 * <pre>{@code
	 * task.terminateTask(); // Task soll bei der nächsten Möglichkeit terminieren (asynchroner Aufruf)
	 * task.join(); // Auf das terminieren synchron warten
	 * }</pre>
	 *
	 * @throws InterruptedException bei Unterbrechung
	 * @see Thread#join()
	 */
	@Override
	public void join() throws InterruptedException {
		Thread thread = _thread;
		if (thread != null) {
			thread.join();
		}
	}

	/**
	 * Gibt {@code true} zurück, wenn der Task aktuell läuft
	 *
	 * @return {@code true}, wenn der Task aktuell läuft, sonst {@code false}
	 */
	@Override
	public boolean isAlive() {
		return _isStarted && !_isTerminated;
	}

	/**
	 * Gibt {@code true} zurück, wenn der Task schon gelaufen ist und komplett terminiert wurde
	 *
	 * @return {@code true}, wenn der Task schon gelaufen ist und komplett terminiert wurde, sonst {@code false}
	 */
	@Override
	public boolean isTerminated() {
		return _isTerminated;
	}
}
